// This program belongs to AKKODIS INGENIERIE PRODUIT SAS.
// It is considered a trade secret, and is not to be divulged or used 
// by parties who have not received written authorization from the owner.
//
// This package includes the opentype.js library, licensed under the MIT license terms.
// This package includes the bowser library, licensed under the MIT license terms.
// This package includes the ajv library (https://ajv.js.org/), licensed under the MIT license terms.
// 
// The MIT License (MIT)
// Copyright (c) 2017 Frederik De Bleser
// Permission is hereby granted, free of charge, to any person obtaining a copy of
// this software and associated documentation files (the "Software"), to deal in
// the Software without restriction, including without limitation the rights to
// use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
// the Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// This package includes the luxon library, licensed under the JS Foundation license.
// 
// Copyright 2019 JS Foundation and other contributors
// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
// 
// package			: infiniteapi
// author			: 3djuump
// date				: 2025-06-03
// Last revision	: Rev: 765
//
/**
 * The {@link AnnotationGetterInterfaceSignal} lists all the signals that may be sent by the {@link AnnotationGetterInterface}.
 *
 * These signals are emitted by the {@link AnnotationGetterInterface} when an annotation fetch result is ready.
 *
 * * {@link AnnotationFetchReady} : the {@link AnnotationGetterInterface} is ready, i.e. when it has
 * finished an annotation fetch, {@link AnnotationGetterInterface.getAnnotationViewsResult} can be called.
 * * {@link AnnotationFetchCancelled} : the fetch request by the {@link AnnotationGetterInterface} has been cancelled.
 *
 * ```typescript
 * [[include:AnnotationGetterInterface_download.ts]]
 * ```
 * <br>
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link AnnotationGetterInterface}
 * @category Events
 */
declare enum AnnotationGetterInterfaceSignal {
    /**
     * Signal sent by the {@link AnnotationGetterInterface} when {@link AnnotationGetterInterface.fetchAnnotationViews} is ready, i.e.
     * {@link AnnotationGetterInterface.getAnnotationViewsResult} can be called.
     *
     * Such a signal does not tell if the call was successful or not but that the result is ready to be fetched.
     *
     * Warning, the AnnotationFetchReady signal is sent later when {@link AnnotationGetterInterface.cancelFetch} is called and the {@link AnnotationGetterInterface} is fetching.
     *
     * The attachment is a string telling the request id of the call to {@link AnnotationGetterInterface.fetchAnnotationViews} ({@link AnnotationGetterInterface.getLastRequestId}).
     */
    AnnotationFetchReady = "ready",
    /**
     * Signal sent by the {@link AnnotationGetterInterface} when it has been cancelled.
     *
     * Such an event is fired if the {@link AnnotationGetterInterface} was fetching some annotation views and {@link AnnotationGetterInterface.cancelFetch} is called.
     * The {@link AnnotationFetchReady} signal will be triggered shortly after.
     *
     * No attachment for events of this type.
     */
    AnnotationFetchCancelled = "cancelled"
}
export { AnnotationGetterInterfaceSignal };
/**
 * The AnnotationResultInterface interface is the result of an
 * annotation retrieval query from an {@link AnnotationGetterInterface}.
 *
 * Its API is very similar to the {@link AnnotationViewInfoInterface} since it is the result
 * of the {@link AnnotationGetterInterface.fetchAnnotationViews} of an {@link AnnotationViewInfoInterface}.
 *
 * The AnnotationResultInterface also contains the application matrix to set for the given annotation view, and
 * of the course the {@link AnnotationViewInterface} to be added to the {@link AnnotationRendererInterface} for display.
 *
 * ```typescript
 * [[include:AnnotationGetterInterface_download.ts]]
 * ```
 * <br>
 * Or with async calls :
 *
 * ```typescript
 * [[include:AnnotationGetterInterface_asyncDownload.ts]]
 * ```
 * <br>
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link AnnotationGetterInterface}
 * @see {@link AnnotationViewInfoInterface}
 * @see {@link AnnotationViewInterface}
 * @category Data Retrievers
 */
interface AnnotationResultInterface {
    /**
     * Gets the annotation type name of the annotations (grouped in an annotation view).
     *
     * @return The type name of the given annotation view.
     * @see {@link AnnotationViewInfoInterface.getAnnotationViewTypeName}
     */
    getAnnotationViewTypeName(): string;
    /**
     * Gets the annotation view, i.e. the annotations contained in the given annotation view.
     *
     * Such an {@link AnnotationViewInterface} may be included in an {@link AnnotationRendererInterface}.
     *
     * @return The {@link AnnotationViewInterface} of the given annotation view.
     * @see {@link AnnotationRendererInterface}
     * @see {@link AnnotationViewInterface}
     */
    getAnnotationView(): AnnotationViewInterface | undefined;
    /**
     * Gets the application matrix of the annotation view.
     *
     * DO NOT modify the resulting {@link Matrix4} in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> the matrix of the given annotation view.
     * @see {@link AnnotationRendererInterface}
     */
    getAnnotationViewMatrix(): Matrix4 | undefined;
    /**
     * Gets the annotation name of the annotations (grouped in an annotation view).
     *
     * @return The name of the given annotation view.
     * @see {@link AnnotationViewInfoInterface.getAnnotationViewName}
     */
    getAnnotationViewName(): string;
    /**
     * Gets the number of annotations in this annotation view.
     *
     * @return The actual number of annotations inside this annotation view.
     * @see {@link AnnotationViewInfoInterface.getAnnotationViewAnnotationCount}
     * @see {@link AnnotationViewInterface.getAnnotationViewAnnotationCount}
     * @see {@link AnnotationRendererInterface.getViewNbAnnotations}
     */
    getAnnotationViewAnnotationCount(): number;
}
export { AnnotationResultInterface };
/**
 * Represents an AnnotationGetterInterface request promise result.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link AnnotationGetterInterface.asyncFetchAnnotationViews}
 * @category Data Retrievers
 */
interface AsyncAnnotationResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     */
    value?: Array<AnnotationResultInterface>;
}
export { AsyncAnnotationResult };
/**
 * The AnnotationGetterInterface interface is used to fetch annotations views in the DMU.
 *
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * The AnnotationGetterInterface interfaces are created through the {@link DataSessionInterface.createAnnotationGetter} method.
 *
 * The list of signals the AnnotationGetterInterface may trigger is available in the {@link AnnotationGetterInterfaceSignal} enumeration.
 *
 * The annotation retrieval request is triggered through the {@link fetchAnnotationViews} method. The result is not available right away, but the
 * event {@link AnnotationGetterInterfaceSignal.AnnotationFetchReady} is triggered when the result of the AnnotationGetterInterface is available.
 * The result is available through the {@link getAnnotationViewsResult} function.<br>
 * As said in <a href="../index.html#annotations">Annotations</a>, the annotation views result consists in opaque annotation view(s) that must be
 * included in an {@link AnnotationRendererInterface} to be usable.
 * Each annotation view is itself composed of multiple annotations.
 *
 * Warning : there may be cases when the {@link getAnnotationViewsResult} is not available such as when the AnnotationGetterInterface is fetching data,
 * i.e. when {@link isRunning} returns true, or when the AnnotationGetterInterface has been cancelled, i.e. when {@link isCancelled} returns true.
 *
 * An AnnotationGetterInterface may be interrupted (cancelled) when the AnnotationGetterInterface is running and {@link cancelFetch}
 * is called. In such cases, the {@link AnnotationGetterInterfaceSignal.AnnotationFetchCancelled} signal is fired, and shortly after, {@link AnnotationGetterInterfaceSignal.AnnotationFetchReady}
 * signal is fired, but {@link getAnnotationViewsResult} will return undefined. Just call {@link fetchAnnotationViews} with another (or the same) {@link AnnotationViewInfoInterface} to trigger a new
 * fetch request.
 *
 * If you call multiple times {@link fetchAnnotationViews} before receiving the {@link AnnotationGetterInterfaceSignal.AnnotationFetchReady},
 * only one {@link AnnotationGetterInterfaceSignal.AnnotationFetchReady}
 * will be sent with the content of the last call to {@link fetchAnnotationViews}.
 *
 * The process is as follows :
 * * retrieve an {@link AnnotationViewInfoInterface} from an idcard request.
 * * asynchronously fetch {@link AnnotationResultInterface} from the AnnotationGetterInterface.
 * * extract the {@link AnnotationViewInterface} and {@link Matrix4} from the {@link AnnotationResultInterface}.
 * * add the {@link AnnotationViewInterface} and {@link Matrix4} to the {@link AnnotationRendererInterface}.
 *
 * ```typescript
 * [[include:AnnotationGetterInterface_download.ts]]
 * ```
 * <br>
 * Or with async calls :
 *
 * ```typescript
 * [[include:AnnotationGetterInterface_asyncDownload.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link AnnotationGetterInterfaceSignal}
 * @see {@link AnnotationResultInterface}
 * @see {@link AnnotationViewInfoInterface}
 * @see {@link AnnotationViewInterface}
 * @category Data Retrievers
 */
interface AnnotationGetterInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Triggers (an) annotation view retrieval request(s).
     *
     * An event {@link AnnotationGetterInterfaceSignal.AnnotationFetchReady} is fired when the fetch procedure is finished,
     * then {@link getLastError | getLastError()} tells if the retrieval was correctly performed.
     *
     * Returns true if the annotation views fetch procedure is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed.
     *
     * @param pAnnotationViewInfos <div class="const">in</div> The annotation views to fetch.
     * @return `true` if the AnnotationGetterInterface has begun running.
     */
    fetchAnnotationViews(pAnnotationViewInfos: AnnotationViewInfoInterface | Array<AnnotationViewInfoInterface>): boolean;
    /**
     * Asynchronously gets (an) annotation view.
     *
     * Returns a promise that will be resolved with the reason, and the eventual result when the fetch request is finished or cancelled.
     *
     * @param pAnnotationViewInfos <div class="const">in</div> The annotation views to fetch.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the
     * requested Array<AnnotationResultInterface>.
     */
    asyncFetchAnnotationViews(pAnnotationViewInfos: AnnotationViewInfoInterface | Array<AnnotationViewInfoInterface>): Promise<AsyncAnnotationResult>;
    /**
     * Each call to {@link fetchAnnotationViews} is assigned a request id.
     *
     * This call tels the id of the last call to {@link fetchAnnotationViews}.
     *
     * @return The id of the last call to {@link fetchAnnotationViews}.
     */
    getLastRequestId(): string;
    /**
     * Cancels the computation of the fetch retrieval (if any).
     *
     * A {@link AnnotationGetterInterfaceSignal.AnnotationFetchCancelled} signal is emitted if the
     * AnnotationGetterInterface is retrieving data.
     *
     * @return `true` if the AnnotationGetterInterface was running, else false.
     * @see {@link AnnotationGetterInterfaceSignal.AnnotationFetchCancelled}
     */
    cancelFetch(): boolean;
    /**
     * Tells if a annotation retrieval procedure is running.
     *
     * This is the case after calling {@link fetchAnnotationViews}.
     *
     * @return `true` if an annotation retrieval procedure request is running.
     */
    isRunning(): boolean;
    /**
     * Tells if a annotation retrieval procedure has been cancelled.
     *
     * This is generally the case after calling {@link cancelFetch} when the AnnotationGetterInterface
     * is preforming an annotation retrieval procedure.
     *
     * @return `true` if the annotation retrieval procedure is cancelled.
     */
    isCancelled(): boolean;
    /**
     * Gets the last error message returned by the annotation retrieval procedure.
     *
     * @return The last error message.
     */
    getLastError(): string;
    /**
     * Gets the list of all annotation views from the last call to the {@link fetchAnnotationViews} function.
     *
     * An array of {@link AnnotationResultInterface} is returned if the AnnotationGetterInterface has finished computing.
     * Use {@link addEventListener} on the event {@link AnnotationGetterInterfaceSignal.AnnotationFetchReady} to
     * know when the AnnotationGetterInterface is ready.
     *
     * The result is of the size of the array of {@link AnnotationViewInfoInterface} that was used in the {@link fetchAnnotationViews} call.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> the list of matching documents, or undefined if the
     * AnnotationGetterInterface is fetching or if the AnnotationGetterInterface is in error or cancelled.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     */
    getAnnotationViewsResult(): Array<AnnotationResultInterface> | undefined;
}
export { AnnotationGetterInterface };
/**
 * An annotation id.
 *
 * Valid annotation ids are strictly positive.
 *
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @category 3D Rendering
 */
type tAnnotationId = number;
export { tAnnotationId };
/**
 * An annotation view id.
 *
 * Valid annotation view ids are strictly positive.
 *
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @category 3D Rendering
 */
type tAnnotationViewId = number;
export { tAnnotationViewId };
/**
 * The AnnotationViewParsingResultInterface is the attachment of the
 * {@link AnnotationRendererInterfaceSignal.AnnotationViewParsed} signal
 * sent by the {@link AnnotationRendererInterface} when new annotations are available.
 * The {@link AnnotationRendererInterfaceSignal} is sent after a call to
 * {@link AnnotationRendererInterface.addAnnotationView}.
 *
 * This object stores information about the annotation view that was parsed :
 * i.e. the annotations that were included inside it.
 *
 * Each annotations is assigned an id ({@link tAnnotationId}), from {@link annotationIdStart} to
 * {@link annotationIdStart} + {@link nbAnnotations} - 1 inclusive, and the added {@link AnnotationViewInterface} that was included
 * in the {@link AnnotationRendererInterface} is also assigned an id ({@link annotationViewId}).
 * A valid id (annotation or view id) is a strictly positive number (0 is invalid).
 * The {@link requestId} is the value that was returned by the successful call to {@link AnnotationRendererInterface.addAnnotationView}.
 *
 * In case of error, {@link annotationIdStart}, {@link annotationViewId} and {@link nbAnnotations} are negative (`-1` actually), {@link errorMessage}
 * is set with the reason for the error.
 *
 * The {@link warnings} field tells if some values were not handled correctly, but the {@link AnnotationViewInterface} can still be displayed.
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_display.ts]]
 * ```
 * <br>
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @category Events
 */
interface AnnotationViewParsingResultInterface {
    /**
     * The annotation view id (-1 if the parsing was not successful) of the annotation view contained in the given
     * {@link AnnotationViewInterface}.
     */
    annotationViewId: tAnnotationViewId;
    /**
     * The starting id (-1 if the parsing was not successful) of the annotation contained in the given
     * {@link AnnotationViewInterface}.
     *
     * Annotations are identified by annotationIdStart, annotationIdStart + 1, ..., annotationIdStart + nbAnnotations -1.
     */
    annotationIdStart: tAnnotationId;
    /**
     * The number of annotations (-1 if the parsing was not successful) inside the annotation view.
     */
    nbAnnotations: number;
    /**
     * The request id that was returned by the call to {@link AnnotationRendererInterface.addAnnotationView}.
     */
    requestId: number;
    /**
     * An error message if relevant (or empty if no error).
     */
    errorMessage: string;
    /**
     * Warning messages. If some content was not handled, warnings are there.
     */
    warnings: Array<string>;
}
export { AnnotationViewParsingResultInterface };
/**
 * The AnnotationRendererInterfaceSignal lists the signal that may be sent by the {@link AnnotationRendererInterface}.
 *
 * This signal is emitted by the {@link AnnotationRendererInterface} when it has finished parsing and inserted
 * a new annotation view after a successful call to {@link AnnotationRendererInterface.addAnnotationView}.
 *
 * The {@link AnnotationRendererInterface} may fire the following signal :
 * * {@link AnnotationViewParsed} : the {@link AnnotationRendererInterface} has finished parsing an {@link AnnotationViewInterface} with
 * an asynchronous call to {@link AnnotationRendererInterface.addAnnotationView}.
 *
 * Please not that if {@link AnnotationRendererInterface.addAnnotationView} returns -1, then no signal will be fired at all.
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_display.ts]]
 * ```
 * <br>
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link AnnotationRendererInterface}
 * @see {@link AnnotationViewInterface}
 * @category Events
 */
declare enum AnnotationRendererInterfaceSignal {
    /**
     * Signal sent by the {@link AnnotationRendererInterface} when it has finished parsing an {@link AnnotationViewInterface}.
     *
     * The attachment for events of this type is an {@link AnnotationViewParsingResultInterface}.
     *
     * This signal is emitted after a successful call to {@link AnnotationRendererInterface.addAnnotationView}.
     *
     * @see {@link AnnotationViewParsingResultInterface}
     */
    AnnotationViewParsed = "annotationViewParsed"
}
export { AnnotationRendererInterfaceSignal };
/**
 * Represents an AnnotationRendererInterface annotation parsing promise result.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link AnnotationRendererInterface.asyncAddAnnotationView}
 * @category 3D Rendering
 */
interface AsyncAnnotationViewParsingResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The error message in case of bad input.
     */
    errorMessage: string;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     */
    value?: AnnotationViewParsingResultInterface;
}
export { AsyncAnnotationViewParsingResult };
/**
 * The AnnotationInstanceState enum tells about the way annotations should be rendered.
 *
 * The way annotations are rendered is set by a `number` which is an `OR` of multiple bit flags.
 * Usually, annotations are hidden below a geometry if it is behind the given geometry.
 *
 * * Annotations are contained in three bucket lists of increasing importance : low, standard and high.
 * Annotations in a more important buckets are rendered on top of annotations in a lower importance bucket,
 * regardless of their depth (AIS_DepthPriorityLow, AIS_DepthPriorityStd, AIS_DepthPriorityHigh).
 * * Annotations can be visible or hidden (AIS_Visible).
 * * Annotations can be set to be visible even if a geometry is above it (AIS_Overprint).
 * * Annotations can be set to be visible if it is behind a geometry but with a color desaturation (AIS_AppearThroughGeometry).
 * * Annotations color can be changed to be pulsing (AIS_Highlight), usually to tell it is selected.
 *
 * In order to avoid setting bits that should not be changed and leave them unchanged, annotations rendering bits are set with
 * a mask telling which bits should be changed. This system allows to change the way annotation are rendered in a single call.
 *
 * <img src="media://annotations.svg" alt="annotation rendering flags effect"/>
 * <br>
 * Displaying annotations may be done with the following :
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_display.ts]]
 * ```
 * <br>
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @category 3D Rendering
 */
declare enum AnnotationInstanceState {
    /**
     * Default state.
     */
    AIS_Null = 0,
    /**
     * Low rendering priority.
     */
    AIS_DepthPriorityLow = 1,
    /**
     * Standard rendering priority.
     */
    AIS_DepthPriorityStd = 2,
    /**
     * High rendering priority.
     */
    AIS_DepthPriorityHigh = 3,
    /**
     * Mask to change priority (access only priority bits).
     */
    AIS_DepthPriorityMask = 3,
    /**
     * Visible bit.
     */
    AIS_Visible = 64,
    /**
     * Overprint bit.
     *
     * Annotations are rendered on top of geometries, be them above or behind the annotations.
     */
    AIS_Overprint = 128,
    /**
     * Highlight bit.
     *
     * Annotations are rendered with a pulsing color.
     */
    AIS_Highlight = 256,
    /**
     * Appear Through bit.
     *
     * Annotations are rendered on top of geometries, with a color desaturation if they are behind geometries.
     */
    AIS_AppearThroughGeometry = 512,
    /**
     * All bit mask.
     */
    AIS_AllBitMask = 963
}
export { AnnotationInstanceState };
/**
 * The AnnotationRendererInterface is the object used to include and display annotations in the 3D view.
 *
 * The AnnotationRendererInterface is part of the {@link InfiniteEngineInterface} and is obtained through
 * {@link InfiniteEngineInterface.getAnnotationRenderer}.
 * The AnnotationRendererInterface is created along with the {@link InfiniteEngineInterface}, and is contained inside it.
 *
 * When annotations are retrieved with the {@link AnnotationGetterInterface}, and contained in the {@link AnnotationResultInterface}, such annotations still need
 * to be parsed (to create triangles, shapes, images, etc...) to be included in the 3D view.
 * The AnnotationRendererInterface parses these annotations and sets the visibility flags of such annotations.
 *
 * Annotations are included in a 3D plane, and grouped together in views : the {@link AnnotationViewInterface}.
 *
 * Annotations are usually defined locally in a geometry frame. For this reason, the {@link AnnotationResultInterface} contains the matrix to set when
 * including an annotation view. An undefined matrix is equivalent to a unit matrix.
 * The {@link AnnotationViewInterface} contains a Local Matrix (immutable, cannot be changed), the {@link AnnotationViewInterface} is included in the AnnotationRendererInterface with an
 * application matrix, such that the resulting transform applied to the annotations contained in the {@link AnnotationViewInterface} is a World matrix :
 *
 * WorldMatrix = ApplicationMatrix x LocalMatrix.
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_matrices.ts]]
 * ```
 * <br>
 * The LocalMatrix is decomposed with 4 vectors : the LocalPlaneX, LocalPlaneY, LocalNormal, and LocalOrigin.
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_localMatrix.ts]]
 * ```
 * <br>
 * The same holds for the WorldMatrix.
 *
 * Each annotation and annotation view, when included in the AnnotationRendererInterface is assigned an id (a number) :
 * * {@link tAnnotationId} for annotations.
 * * {@link tAnnotationViewId} for annotation views.
 *
 * A valid id is a strictly positive number (0 is invalid).
 *
 * The way annotations are rendered is done through the use of an "OR" combination of bits : the {@link AnnotationInstanceState}.
 * Annotations state is set with the bit to set and the mask of bits to change.
 *
 * Say I want to change the visibility bit (only) of an annotation :
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_changeVisibility.ts]]
 * ```
 * <br>
 * Say I want to change the visibility bit and the priority of an annotation :
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_changeVisibilityPriority.ts]]
 * ```
 * <br>
 * <img src="media://annotations.svg" alt="annotation rendering flags effect">
 * <br>And the full example :
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_display.ts]]
 * ```
 * <br>
 * Or asynchronously :
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_asyncDisplay.ts]]
 * ```
 * <br>
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link AnnotationInstanceState}
 * @see {@link InfiniteEngineInterface}
 * @see {@link AnnotationGetterInterface}
 * @see {@link AnnotationResultInterface}
 * @see {@link AnnotationViewInterface}
 * @category 3D Rendering
 */
interface AnnotationRendererInterface extends EventDispatcherInterface {
    /**
     * Asynchronously adds an annotation view with a matrix, and sets the rendering state of the annotations
     * contained within.
     *
     * This function is asynchronous, the event {@link AnnotationRendererInterfaceSignal.AnnotationViewParsed} will be triggered
     * shortly after with the result of the annotation view parsing.
     *
     * If the function succeeds, a strictly positive number is returned, which is the id of the request, which should match the
     * {@link AnnotationViewParsingResultInterface.requestId} of the correspondent signal.
     *
     * If the function fails, -1 is returned and no signal will be triggered.
     *
     * `pApplicationMatrix` MUST be a rotation / translation matrix (no scale), there is no check to ensure that the matrix is correct.
     * You will experience some rendering and positioning trouble if pApplicationMatrix is not a rotation / translation matrix.
     *
     * @param pView <div class="const">in</div> The annotation view to include.
     * @param pApplicationMatrix <div class="const">in</div> The matrix of the annotation view. An undefined value is equivalent to a unit matrix.
     * @param pRenderingMask <div class="const">in</div> The mask of the rendering state of all the annotations.
     * @param pRenderingState <div class="const">in</div> The rendering state of all the annotations.
     * @return A strictly positive request id of the add procedure, -1 if there is an error.
     */
    addAnnotationView(pView: AnnotationViewInterface | string | ArrayBuffer, pApplicationMatrix: Matrix4 | undefined, pRenderingMask: number, pRenderingState: number): number;
    /**
     * Asynchronously adds an annotation view with a matrix, and sets the rendering state of the annotations
     * contained within with a promise.
     *
     * A promise is returned.
     *
     * `pApplicationMatrix` MUST be a rotation / translation matrix (no scale), there is no check to ensure that the matrix is correct.
     * You will experience some rendering and positioning trouble if pApplicationMatrix is not a rotation / translation matrix.
     *
     * @param pView <div class="const">in</div> The annotation view to include.
     * @param pApplicationMatrix <div class="const">in</div> The matrix of the annotation view. An undefined value is equivalent to a unit matrix.
     * @param pRenderingMask <div class="const">in</div> The mask of the rendering state of all the annotations.
     * @param pRenderingState <div class="const">in</div> The rendering state of all the annotations.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the parsing result.
     */
    asyncAddAnnotationView(pView: AnnotationViewInterface | string | ArrayBuffer, pApplicationMatrix: Matrix4 | undefined, pRenderingMask: number, pRenderingState: number): Promise<AsyncAnnotationViewParsingResult>;
    /**
     * Tells if the AnnotationRendererInterface is computing annotations at the moment.
     *
     * This tells if any {@link addAnnotationView} has been called but has not finished yet.
     *
     * @return `true` if the AnnotationRendererInterface is computing annotations.
     */
    isComputingAnnotations(): boolean;
    /**
     * Removes the views (i.e. all the annotations of the given views) from being rendered.
     *
     * @param pViews <div class="const">in</div> The view ids of the views that will be removed.
     * @param pRemovedAnnotations <div class="const">out</div> The annotation ids that have been removed.
     * @return The number of annotation views effectively removed.
     */
    removeAnnotationViews(pViews: Array<tAnnotationViewId> | Uint32Array, pRemovedAnnotations: Array<tAnnotationId>): number;
    /**
     * Removes all the annotations contained in the AnnotationRendererInterface.
     *
     * Any pending request with {@link addAnnotationView} will be cancelled and a {@link AnnotationRendererInterfaceSignal.AnnotationViewParsed} signal will be triggered
     * for each pending annotation view to be parsed with an error.
     */
    removeAllAnnotations(): void;
    /**
     * Gets all the annotations view ids handled by this annotation renderer.
     *
     * Calling `getViewIds(undefined)` will return the number of annotation views currently handled.
     * Calling getViewIds with an Uint32Array without sufficient length will return -1.
     *
     * Returns the number of annotation views handled by the annotation renderer, or -1 if pViewIds
     * is a Uint32Array without sufficient memory.
     *
     * @param pViewIds <div class="const">out</div> The ids of the annotation views.
     * @return The number of annotation views handled by the annotation renderer, or -1 if pViewIds
     * is a Uint32Array without sufficient memory.
     */
    getViewIds(pViewIds: Array<tAnnotationViewId> | Uint32Array | undefined): number;
    /**
     * Sets the new application matrix to put for the given view.
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     * This call replaces the matrix set by {@link addAnnotationView}.
     *
     * `pApplicationMatrix` MUST be a rotation / translation matrix (no scale), there is no check to ensure that the matrix is correct.
     * You will experience some rendering and positioning trouble if pApplicationMatrix is not a rotation / translation matrix.
     *
     * Returns true if the application matrix for the given view has been updated, false if pViewId is invalid or refer to a nonexistent
     * annotation view.
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to modify.
     * @param pApplicationMatrix <div class="const">in</div> The new application matrix (rotation / translation) to set.
     * @return `true` if the application matrix for the given view has been updated.
     */
    setViewApplicationMatrix(pViewId: tAnnotationViewId, pApplicationMatrix: Matrix4): boolean;
    /**
     * Gets the start annotation id of the given annotation view.
     *
     * Annotations contained within this annotation view are numbered from (including)
     * getViewStartAnnotationId(pViewId) to getViewStartAnnotationId(pViewId) + {@link getViewNbAnnotations}(pViewId) - 1.
     *
     * {@link getAnnotationViewId}(getViewStartAnnotationId(pViewId)) should return pViewId.
     *
     * Returns the start annotation id of the given annotation view, or -1 if pViewId could not be found.
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @return The start annotation id of the given annotation view, or -1 if pViewId could not be found.
     */
    getViewStartAnnotationId(pViewId: tAnnotationViewId): tAnnotationId;
    /**
     * Gets the number of annotations contained within the given annotation view.
     *
     * Annotations contained within this annotation view are numbered from (including)
     * {@link getViewStartAnnotationId}(pViewId) to {@link getViewStartAnnotationId}(pViewId) + getViewNbAnnotations(pViewId) - 1.
     *
     * Returns the number of annotations contained within the given annotation view, or -1 if pViewId could not be found.
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @return The number of annotations contained within the given annotation view, or -1 if pViewId could not be found.
     */
    getViewNbAnnotations(pViewId: tAnnotationViewId): number;
    /**
     * Gets the Axis Aligned Bounding Box (AABB) of the given annotation view.
     *
     * Please note that the World Plane Origin of the view is not necessary contained in such an AABB since
     * all annotations may be set with an offset from the Local Plane Origin (see {@link getViewWorldPlaneOrigin}).
     *
     * Returns true if pViewId refers to a known annotation view and pAABB is updated.
     * If the call fails, then pAABB is left unchanged.
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pAABB <div class="const">out</div> The resulting AABB of the given annotation view.
     * @return `true` if pViewId refers to a valid annotation view.
     */
    getViewWorldAABB(pViewId: tAnnotationViewId, pAABB: AABB): boolean;
    /**
     * Gets the application matrix set for the given annotation view.
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     * This call gets the matrix set by the last call to {@link addAnnotationView} or {@link setViewApplicationMatrix} for the
     * given annotation view.
     *
     * `pApplicationMatrix` will be a rotation / translation matrix (no scale).
     *
     * Returns true if the application matrix for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pApplicationMatrix is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationApplicationMatrix}({@link getViewStartAnnotationId}(pViewId), pApplicationMatrix).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pApplicationMatrix <div class="const">out</div> The application matrix for the given view.
     * @return `true` if the application matrix of the given annotation view has been retrieved.
     */
    getViewApplicationMatrix(pViewId: tAnnotationViewId, pApplicationMatrix: Matrix4): boolean;
    /**
     * Gets the computed World matrix for the given annotation view.
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     * This call computes the matrix set by the last call to {@link addAnnotationView} or {@link setViewApplicationMatrix} with the Local Matrix for the
     * given annotation view.
     *
     * `pWorldMatrix` will be a rotation / translation matrix (no scale).
     *
     * Returns true if the World matrix for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pWorldMatrix is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationWorldMatrix}({@link getViewStartAnnotationId}(pViewId), pWorldMatrix).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pWorldMatrix <div class="const">out</div> The World matrix for the given view.
     * @return `true` if the World matrix of the given annotation view has been retrieved.
     */
    getViewWorldMatrix(pViewId: tAnnotationViewId, pWorldMatrix: Matrix4): boolean;
    /**
     * Gets the Local matrix for the given annotation view.
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     * This call gets the immutable Local matrix of the given annotation view, as retrieved with {@link AnnotationViewInterface.getPlaneXAxis},
     * {@link AnnotationViewInterface.getPlaneYAxis}, {@link AnnotationViewInterface.getPlaneNormal}, {@link AnnotationViewInterface.getPlaneOrigin}.
     *
     * `pLocalMatrix` will be a rotation / translation matrix (no scale).
     *
     * Returns true if the Local matrix for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pLocalMatrix is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationLocalMatrix}({@link getViewStartAnnotationId}(pViewId), pLocalMatrix).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pLocalMatrix <div class="const">out</div> The Local matrix for the given view.
     * @return `true` if the Local matrix of the given annotation view has been retrieved.
     */
    getViewLocalMatrix(pViewId: tAnnotationViewId, pLocalMatrix: Matrix4): boolean;
    /**
     * Gets the translation part of the computed World matrix for the given annotation view.
     *
     * This is therefore the World Plane Origin.
     * `getViewWorldPlaneOrigin(pViewId, pWorldPlaneOrigin)` is equivalent to {@link getViewWorldMatrix}(pViewId, lMatrix), lMatrix.getColumn(3, pWorldPlaneOrigin).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the World Plane Origin for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pWorldPlaneOrigin is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationWorldPlaneOrigin}({@link getViewStartAnnotationId}(pViewId), pWorldPlaneOrigin).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pWorldPlaneOrigin <div class="const">out</div> The World Plane Origin for the given view.
     * @return `true` if the World Plane Origin of the given annotation view has been retrieved.
     */
    getViewWorldPlaneOrigin(pViewId: tAnnotationViewId, pWorldPlaneOrigin: Vector3): boolean;
    /**
     * Gets the normal part of the computed World matrix for the given annotation view.
     *
     * `getViewWorldPlaneNormal(pViewId, pWorldPlaneNormal)` is equivalent to {@link getViewWorldMatrix}(pViewId, lMatrix), lMatrix.getColumn(2, pWorldPlaneNormal).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the World Plane Normal for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pWorldPlaneNormal is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationWorldPlaneNormal}({@link getViewStartAnnotationId}(pViewId), pWorldPlaneNormal).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pWorldPlaneNormal <div class="const">out</div> The World Plane Normal for the given view.
     * @return `true` if the World Plane Normal of the given annotation view has been retrieved.
     */
    getViewWorldPlaneNormal(pViewId: tAnnotationViewId, pWorldPlaneNormal: Vector3): boolean;
    /**
     * Gets the X axis direction of the computed World matrix for the given annotation view.
     *
     * `getViewWorldPlaneXAxis(pViewId, pWorldPlaneXAxis)` is equivalent to {@link getViewWorldMatrix}(pViewId, lMatrix), lMatrix.getColumn(0, pWorldPlaneXAxis).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the World Plane X axis direction for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pWorldPlaneXAxis is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationWorldPlaneXAxis}({@link getViewStartAnnotationId}(pViewId), pWorldPlaneXAxis).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pWorldPlaneXAxis <div class="const">out</div> The World Plane X Axis for the given view.
     * @return `true` if the World Plane X Axis of the given annotation view has been retrieved.
     */
    getViewWorldPlaneXAxis(pViewId: tAnnotationViewId, pWorldPlaneXAxis: Vector3): boolean;
    /**
     * Gets the Y axis direction of the computed World matrix for the given annotation view.
     *
     * `getViewWorldPlaneYAxis(pViewId, pWorldPlaneYAxis)` is equivalent to {@link getViewWorldMatrix}(pViewId, lMatrix), lMatrix.getColumn(1, pWorldPlaneYAxis).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the World Plane Y axis direction for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pWorldPlaneYAxis is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationWorldPlaneYAxis}({@link getViewStartAnnotationId}(pViewId), pWorldPlaneYAxis).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pWorldPlaneYAxis <div class="const">out</div> The World Plane Y Axis for the given view.
     * @return `true` if the World Plane Y Axis of the given annotation view has been retrieved.
     */
    getViewWorldPlaneYAxis(pViewId: tAnnotationViewId, pWorldPlaneYAxis: Vector3): boolean;
    /**
     * Gets the translation part of the Local matrix for the given annotation view.
     *
     * This is therefore the Local Plane Origin.
     * `getViewLocalPlaneOrigin(pViewId, pLocalPlaneOrigin)` is equivalent to {@link getViewLocalMatrix}(pViewId, lMatrix), lMatrix.getColumn(3, pLocalPlaneOrigin).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the Local Plane Origin for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pLocalPlaneOrigin is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationLocalPlaneOrigin}({@link getViewStartAnnotationId}(pViewId), pLocalPlaneOrigin).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pLocalPlaneOrigin <div class="const">out</div> The Local Plane Origin for the given view.
     * @return `true` if the Local Plane Origin of the given annotation view has been retrieved.
     */
    getViewLocalPlaneOrigin(pViewId: tAnnotationViewId, pLocalPlaneOrigin: Vector3): boolean;
    /**
     * Gets the normal part of the Local matrix for the given annotation view.
     *
     * `getViewLocalPlaneNormal(pViewId, pLocalPlaneNormal)` is equivalent to {@link getViewLocalMatrix}(pViewId, lMatrix), lMatrix.getColumn(2, pLocalPlaneNormal).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the Local Plane Normal for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pLocalPlaneNormal is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationLocalPlaneNormal}({@link getViewStartAnnotationId}(pViewId), pLocalPlaneNormal).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pLocalPlaneNormal <div class="const">out</div> The Local Plane Normal for the given view.
     * @return `true` if the Local Plane Normal of the given annotation view has been retrieved.
     */
    getViewLocalPlaneNormal(pViewId: tAnnotationViewId, pLocalPlaneNormal: Vector3): boolean;
    /**
     * Gets the X Axis direction part of the Local matrix for the given annotation view.
     *
     * `getViewLocalPlaneXAxis(pViewId, pLocalPlaneXAxis)` is equivalent to {@link getViewLocalMatrix}(pViewId, lMatrix), lMatrix.getColumn(0, pLocalPlaneXAxis).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the Local Plane X Axis for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pLocalPlaneXAxis is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationLocalPlaneXAxis}({@link getViewStartAnnotationId}(pViewId), pLocalPlaneXAxis).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pLocalPlaneXAxis <div class="const">out</div> The Local Plane X Axis for the given view.
     * @return `true` if the Local Plane X Axis of the given annotation view has been retrieved.
     */
    getViewLocalPlaneXAxis(pViewId: tAnnotationViewId, pLocalPlaneXAxis: Vector3): boolean;
    /**
     * Gets the Y Axis direction part of the Local matrix for the given annotation view.
     *
     * `getViewLocalPlaneYAxis(pViewId, pLocalPlaneYAxis)` is equivalent to {@link getViewLocalMatrix}(pViewId, lMatrix), lMatrix.getColumn(1, pLocalPlaneYAxis).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the Local Plane Y Axis for the given view has been retrieved, false if pViewId is invalid or refer to a nonexistent
     * annotation view, in this last case, pLocalPlaneYAxis is left unchanged.
     *
     * This call is equivalent to calling {@link getAnnotationLocalPlaneYAxis}({@link getViewStartAnnotationId}(pViewId), pLocalPlaneYAxis).
     *
     * @param pViewId <div class="const">in</div> The <code>annotation view id</code> of the annotation view to fetch.
     * @param pLocalPlaneYAxis <div class="const">out</div> The Local Plane Y Axis for the given view.
     * @return `true` if the Local Plane Y Axis of the given annotation view has been retrieved.
     */
    getViewLocalPlaneYAxis(pViewId: tAnnotationViewId, pLocalPlaneYAxis: Vector3): boolean;
    /**
     * Gets all the annotations ids handled by this annotation renderer.
     *
     * Calling `getAnnotationsIds(undefined)` will return the number of annotations currently handled.
     * Calling getAnnotationsIds with an Uint32Array without sufficient length will return -1.
     *
     * Returns the number of annotations handled by the annotation renderer, or -1 if pAnnotationIds
     * is a Uint32Array without sufficient memory.
     *
     * @param pAnnotationIds <div class="const">out</div> The ids of the annotation views.
     * @return The number of annotations handled by the annotation renderer, or -1 if pAnnotationIds
     * is a Uint32Array without sufficient memory.
     */
    getAnnotationsIds(pAnnotationIds: Array<tAnnotationId> | Uint32Array | undefined): number;
    /**
     * Sets the rendering state of the given annotations.
     *
     * <img src="media://annotations.svg" alt="annotation rendering flags effect"><br>
     * Say I want to change the visibility bit and the priority of an annotation:
     *
     * ```typescript
     * [[include:AnnotationRendererInterface_changeVisibilityPriority.ts]]
     * ```
     * <br>
     * Returns true if the call was successful (i.e. pAnnotations is of the correct type).
     *
     * @param pAnnotations <div class="const">in</div> The annotation(s) whose state will be updated.
     * @param pRenderingMask <div class="const">in</div> The mask of the rendering state to update the given annotations.
     * @param pRenderingState <div class="const">in</div> The new rendering state of the given annotations.
     * @return `true` if the rendering state was set.
     */
    setAnnotationRenderingState(pAnnotations: Array<tAnnotationId> | Uint32Array | tAnnotationId, pRenderingMask: number, pRenderingState: number): boolean;
    /**
     * Sets the rendering state of all the annotations.
     *
     * <img src="media://annotations.svg" alt="annotation rendering flags effect">
     * <br>
     * Say I want to change the visibility bit and the priority of an annotation:
     * <br>
     * ```typescript
     * [[include:AnnotationRendererInterface_changeVisibilityPriority.ts]]
     * ```
     * <br>
     * Returns true if the annotation state and flags are correct (i.e. integers).
     *
     * @param pRenderingMask <div class="const">in</div> The mask of the rendering state to update the given annotations.
     * @param pRenderingState <div class="const">in</div> The new rendering state of the given annotations.
     * @return `true` if pRenderingMask and pRenderingState are integers.
     */
    setAnnotationRenderingStateForAll(pRenderingMask: number, pRenderingState: number): boolean;
    /**
     * Gets the rendering state of the given annotation.
     *
     * <img src="media://annotations.svg" alt="annotation rendering flags effect">
     * <br>
     * Returns an `OR`ed value of {@link AnnotationInstanceState} that reflect the rendering of the given annotation,
     * or -1 if the given `annotation id` does not exist.
     *
     * @param pAnnotationId <div class="const">in</div> The id of the annotation to fetch.
     * @return The annotation state of the given annotation, or -1 if pAnnotationId refers to an invalid or nonexistent annotation id.
     */
    getAnnotationRenderingState(pAnnotationId: tAnnotationId): number;
    /**
     * Gets the rendering state of the given annotations.
     *
     * Stores an `OR`ed value of {@link AnnotationInstanceState} that reflect the rendering of the given annotation,
     * or -1 if the given `annotation id` does not exist in pAnnotationStates.
     *
     * If pAnnotationStates is an Uint32Array and its length is strictly inferior to pAnnotationIds.length, then
     * only pAnnotationStates.length values will be computed (the first pAnnotationStates.length values of pAnnotationIds).
     *
     * `pAnnotationStates[i]` will contain the rendering state of pAnnotationIds[i].
     *
     * @param pAnnotationIds <div class="const">in</div> The ids of the annotations to fetch.
     * @param pAnnotationStates <div class="const">out</div> The annotation states of the annotations. Any negative value
     * means the annotation state of the given annotation could not be fetched.
     */
    getAnnotationsRenderingState(pAnnotationIds: Uint32Array | Array<tAnnotationId>, pAnnotationStates: Uint32Array | Array<number>): void;
    /**
     * Gets the annotation view id this annotation belongs to.
     *
     * `getAnnotationViewId({@link getViewStartAnnotationId}(pViewId))` should return pViewId.
     *
     * Returns the annotation view id of the given annotation, or -1 if pAnnotationId could not be found (i.e. no
     * annotation in the AnnotationRendererInterface has this id).
     *
     * @param pAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @return The annotation view id this annotation belongs to, or -1 if pAnnotationId could not be found.
     */
    getAnnotationViewId(pAnnotationId: tAnnotationId): tAnnotationViewId;
    /**
     * Gets the Axis Aligned Bounding Box (AABB) of the given annotation.
     *
     * Please note that the World Plane Origin of the view this annotation belongs to is not necessary contained in such an AABB since
     * the annotation may be set with an offset from the Local Plane Origin (see {@link getAnnotationWorldPlaneOrigin}).
     *
     * `pIsDynamic` tells if the AABB should be updated depending on the current camera orientation. Indeed, some
     * annotation(s) may be rendered as a billboard, and thus their bounding box should be a sphere centered on the
     * pivot on the annotation. `pIsDynamic` has no effect on annotations that are not billboard-ed, the result is the same.
     *
     * If pDynamic is true, then the current orientation of billboard-ed annotation is taken into account, if false, a sphere
     * centered on the pivot of the annotation is used to compute the AABB, and the result will always be the same for this annotation.
     *
     * Returns true if pAnnotationId refers to a known annotation and pAABB is updated.
     *
     * If the call fails, then pAABB is left unchanged.
     *
     * @param pAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pDynamic <div class="const">in</div> If true, then billboard-ed annotation AABB is computed with the current camera orientation.
     * @param pAABB <div class="const">out</div> The resulting AABB of the given annotation.
     * @return `true` if pAnnotationId refers to a valid annotation.
     */
    getAnnotationWorldAABB(pAnnotationId: tAnnotationId, pDynamic: boolean, pAABB: AABB): boolean;
    /**
     * Gets the application matrix set for the annotation view that contains this annotation.
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     * This call gets the matrix set by the last call to {@link addAnnotationView} or {@link setViewApplicationMatrix} for the
     * annotation view this annotation belongs to.
     *
     * `pApplicationMatrix` will be a rotation / translation matrix (no scale).
     *
     * Returns true if the application matrix for the view that contains the given annotation id has been retrieved,
     * false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, `pApplicationMatrix` is left unchanged.
     *
     * This call is equivalent to calling {@link getViewApplicationMatrix}({@link getAnnotationViewId}(pAnnotationId), pApplicationMatrix).
     *
     * @param pAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pApplicationMatrix <div class="const">out</div> The application matrix for the view that contains the given annotation id.
     * @return `true` if the application matrix of the given annotation has been retrieved.
     */
    getAnnotationApplicationMatrix(pAnnotationId: tAnnotationId, pApplicationMatrix: Matrix4): boolean;
    /**
     * Gets the computed World matrix for the annotation view that contains this annotation.
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     * This call computes the matrix set by the last call to {@link addAnnotationView} or {@link setViewApplicationMatrix} with the Local Matrix for the
     * annotation view this annotation belongs to.
     *
     * `pWorldMatrix` will be a rotation / translation matrix (no scale).
     *
     * Returns true if the World matrix for the view that contains the given annotation id has been retrieved, false if pAnnotationId is invalid or
     * refer to a nonexistent annotation, in this last case, `pWorldMatrix` is left unchanged.
     *
     * This call is equivalent to calling {@link getViewWorldMatrix}({@link getAnnotationViewId}(pAnnotationId), pWorldMatrix).
     *
     * @param pAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pWorldMatrix <div class="const">out</div> The World matrix for the view that contains the given annotation id.
     * @return `true` if the World matrix of the given annotation has been retrieved.
     */
    getAnnotationWorldMatrix(pAnnotationId: tAnnotationId, pWorldMatrix: Matrix4): boolean;
    /**
     * Gets the Local matrix for the annotation view that contains this annotation.
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     * This call gets the immutable Local matrix of the annotation view that contains this annotation, as retrieved with {@link AnnotationViewInterface.getPlaneXAxis},
     * {@link AnnotationViewInterface.getPlaneYAxis}, {@link AnnotationViewInterface.getPlaneNormal}, {@link AnnotationViewInterface.getPlaneOrigin}.
     *
     * `pLocalMatrix` will be a rotation / translation matrix (no scale).
     *
     * Returns true if the Local matrix for the view that contains this annotation has been retrieved, false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, `pLocalMatrix` is left unchanged.
     *
     * This call is equivalent to calling {@link getViewLocalMatrix}({@link getAnnotationViewId}(pAnnotationId), pLocalMatrix).
     *
     * @param pAnnotationId <div class="const">in</div> The <code>annotation view id</code> of the annotation to fetch.
     * @param pLocalMatrix <div class="const">out</div> The Local matrix for the view that contains the given annotation id.
     * @return `true` if the Local matrix of the given annotation has been retrieved.
     */
    getAnnotationLocalMatrix(pAnnotationId: tAnnotationId, pLocalMatrix: Matrix4): boolean;
    /**
     * Gets the translation part of the computed World matrix for the annotation view that contains this annotation.
     *
     * This is therefore the World Plane Origin.
     * `getAnnotationWorldPlaneOrigin(pAnnotationId, pWorldPlaneOrigin)` is equivalent to {@link getAnnotationWorldMatrix}(pAnnotationId, lMatrix), lMatrix.getColumn(3, pWorldPlaneOrigin).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the World Plane Origin for the view that contains this annotation has been retrieved, false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, pWorldPlaneOrigin is left unchanged.
     *
     * This call is equivalent to calling {@link getViewWorldPlaneOrigin}({@link getAnnotationViewId}(pAnnotationId), pWorldPlaneOrigin).
     *
     * @param pAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pWorldPlaneOrigin <div class="const">out</div> The World Plane Origin for the view that contains the given annotation id.
     * @return `true` if the World Plane Origin of the given annotation has been retrieved.
     */
    getAnnotationWorldPlaneOrigin(pAnnotationId: tAnnotationId, pWorldPlaneOrigin: Vector3): boolean;
    /**
     * Gets the normal part of the computed World matrix for the annotation view that contains this annotation.
     *
     * `getAnnotationWorldPlaneNormal(pAnnotationId, pWorldPlaneNormal)` is equivalent to {@link getAnnotationWorldMatrix}(pAnnotationId, lMatrix), lMatrix.getColumn(2, pWorldPlaneNormal).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the World Plane Normal for the view that contains this annotation has been retrieved, false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, pWorldPlaneNormal is left unchanged.
     *
     * This call is equivalent to calling {@link getViewWorldPlaneNormal}({@link getAnnotationViewId}(pAnnotationId), pWorldPlaneNormal).
     *
     * @param pApplicativeAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pWorldPlaneNormal <div class="const">out</div> The World Plane Normal for the view that contains the given annotation id.
     * @return `true` if the World Plane Normal of the given annotation has been retrieved.
     */
    getAnnotationWorldPlaneNormal(pApplicativeAnnotationId: tAnnotationId, pWorldPlaneNormal: Vector3): boolean;
    /**
     * Gets the X axis direction of the computed World matrix for the annotation view that contains this annotation.
     *
     * `getAnnotationWorldPlaneXAxis(pAnnotationId, pWorldPlaneXAxis)` is equivalent to {@link getAnnotationWorldMatrix}(pAnnotationId, lMatrix), lMatrix.getColumn(0, pWorldPlaneXAxis).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the World Plane X axis direction for the view that contains this annotation has been retrieved, false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, pWorldPlaneXAxis is left unchanged.
     *
     * This call is equivalent to calling {@link getViewWorldPlaneXAxis}({@link getAnnotationViewId}(pAnnotationId), pWorldPlaneXAxis).
     *
     * @param pApplicativeAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pWorldPlaneXAxis <div class="const">out</div> The World Plane X axis direction for the view that contains the given annotation id.
     * @return `true` if the World Plane X axis direction of the given annotation has been retrieved.
     */
    getAnnotationWorldPlaneXAxis(pApplicativeAnnotationId: tAnnotationId, pWorldPlaneXAxis: Vector3): boolean;
    /**
     * Gets the Y axis direction of the computed World matrix for the annotation view that contains this annotation.
     *
     * `getAnnotationWorldPlaneYAxis(pAnnotationId, pWorldPlaneYAxis)` is equivalent to {@link getAnnotationWorldMatrix}(pAnnotationId, lMatrix), lMatrix.getColumn(1, pWorldPlaneYAxis).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the World Plane Y axis direction for the view that contains this annotation has been retrieved, false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, pWorldPlaneYAxis is left unchanged.
     *
     * This call is equivalent to calling {@link getViewWorldPlaneYAxis}({@link getAnnotationViewId}(pAnnotationId), pWorldPlaneYAxis).
     *
     * @param pApplicativeAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pWorldPlaneYAxis <div class="const">out</div> The World Plane Y axis direction for the view that contains the given annotation id.
     * @return `true` if the World Plane Y axis direction of the given annotation has been retrieved.
     */
    getAnnotationWorldPlaneYAxis(pApplicativeAnnotationId: tAnnotationId, pWorldPlaneYAxis: Vector3): boolean;
    /**
     * Gets the translation part of the Local matrix for the annotation view that contains this annotation.
     *
     * This is therefore the Local Plane Origin.
     * `getAnnotationLocalPlaneOrigin(pAnnotationId, pLocalPlaneOrigin)` is equivalent to {@link getAnnotationLocalMatrix}(pAnnotationId, lMatrix), lMatrix.getColumn(3, pLocalPlaneOrigin).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the Local Plane Origin for the view that contains this annotation has been retrieved, false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, pLocalPlaneOrigin is left unchanged.
     *
     * This call is equivalent to calling {@link getViewLocalPlaneOrigin}({@link getAnnotationViewId}(pAnnotationId), pLocalPlaneOrigin).
     *
     * @param pAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pLocalPlaneOrigin <div class="const">out</div> The Local Plane Origin for the view that contains the given annotation id.
     * @return `true` if the Local Plane Origin of the given annotation has been retrieved.
     */
    getAnnotationLocalPlaneOrigin(pAnnotationId: tAnnotationId, pLocalPlaneOrigin: Vector3): boolean;
    /**
     * Gets the normal part of the Local matrix for the annotation view that contains this annotation.
     *
     * `getAnnotationLocalPlaneNormal(pAnnotationId, pLocalPlaneNormal)` is equivalent to {@link getAnnotationLocalMatrix}(pAnnotationId, lMatrix), lMatrix.getColumn(2, pLocalPlaneNormal).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the Local Plane Normal for the view that contains this annotation has been retrieved, false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, pLocalPlaneNormal is left unchanged.
     *
     * This call is equivalent to calling {@link getViewLocalPlaneNormal}({@link getAnnotationViewId}(pAnnotationId), pLocalPlaneNormal).
     *
     * @param pApplicativeAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pLocalPlaneNormal <div class="const">out</div> The Local Plane Normal for the view that contains the given annotation id.
     * @return `true` if the Local Plane Normal of the given annotation has been retrieved.
     */
    getAnnotationLocalPlaneNormal(pApplicativeAnnotationId: tAnnotationId, pLocalPlaneNormal: Vector3): boolean;
    /**
     * Gets the X Axis direction of the Local matrix for the annotation view that contains this annotation.
     *
     * `getAnnotationLocalPlaneXAxis(pAnnotationId, pLocalPlaneXAxis)` is equivalent to {@link getAnnotationLocalMatrix}(pAnnotationId, lMatrix), lMatrix.getColumn(0, pLocalPlaneXAxis).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the Local Plane X Axis direction for the view that contains this annotation has been retrieved, false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, pLocalPlaneXAxis is left unchanged.
     *
     * This call is equivalent to calling {@link getViewLocalPlaneXAxis}({@link getAnnotationViewId}(pAnnotationId), pLocalPlaneXAxis).
     *
     * @param pApplicativeAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pLocalPlaneXAxis <div class="const">out</div> The Local Plane X Axis direction for the view that contains the given annotation id.
     * @return `true` if the Local Plane X Axis direction of the given annotation has been retrieved.
     */
    getAnnotationLocalPlaneXAxis(pApplicativeAnnotationId: tAnnotationId, pLocalPlaneXAxis: Vector3): boolean;
    /**
     * Gets the Y Axis direction of the Local matrix for the annotation view that contains this annotation.
     *
     * `getAnnotationLocalPlaneYAxis(pAnnotationId, pLocalPlaneYAxis)` is equivalent to {@link getAnnotationLocalMatrix}(pAnnotationId, lMatrix), lMatrix.getColumn(1, pLocalPlaneYAxis).
     *
     * Keep in mind that : WorldMatrix = ApplicationMatrix x LocalMatrix.
     *
     * Returns true if the Local Plane Y Axis direction for the view that contains this annotation has been retrieved, false if pAnnotationId is invalid or refer to a nonexistent
     * annotation, in this last case, pLocalPlaneYAxis is left unchanged.
     *
     * This call is equivalent to calling {@link getViewLocalPlaneYAxis}({@link getAnnotationViewId}(pAnnotationId), pLocalPlaneYAxis).
     *
     * @param pApplicativeAnnotationId <div class="const">in</div> The <code>annotation id</code> of the annotation to fetch.
     * @param pLocalPlaneYAxis <div class="const">out</div> The Local Plane Y Axis direction for the view that contains the given annotation id.
     * @return `true` if the Local Plane Y Axis direction of the given annotation has been retrieved.
     */
    getAnnotationLocalPlaneYAxis(pApplicativeAnnotationId: tAnnotationId, pLocalPlaneYAxis: Vector3): boolean;
}
export { AnnotationRendererInterface };
/**
 * The AnnotationViewInfoInterface interface is used to store the annotations properties linked to a `part instance`.
 *
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * The AnnotationViewInfoInterface is retrieved when using a {@link PartInstanceInfoInterface.getAnnotationViews} i.e. when the metadata
 * attached to a `part instance` are retrieved from an id card request.
 *
 * The actual data will be retrieved by using an {@link AnnotationGetterInterface} since the AnnotationViewInfoInterface only stores
 * general properties of an annotation view, but not the annotation view internal data.
 *
 * ```typescript
 * [[include:AnnotationGetterInterface_download.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:AnnotationGetterInterface_asyncDownload.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link IdCardGetterInterface}
 * @see {@link AnnotationGetterInterface}
 * @see {@link AnnotationGetterInterfaceSignal}
 * @category Data Retrievers
 */
interface AnnotationViewInfoInterface {
    /**
     * Gets the annotation type name of the annotations (grouped in an annotation view).
     *
     * All annotation types available in the DMU can be retrieved with {@link DataSessionInterface.getAnnotationTypes}.
     *
     * @return The type name of the given annotation view.
     * @see {@link DataSessionInterface}
     */
    getAnnotationViewTypeName(): string;
    /**
     * Gets the annotation name of the annotations (grouped in an annotation view).
     *
     * @return The name of the given annotation view.
     */
    getAnnotationViewName(): string;
    /**
     * Gets the number of annotations in this annotation view.
     *
     * @return The actual number of annotations inside this annotation view.
     */
    getAnnotationViewAnnotationCount(): number;
}
export { AnnotationViewInfoInterface };
/**
 * The AnnotationViewInterface interface contains the annotations data of an annotation view.
 *
 * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
 *
 * The AnnotationViewInterface features a 3D plane (X axis, Y Axis, Normal, Origin) from which 2D data (texts,
 * 2D shapes, images) are positioned. The AnnotationViewInterface represents a set of annotations
 * (Functional Tolerancing and Annotations, Product Manufacturing Information, Measures, etc ...) placed in a 3D
 * plane. AnnotationViewInterfaces should be included in the {@link AnnotationRendererInterface}.
 *
 * The AnnotationViewInterface is retrieved from an {@link AnnotationResultInterface} that
 * was fetched from an {@link AnnotationGetterInterface}.
 *
 * Below is some code to retrieve an AnnotationViewInterface from an id card request.
 *
 * ```typescript
 * [[include:AnnotationGetterInterface_download.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:AnnotationGetterInterface_asyncDownload.ts]]
 * ```
 * <br>
 * Once retrieved, an AnnotationViewInterface may be included in an {@link AnnotationRendererInterface} to be displayed.
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_display.ts]]
 * ```
 * <br>
 * Or asynchronously :
 *
 * ```typescript
 * [[include:AnnotationRendererInterface_asyncDisplay.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link AnnotationGetterInterface}
 * @see {@link AnnotationRendererInterface}
 * @see {@link AnnotationResultInterface}
 * @category 3D Rendering
 */
interface AnnotationViewInterface {
    /**
     * Gets the origin of the annotation view in the annotation frame.
     *
     * If pPlaneOrigin is omitted, then a new Vector3 is allocated.
     *
     * @param pPlaneOrigin <div class="const">out</div> The optional 3D point from which any annotation data is calculated.
     * @return `pPlaneOrigin`.
     */
    getPlaneOrigin(pPlaneOrigin?: Vector3): Vector3;
    /**
     * An AnnotationViewInterface is defined as a plane.
     *
     * The plane normal is the normal to the plane of this AnnotationViewInterface (unit length).
     *
     * If pPlaneNormal is omitted, then a new Vector3 is allocated.
     *
     * NB : getPlaneNormal = cross({@link getPlaneXAxis}, {@link getPlaneYAxis}).
     *
     * @param pPlaneNormal <div class="const">out</div> The optional normal of the plane of the AnnotationViewInterface.
     * @return `pPlaneNormal`.
     */
    getPlaneNormal(pPlaneNormal?: Vector3): Vector3;
    /**
     * An AnnotationViewInterface is defined as a plane.
     *
     * The pPlaneXAxis is the directional unit vector in the annotation view to the right of the annotation view.
     * This is usually the direction of the text contained inside any annotation of this AnnotationViewInterface.
     *
     * If pPlaneXAxis is omitted, then a new Vector3 is allocated.
     *
     * NB : getPlaneXAxis = cross({@link getPlaneYAxis}, {@link getPlaneNormal}).
     *
     * @param pPlaneXAxis <div class="const">out</div> The optional right vector of the plane of the AnnotationViewInterface.
     * @return `pPlaneXAxis`.
     */
    getPlaneXAxis(pPlaneXAxis?: Vector3): Vector3;
    /**
     * An AnnotationViewInterface is defined as a plane.
     *
     * The pPlaneYAxis is the directional unit vector in the annotation view to the up of the annotation view.
     * This is usually the direction of the height of the text contained inside any annotation of this AnnotationViewInterface.
     *
     * If pPlaneYAxis is omitted, then a new Vector3 is allocated.
     *
     * NB : pPlaneYAxis = cross({@link getPlaneNormal}, {@link getPlaneXAxis}).
     *
     * @param pPlaneYAxis <div class="const">out</div> The optional up vector of the plane of the AnnotationViewInterface.
     * @return `pPlaneYAxis`.
     */
    getPlaneYAxis(pPlaneYAxis?: Vector3): Vector3;
    /**
     * Gets the number of annotations in this AnnotationViewInterface.
     *
     * @return The actual number of annotations inside this AnnotationViewInterface.
     * @see {@link AnnotationViewInfoInterface.getAnnotationViewAnnotationCount}
     * @see {@link AnnotationResultInterface.getAnnotationViewAnnotationCount}
     */
    getAnnotationViewAnnotationCount(): number;
}
export { AnnotationViewInterface };
/**
 * The AttributeType enum tells the type of the attributes of the elements of the DMU.
 *
 * The DMU contains a collection of pieces (3D or not). All these pieces have information located inside them.
 * The AttributeType enum tells about the type of the attributes.
 *
 * Any attribute may have the given type, an array of the given type or null.
 * Any part instance may have also other metadata, in this case, such metadata cannot be filtered, since it cannot be indexed.
 * Such data that cannot be indexed can be found in the {@link ObjectMetadataInterface.nonindexedpartmd}, {@link ObjectMetadataInterface.nonindexedlinkmd}
 * and {@link ObjectMetadataInterface.nonindexedinstancemd} fields.
 *
 * Assume we have the following attributes dictionary :
 *
 * ```json
 * [[include:dictionary.json]]
 * ```
 * <br>
 * When requesting the metadata of a part instance, we get the following result : ({@link IdCardGetterInterface}, {@link ObjectMetadataInterface}) :
 *
 * ```json
 * [[include:metadata.js]]
 * ```
 * <br>
 * AttributeTypes are contained inside an {@link AttributeInfoInterface}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @category Metadata
 */
declare enum AttributeType {
    /**
     * Attribute is a string.
     *
     * Any metadata of this type may be a string, an array of string, or null.
     * There may be cases when metadata of this type are not of the expected type (DMU provider error).
     */
    ATTR_STRING = 0,
    /**
     * Attribute is a number.
     *
     * Any metadata of this type may be a number, an array of number, or null.
     * There may be cases when metadata of this type are not of the expected type (DMU provider error).
     */
    ATTR_NUMBER = 1,
    /**
     * Attribute is a date.
     *
     * Any metadata of this type may be a date, an array of date or null.
     * Dates are expressed as the **number of milliseconds since January 1, 1970, 00:00:00 UTC**.
     * There may be cases when metadata of this type are not of the expected type (DMU provider error).
     */
    ATTR_DATE = 2,
    /**
     * Attribute is a boolean.
     *
     * Any metadata of this type may be a boolean, an array of boolean, or null.
     * There may be cases when metadata of this type are not of the expected type (DMU provider error).
     */
    ATTR_BOOLEAN = 3,
    /**
     * Attribute is a range of date.
     *
     * Any metadata of this type may be a date, an array of range of date, or null.
     * Dates are expressed as the **number of milliseconds since January 1, 1970, 00:00:00 UTC**.
     * There may be cases when metadata of this type are not of the expected type (DMU provider error).
     */
    ATTR_DATE_RANGE = 4,
    /**
     * Attribute is a range of number.
     *
     * Any metadata of this type may be a number, an array of range of number, or null.
     * There may be cases when metadata of this type are not of the expected type (DMU provider error).
     */
    ATTR_NUMBER_RANGE = 5,
    /**
     * Attribute is unknown, an object or invalid.
     */
    ATTR_UNKNOWN = 6
}
export { AttributeType };
/**
 * The AttributeNumericType enum tells the numeric type of the attributes of the elements of the DMU.
 *
 * The DMU contains a collection of pieces (3D or not). All these pieces have information located inside them.
 * The AttributeNumericType enum tells about the numeric type of the attributes, or ATTR_NUM_UNKNOWN if the
 * numeric type is not known.
 *
 * Javascript only exposes untyped numbers, but the 3djuump Infinite API uses several numeric types internally.
 * If an attribute has the type ATTR_NUMBER or ATTR_NUMBER_RANGE, then its AttributeNumericType is known, else
 * all other attribute types have the numeric type ATTR_NUM_UNKNOWN.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @category Metadata
 */
declare enum AttributeNumericType {
    /**
     * Unknown attribute numeric type, the attribute is not numeric.
     */
    ATTR_NUM_UNKNOWN = -1,
    /**
     * The attribute has the numeric type byte (8 bits) ranging from [-128,127].
     */
    ATTR_NUM_BYTE = 0,
    /**
     * The attribute has the numeric type short (16 bits) ranging from [-32768,32767].
     */
    ATTR_NUM_SHORT = 1,
    /**
     * The attribute has the numeric type int (32 bits) ranging from [-2147483648,2147483647].
     */
    ATTR_NUM_INT = 2,
    /**
     * The attribute has the numeric type long long (64 bits) ranging from [-9223372036854775808,9223372036854775807].
     */
    ATTR_NUM_LONG_LONG = 3,
    /**
     * The attribute has the numeric type float (32 bits) ranging from [-3.402823466e+38,3.402823466e+38].
     */
    ATTR_NUM_FLOAT = 4,
    /**
     * The attribute has the numeric type double (64 bits) ranging from [-Number.MAX_VALUE,Number.MAX_VALUE].
     */
    ATTR_NUM_DOUBLE = 5
}
export { AttributeNumericType };
/**
 * The AttributeInfoInterface gives information about the attributes of a `part instance`.
 *
 * The DMU contains a collection of pieces (3D or not). All these pieces have information located inside them.
 * The Attribute Info interface stores the type of the attribute, its name, the full "path of this attribute" (in case of nested attributes)
 * and some range / min / max values if available.
 *
 * Any AttributeInfoInterface with the type {@link AttributeType.ATTR_STRING} ({@link getAttributeType}), and with a limited number
 * of available values is called `enumerable`.
 *
 * Any `enumerable` attribute has a special use in the {@link FilterAttributeInterface}.
 *
 * ```typescript
 * [[include:AttributesDictionaryInterface_iterate.ts]]
 * ```
 * <br>
 * There may be cases when the results from an idcard request ({@link ObjectMetadataInterface} inside the {@link PartInstanceInfoInterface}) are
 * not of the expected type, such data cannot be requested since the field content is not indexed.
 * In this case the attribute name is included in the {@link ObjectMetadataInterface.nonindexedpartmd},
 * {@link ObjectMetadataInterface.nonindexedlinkmd} or {@link ObjectMetadataInterface.nonindexedinstancemd} fields,
 * depending on the case.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @category Metadata
 */
interface AttributeInfoInterface {
    /**
     * Gets the name of the attribute. This name is the one stored inside the database.
     *
     * @return The name of the attribute in the database.
     */
    getAttributeName(): string;
    /**
     * If this attribute is included in a sub-document (nested), gets the path of nested array in parent.
     *
     * All attributes that share the same nested path are included in the same nested document.
     *
     * @return The nested path if the attribute is nested, undefined if not.
     */
    getAttributeNestedPath(): string;
    /**
     * Gets the type of the attribute.
     *
     * @return The type of the attribute.
     * @see {@link AttributeType}
     */
    getAttributeType(): AttributeType;
    /**
     * Gets the numeric type of the attribute.
     *
     * It the attribute has the type `AttributeType.ATTR_NUMBER` or `AttributeType.ATTR_NUMBER_RANGE` then
     * the real numeric type is known (and not `AttributeNumericType.ATTR_NUM_UNKNOWN`).
     * This allows to create more precise GUI elements and know the precise boundaries for acceptable
     * values for this attribute.
     *
     * @return The numeric type of the attribute.
     * @see {@link AttributeNumericType}
     * @see {@link getAttributeMinFromNumericType}
     * @see {@link getAttributeMaxFromNumericType}
     */
    getAttributeNumericType(): AttributeNumericType;
    /**
     * Gets the list of existing values for this attribute in the database.
     *
     * If the number of values exceeds the directory enumeration limit, then it returns undefined.
     *
     * @return <div class="const">const</div> The list of existing values in the database.
     */
    getAttributeValues(): Array<string | number> | undefined;
    /**
     * Gets the minimum value of the attribute in the database (only for number and date attributes, and if known) else returns undefined.
     *
     * @return The minimum existing value for this attribute.
     */
    getAttributeMin(): number | undefined;
    /**
     * Gets the maximum value of the attribute in the database (only for number and date attributes, and if known) else returns undefined.
     *
     * @return The maximum existing value for this attribute.
     */
    getAttributeMax(): number | undefined;
    /**
     * Gets the minimum value of the attribute inferred from its numeric type (byte, short, int, long long, float, double).
     *
     * Javascript only exposes numbers, but the 3djuump Infinite API exposes several numeric types. All numeric types
     * have the type `AttributeType.ATTR_NUMBER` or `AttributeType.ATTR_NUMBER_RANGE`. It may be interesting to know the
     * minimum value inferred by its type.
     *
     * @return The minimum numeric value inferred by the type of this attribute.
     */
    getAttributeMinFromNumericType(): number;
    /**
     * Gets the maximum value of the attribute inferred from its numeric type (byte, short, int, long long, float, double).
     *
     * Javascript only exposes numbers, but the 3djuump Infinite API exposes several numeric types. All numeric types
     * have the type `AttributeType.ATTR_NUMBER` or `AttributeType.ATTR_NUMBER_RANGE`. It may be interesting to know the
     * maximum value inferred by its type.
     *
     * @return The maximum numeric value inferred by the type of this attribute.
     */
    getAttributeMaxFromNumericType(): number;
    /**
     * Tells if the attribute is `enumerable`.
     *
     * Any AttributeInfoInterface with the type {@link AttributeType.ATTR_STRING} ({@link getAttributeType}), and with a limited number
     * of available values is called `enumerable`.
     *
     * Any `enumerable` attribute has a special use in the {@link FilterAttributeInterface}.
     *
     * @return `true` if the attribute is enumerable.
     */
    isEnumerable(): boolean;
}
export { AttributeInfoInterface };
/**
 * The AttributesDictionaryInterface stores all the information of all the different attributes in the DMU.
 *
 * The DMU contains a collection of pieces (3D or not). All these pieces have information located inside them.
 * The Attribute Dictionary interface is used to query information of all the attributes available in the DMU.
 *
 * ```typescript
 * [[include:AttributesDictionaryInterface_iterate.ts]]
 * ```
 * <br>
 * The AttributesDictionaryInterface is available through the {@link DataSessionInterface.getAttributesDictionary} and
 * {@link DataSessionInterface.getAttributesDictionary} functions.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @see {@link DataSessionInterface}
 * @category Metadata
 */
interface AttributesDictionaryInterface {
    /**
     * Gets the info of an attribute given its name, or undefined if such an attribute does not exist.
     *
     * @param pAttributeName <div class="const">in</div> The name of the attribute to fetch.
     * @return <div class="const">const</div> The attribute info of the given attribute, or undefined if not found.
     */
    getAttributeInfo(pAttributeName: string): AttributeInfoInterface | undefined;
    /**
     * Gets all the available attributes and their information.
     *
     * @return <div class="const">const</div> The list of all attributes.
     */
    getDictionary(): Array<AttributeInfoInterface>;
}
export { AttributesDictionaryInterface };
/**
 * The Browser interface gives information on the type of browser the api is running on.
 *
 * This interface is now deprecated. Browser detection system is now handled by the {@link InfiniteApiControllerInterface}.
 *
 * The browser stores the name and the version of the current internet browser.
 *
 * [[include:constStyle.md]]<br><div class="const">Deprecated</div><br>
 * @deprecated Please use {@link InfiniteApiControllerInterface}.
 * @see {@link InfiniteApiControllerInterface}
 * @category Deprecated
 */
interface Browser {
    /**
     * The name of the browser (opera|chrome|safari|firefox|IE).
     */
    name: string;
    /**
     * The version of the browser, useful to get the capabilities of the browser.
     */
    version: string;
}
export { Browser };
/**
 * The BrowserDetector provides a way to detect the current browser the api is running on.
 *
 * This class is now deprecated. Browser detection system is not handled by the {@link InfiniteApiControllerInterface}.
 *
 * This class only provides a static function to detect the current Browser.
 *
 * [[include:constStyle.md]]<br><div class="const">Deprecated</div><br>
 * @see {@link Browser}
 * @deprecated Please use {@link InfiniteApiControllerInterface}.
 * @see {@link InfiniteApiControllerInterface}
 * @category Deprecated
 */
declare class BrowserDetector {
    /**
     * Detects the current browser.
     *
     * This class is now deprecated. Browser detection system is not handled by the {@link InfiniteApiControllerInterface}.
     *
     * @return <div class="const">const</div> The Browser object containing the browser information.
     * @deprecated Please use {@link InfiniteApiControllerInterface}.
     * @see {@link InfiniteApiControllerInterface}
     */
    static GetBrowser(): Browser;
}
export { BrowserDetector };
/**
 * The CameraManagerInterfaceSignal list all the signals that may be sent by the {@link CameraManagerInterface}.
 *
 * These signals are emitted by the {@link CameraManagerInterface} when the camera is modified or an animation is triggered.
 *
 * 4 types of signals are emitted :
 * * When the camera orientation is changed : CameraFrameChanged.
 * * When the camera location (position) is changed : CameraLocationChanged.
 * * When an animation is triggered : CameraAnimationStarted.
 * * When an animation is ended : CameraAnimationEnded.
 *
 * If a move changes the location and the position of the camera then
 * the CameraFrameChanged signal is always sent prior to the CameraLocationChanged.
 * These signals are not sent when the user presses the mouse, but only once per rendered frame when the
 * user inputs are processed. Awaiting these signals is not a reliable way to detect user inputs as they are emitted.
 * User inputs are processed once per frame, in batch.
 *
 * ```typescript
 * [[include:CameraManagerInterface_signal.ts]]
 * ```
 * <br>
 * Please refer to the {@link CameraManagerInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link CameraManagerInterface}
 * @category Events
 */
declare enum CameraManagerInterfaceSignal {
    /**
     * Signal sent by the {@link CameraManagerInterface} when the camera location (position) is changed.
     *
     * No attachment for events of this type.
     */
    CameraLocationChanged = "cameraLocationChanged",
    /**
     * Signal sent by the {@link CameraManagerInterface} when the camera orientation is changed.
     *
     * No attachment for events of this type.
     */
    CameraFrameChanged = "cameraFrameChanged",
    /**
     * Signal sent by the {@link CameraManagerInterface} when a camera animation is started.
     *
     * The attachment for events of this type is a number, the duration in milliseconds of the animation.
     */
    CameraAnimationStarted = "cameraAnimationStarted",
    /**
     * Signal sent by the {@link CameraManagerInterface} when a camera animation is over.
     *
     * No attachment for events of this type.
     */
    CameraAnimationEnded = "cameraAnimationEnded",
    /**
     * Signal sent by the {@link CameraManagerInterface} when the computation of the camera has ended (sent event if the position has not changed).
     *
     * No attachment for events of this type.
     */
    CameraComputationEnded = "cameraComputationEnded"
}
export { CameraManagerInterfaceSignal };
/**
 * The CameraControllerMode enumeration tells how user input are processed to move the camera.
 *
 * The CameraManagerInterface have several ways to process user inputs and deduce a camera
 * position and orientation from them.
 * This enum governs the way the camera is changed according to these user inputs.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link CameraManagerInterface}
 * @category 3D Rendering
 */
declare enum CameraControllerMode {
    /**
     * Orbit mode.
     *
     * This mode operates with a center of interest, the camera rotates in a sphere around this
     * center of interest. Rolling is not permitted.
     * The end-user can move the camera around its center of
     * interest, zoom in or out, change the center of interest and pan.
     */
    CCM_ORBIT = 0,
    /**
     * Examine mode.
     *
     * Very similar to the Orbit mode but follows a metaphor more akin to the one
     * used in a popular 3D CAD Design tool. It may feel more natural than the orbit mode to CAD
     * designers.
     */
    CCM_EXAMINE = 1,
    /**
     * None mode.
     *
     * This mode disables the camera manager i.e. no input like mouse, keyboard or touch event changes
     * the camera location and orientation.
     */
    CCM_NONE = 2,
    /**
     * Fly mode.
     *
     * This mode operates like a quake-like game controller. The mouse control the orientation of the view
     * and the position will be changed with keyboard, gamepad or with auto translation on camera direction.
     */
    CCM_FLY = 3,
    /**
     * (WebXR) XR mode.
     *
     * This mode operates automatically when an XR session is started, and cannot be changed till the WebXR session is stopped.
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_webxr.ts]]
     * ```
     * <br>
     * Please see {@link InfiniteEngineInterface.setView}.
     */
    CCM_XR = 4
}
export { CameraControllerMode };
/**
 * The ViewFrustum enumeration tells the position of the values when querying {@link CameraManagerInterface.getFrustum}.
 *
 * The frustum values are used to compute the perspective projection of the virtual camera.
 *
 * <img src="https://upload.wikimedia.org/wikipedia/commons/0/02/ViewFrustum.svg" alt="frustum illustration">
 * <br>
 * The camera is composed of 6 planes : A near and far plane orthogonal to the camera at the distance respectively near and far.
 * Up, Bottom, Right and Left planes go through the camera location and intersect the near plane forming 4 points (see the figure).<br>
 * The coordinates of these 4 points can be expressed as :<br>
 * Be P0 the bottom left point, P1 the bottom right point, P2 the top right point, P3 the top left point.<br>
 * Be D the normalized directional vector of the camera, U the normalized Up vector of the camera, R the normalized right vector of the camera and
 * L the camera location.<br>
 * P0 = L + Dmin * D + Rmin * R + Umin * U <br>
 * P1 = L + Dmin * D + Rmax * R + Umin * U <br>
 * P2 = L + Dmin * D + Rmax * R + Umax * U <br>
 * P3 = L + Dmin * D + Rmin * R + Umax * U <br>
 * <br>
 * In cases of symmetrical frustum (usually the case) :<br>
 * Rmin = -Rmax<br>
 * Dmin = -Dmax.
 *
 * ```typescript
 * [[include:CameraManagerInterface_ViewFrustum.ts]]
 * ```
 * <br>
 * There are a lot of online courses available if you want to get more information about perspective projections.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link CameraManagerInterface}
 * @category 3D Rendering
 */
declare enum ViewFrustum {
    /**
     * Invalid frustum offset.
     */
    VF_INVALID = -1,
    /**
     * Offset to get the distance from the camera to the near plane.
     */
    VF_DMIN = 0,
    /**
     * Offset to get the distance from the camera to the far plane.
     */
    VF_DMAX = 1,
    /**
     * Offset to get the "distance" to the down plane.
     */
    VF_UMIN = 2,
    /**
     * Offset to get the "distance" to the up plane.
     */
    VF_UMAX = 3,
    /**
     * Offset to get the "distance" to the left plane.
     */
    VF_RMIN = 4,
    /**
     * Offset to get the "distance" to the right plane.
     */
    VF_RMAX = 5,
    /**
     * Number of values "in" the frustum.
     */
    VF_FRUSTUM_SIZE = 6
}
export { ViewFrustum };
/**
 * (WebXR) An InfiniteXRReferenceSpace is actually an XRReferenceSpace.
 *
 * The XRReferenceSpace type is not used directly to avoid having all the times the XR types included even if not relevant.
 * Please cast InfiniteXRReferenceSpace to XRReferenceSpace.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @category 3D Rendering
 */
type InfiniteXRReferenceSpace = any;
export { InfiniteXRReferenceSpace };
/**
 * The CameraManagerInterface interface is used to manipulate the camera, and thus to change the viewpoint.
 *
 * This interface allows to create camera animations and manipulate the orientation cube (a cube that
 * allows the user to see its current orientation).
 *
 * The CameraManagerInterface also features a way to change the camera manipulator, which is the way user inputs are processed
 * to infer a camera move, by using the {@link CameraControllerMode}. The CameraManagerInterface uses the {@link CameraControllerMode.CCM_ORBIT} as default.
 * Whenever the camera location and orientation (frame) are changed, {@link CameraManagerInterfaceSignal} signals are sent.
 *
 * This interface also allows to set the camera sensibility, which is the default distance and rotation angle induced by a single user input.
 * This may increase or lower the camera speed. The camera sensitivity is a strictly positive number, that defaults to 1.
 *
 * If the developer wants to create its own camera manipulator, he may disable the CameraManagerInterface by :
 * ```typescript
 * [[include:CameraManagerInterface_controller_mode.ts]]
 * ```
 * <br>
 * There is no way to create a new CameraManagerInterface, the camera handler is owned and accessed by the InfiniteEngineInterface.
 *
 * The camera features a perspective projection (pyramid) expressed as frustum values. See {@link ViewFrustum}.
 * The frustum values are used to compute the perspective projection of the virtual camera.
 * There are a lot of online courses available if you want to get more information about perspective projections.
 *
 * <img src="https://upload.wikimedia.org/wikipedia/commons/0/02/ViewFrustum.svg" alt="frustum illustration">
 * <br>
 * The camera is composed of 6 planes : A near and far plane orthogonal to the camera at the distance respectively near and far.
 * Up, Bottom, Right and Left planes go through the camera location and intersect the near plane forming 4 points (see the figure).<br>
 * The coordinates of these 4 points can be expressed as :<br>
 * Be P0 the bottom left point, P1 the bottom right point, P2 the top right point, P3 the top left point.<br>
 * Be D the normalized directional vector of the camera, U the normalized Up vector of the camera, R the normalized right vector of the camera and
 * L the camera location.<br>
 * P0 = L + Dmin * D + Rmin * R + Umin * U <br>
 * P1 = L + Dmin * D + Rmax * R + Umin * U <br>
 * P2 = L + Dmin * D + Rmax * R + Umax * U <br>
 * P3 = L + Dmin * D + Rmin * R + Umax * U <br>
 * <br>
 * In cases of symmetrical frustum (usually the case) :<br>
 * Rmin = -Rmax<br>
 * Dmin = -Dmax.
 *
 * ```typescript
 * [[include:CameraManagerInterface_properties.ts]]
 * ```
 * <br>
 * The CameraManagerInterface send signals when animations are started and stopped.
 *
 * The orientation cube allows the user to set some specific orientation by clicking on some
 * area of the cube (orthogonal to a face, straight to an edge, to a vertex of the cube).
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link CameraManagerInterfaceSignal}
 * @see {@link CameraControllerMode}
 * @see {@link ViewFrustum}
 * @category 3D Rendering
 */
interface CameraManagerInterface extends EventDispatcherInterface {
    /**
     * Resets the camera position to fit the entire scene.
     *
     * The current orientation of the camera and the center of the DMU are used to
     * get a virtual line. The camera changes its position to get on that line, its orientation is unchanged.
     * The camera position is set along this line to
     * maximize the drawing of the DMU, without any loss of content.
     *
     * This is actually a fitBox to the AABB of the DMU.
     *
     * @param pAnimationDurationInMilliseconds <div class="const">in</div> The duration of the animation in milliseconds. 0 means immediate movement.
     */
    resetCamera(pAnimationDurationInMilliseconds: number): void;
    /**
     * Resets the camera position to fit the entire scene.
     *
     * The current orientation of the camera and the center of the DMU are used to
     * get a virtual line. The camera changes its position to get on that line, its orientation is unchanged.
     * The camera position is set along this line to
     * maximize the drawing of the DMU, without any loss of content.
     *
     * The default animation duration is 350 ms.
     *
     * This is equivalent to :
     *
     * ```typescript
     * resetCamera(350);
     * ```
     * <br>
     * This is actually a fitBox to the AABB of the DMU.
     */
    resetCamera(): void;
    /**
     * Sets the position of the center of interest (COI), the camera will turn to look at the point without changing its location.
     *
     * This function also sets the center of interest of the CameraManagerInterface.
     * The default animation duration is 250 ms.
     *
     * @param pCenterOfInterest <div class="const">in</div> The new center of interest.
     * @param pAnimationDurationInMilliseconds <div class="const">in</div> The duration of the animation in milliseconds (default value is 250). 0 means immediate movement.
     */
    lookAt(pCenterOfInterest: Vector3, pAnimationDurationInMilliseconds: number): void;
    /**
     * Sets the position of the center of interest (COI), the camera will turn to look at the point without changing its location.
     *
     * This function also sets the center of interest of the CameraManagerInterface.
     *
     * The default animation duration is 250 ms.
     *
     * @param pCenterOfInterest <div class="const">in</div> The new center of interest.
     */
    lookAt(pCenterOfInterest: Vector3): void;
    /**
     * Changes the camera position to fit the given AABB to the screen.
     *
     * The current orientation of the camera and the center of the Box are used to
     * get a virtual line. The camera changes its position to get on that line, its orientation is unchanged.
     * The camera position is set along this line to
     * maximize the drawing of the Box, without any loss of content.
     *
     * ```typescript
     * [[include:CameraManagerInterface_fitbox.ts]]
     * ```
     * <br>
     * {@link resetCamera} uses FitBox internally.
     *
     * @param pAABB <div class="const">in</div> The box to fit on screen.
     * @param pAnimationDurationInMilliseconds <div class="const">in</div> The duration of the animation in milliseconds. 0 means immediate movement.
     */
    fitBox(pAABB: AABB, pAnimationDurationInMilliseconds: number): void;
    /**
     * Changes the camera position to fit the given AABB to the screen.
     *
     * The current orientation of the camera and the center of the Box are used to
     * get a virtual line. The camera changes its position to get on that line, its orientation is unchanged.
     * The camera position is set along this line to
     * maximize the drawing of the Box, without any loss of content.
     *
     * The default animation duration is 350 ms.
     *
     * This is equivalent to :
     *
     * ```typescript
     * fitBox(pAABB,350);
     * ```
     * <br>
     * You may use {@link fitGeometry} instead.
     *
     * @param pAABB <div class="const">in</div> The box to fit on screen.
     */
    fitBox(pAABB: AABB): void;
    /**
     * Changes the camera position to fit the given geometry to the screen.
     *
     * This is equivalent to a fixBox with the AABB of the given geometry.
     *
     * This is equivalent to :
     *
     * ```typescript
     * [[include:CameraManagerInterface_fitbox.ts]]
     * ```
     * <br>
     * Please see {@link fitBox}.
     *
     * @param pGeometryInstanceId <div class="const">in</div> The geometry instance id of the object to "move to".
     * @param pAnimationDurationInMilliseconds <div class="const">in</div> The duration of the animation in milliseconds. 0 means immediate movement.
     */
    fitGeometry(pGeometryInstanceId: number, pAnimationDurationInMilliseconds: number): void;
    /**
     * Changes the camera position to fit the given geometry to the screen.
     *
     * This is equivalent to a fixBox with the AABB of the given geometry.
     *
     * The default animation duration is 350 ms.
     *
     * This is equivalent to :
     *
     * ```typescript
     * fitGeometry(pGeometryInstanceId,350);
     * ```
     * <br>
     * Please see {@link fitBox}.
     *
     * @param pGeometryInstanceId <div class="const">in</div> The geometry instance id of the object to "move to".
     */
    fitGeometry(pGeometryInstanceId: number): void;
    /**
     * Gets the current center of interest of the camera.
     *
     * The camera rotates around this point.
     *
     * @param pOut <div class="const">out</div> The current center of interest.
     * @return `pOut`.
     */
    getCenterOfInterest(pOut: Vector3): Vector3;
    /**
     * Sets the position and direction of the camera.
     *
     * The camera orientation will be set
     * on a virtual plane composed of the given direction and the reference frame of the camera.
     *
     * @param pPosition <div class="const">in</div> The position of the camera.
     * @param pDirection <div class="const">in</div> The direction of the camera, up and right vectors will be calculated accordingly.
     * @param pAnimationDurationInMilliseconds <div class="const">in</div> The duration of the animation in milliseconds. 0 means immediate movement.
     */
    moveTo(pPosition: Vector3, pDirection: Vector3, pAnimationDurationInMilliseconds: number): void;
    /**
     * Sets the position and direction of the camera.
     *
     * The camera orientation will be set
     * on a virtual plane composed of the given direction and the reference frame of the camera.
     *
     * The default animation duration is 350 ms.
     *
     * This is equivalent to :
     *
     * ```typescript
     * moveTo(pPosition,pDirection,350);
     * ```
     * <br>
     * Up and right vectors will be recalculated.
     *
     * @param pPosition <div class="const">in</div> The position of the camera.
     * @param pDirection <div class="const">in</div> The direction of the camera, up and right vectors will be calculated accordingly.
     */
    moveTo(pPosition: Vector3, pDirection: Vector3): void;
    /**
     * Gets the camera location.
     *
     * @param pOut <div class="const">out</div> The camera location.
     * @return `pOut`.
     */
    getCameraLocation(pOut: Vector3): Vector3;
    /**
     * Sets the camera location.
     *
     * The current orientation of the camera is left unchanged.
     *
     * @param pLocation <div class="const">in</div> The location of the camera.
     */
    setCameraLocation(pLocation: Vector3): void;
    /**
     * Sets the camera location.
     *
     * The current orientation of the camera is changed according to the current mode.
     * For example, if the controller mode is {@link CameraControllerMode.CCM_ORBIT} or {@link CameraControllerMode.CCM_EXAMINE},
     * the camera frame is set according to the current Center Of Interest.
     *
     * @param pLocation <div class="const">in</div> The location of the camera.
     * @param pChangeOrientationAccordingToControllerMode <div class="const">in</div> If true, and if the controller mode has a COI,
     * then the camera orientation is set from this COI.
     */
    setCameraLocation(pLocation: Vector3, pChangeOrientationAccordingToControllerMode: boolean): void;
    /**
     * Gets the current frame of the camera.
     *
     * @param pDirVector <div class="const">out</div> The directional vector of the camera.
     * @param pUpVector <div class="const">out</div> The up vector of the camera.
     * @param pRightVector <div class="const">out</div> The right vector of the camera.
     */
    getCameraFrame(pDirVector: Vector3 | undefined, pUpVector?: Vector3 | undefined, pRightVector?: Vector3 | undefined): void;
    /**
     * Gets the frustum values of the camera.
     *
     * @param pFrustum <div class="const">out</div> The frustum values will be copied there.
     * @return `true` if pFrustum is a number array, or if pFrustum is a Float32Array of 6 values.
     * @see {@link ViewFrustum}
     */
    getFrustum(pFrustum: Float32Array | Array<number>): boolean;
    /**
     * Gets the distance from the camera to the near view plane.
     *
     * @return The distance from the camera to the near view plane.
     * @see {@link ViewFrustum}
     */
    getDMin(): number;
    /**
     * Gets the distance from the camera to the far view plane.
     *
     * @return The distance from the camera to the far view plane.
     * @see {@link ViewFrustum}
     */
    getDMax(): number;
    /**
     * Gets the "distance" to the top view plane.
     *
     * @return The "distance" to the top view plane.
     * @see {@link ViewFrustum}
     */
    getUMax(): number;
    /**
     * Gets the "distance" to the bottom view plane.
     *
     * @return The "distance" to the bottom view plane.
     * @see {@link ViewFrustum}
     */
    getUMin(): number;
    /**
     * Gets the "distance" to the right view plane.
     *
     * @return The "distance" to the right view plane.
     * @see {@link ViewFrustum}
     */
    getRMax(): number;
    /**
     * Gets the "distance" to the left view plane.
     *
     * @return The "distance" to the left view plane.
     * @see {@link ViewFrustum}
     */
    getRMin(): number;
    /**
     * Sets the camera frame (orientation) of the camera.
     *
     * Warning, the three vectors must be orthogonal between them.
     *
     * @param pDirVector <div class="const">in</div> The new camera direction.
     * @param pUpVector <div class="const">in</div> The new camera up.
     * @param pRightVector <div class="const">in</div> The new camera right.
     * @return `false` if the three vectors do not represent an orthogonal coordinate system or at least one vector is not normalized, else `true`.
     */
    setCameraFrame(pDirVector: Vector3, pUpVector: Vector3, pRightVector: Vector3): boolean;
    /**
     * Sets the navigation cube visible.
     *
     * The navigation cube is visible by default.
     *
     * The navigation cube is an html widget located on the bottom right of the 3d rendering area, which
     * helps the user knowing its current orientation. Information like FRONT, LEFT etc are displayed.
     *
     * @param pVisible <div class="const">in</div> The navigation cube is visible if <code>true</code> else it is hidden.
     */
    setNavigationCubeVisible(pVisible: boolean): void;
    /**
     * Tells if the navigation cube is visible.
     *
     * The navigation cube is visible by default.
     *
     * The navigation cube is an html widget located on the bottom right of the 3d rendering area, which
     * helps the user knowing its current orientation. Information like FRONT, LEFT etc are displayed.
     *
     * @return `true` if the navigation cube is visible.
     */
    isNavigationCubeVisible(): boolean;
    /**
     * Enables/Disables the navigation cube (i.e. it responds to user inputs).
     *
     * The navigation cube is enabled by default.
     *
     * The navigation cube is an html widget located on the bottom right of the 3d rendering area, which
     * helps the user knowing its current orientation. Information like FRONT, LEFT etc are displayed.
     *
     * @param pEnabled <div class="const">in</div> The navigation cube is enabled if <code>true</code> else it is disabled.
     */
    enableNavigationCube(pEnabled: boolean): void;
    /**
     * Tells if the navigation cube is enabled (i.e. it responds to user inputs).
     *
     * The navigation cube is enabled by default.
     *
     * The navigation cube is an html widget located on the bottom right of the 3d rendering area, which
     * helps the user knowing its current orientation. Information like FRONT, LEFT etc are displayed.
     *
     * @return `true` if the navigation cube is enabled.
     */
    isNavigationCubeEnabled(): boolean;
    /**
     * Gets the HTML representation of the navigation cube.
     *
     * You may change its look and feel.
     *
     * If no view is set, then returns undefined.
     *
     * @return The HTML representation of the navigation cube or undefined if no view is set.
     */
    getNavigationCubeHTMLElement(): HTMLElement | undefined;
    /**
     * Tells if the camera is moving at the current time.
     *
     * @return `true` if a camera animation is running at the moment.
     */
    isCameraAnimationRunning(): boolean;
    /**
     * Specifies the sensibility of the controller.
     *
     * The controller sensitivity is 1 by default.
     *
     * A value greater than 1 will increase the speed of the camera, a value lower than 1 will decrease the speed of the camera.
     * Setting a zero or negative value is invalid.
     *
     * Returns true if the sensitivity is valid and if the sensitivity was changed.
     *
     * @param pSensitivity <div class="const">in</div> the new sensitivity of the camera (1 by default).
     * @return `true` if pSensitivity is strictly superior to 1.
     */
    setControllerSensitivity(pSensitivity: number): boolean;
    /**
     * Gets the sensibility of the controller.
     *
     * The controller sensitivity is 1 by default.
     *
     * A value greater than 1 increases the speed of the camera, a value lower than 1 decreases the speed of the camera.
     * A zero or negative value is invalid.
     *
     * Returns the sensitivity of the camera controller in the range ]0,+&#x221E;[.
     *
     * @return The actual sensitivity of the camera controller.
     */
    getControllerSensitivity(): number;
    /**
     * Sets the DMU main vectors.
     *
     * This will set the orientation of the orientation cube (FRONT, TOP etc) and also sets the
     * up vector of the DMU, which affects the way the camera is managed, since the camera needs a top reference when rotating around an object.
     *
     * @param pFrontDir <div class="const">in</div> The new front vector of the scene.
     * @param pTopDir <div class="const">in</div> The new up vector of the scene.
     */
    setFrameReference(pFrontDir: Vector3, pTopDir: Vector3): void;
    /**
     * Gets the DMU main vectors.
     *
     * This will get the orientation of the orientation cube (FRONT, TOP etc).
     *
     * @param pFrontDir <div class="const">out</div> The current front vector of the scene (if set to undefined, no change).
     * @param pTopDir <div class="const">out</div> The current up vector of the scene (if set to undefined, no change).
     */
    getFrameReference(pFrontDir: Vector3 | undefined, pTopDir?: Vector3 | undefined): void;
    /**
     * Gets the current camera controller mode.
     *
     * @return The current camera controller mode.
     * @see {@link CameraControllerMode}
     */
    getCameraControllerMode(): CameraControllerMode;
    /**
     * Sets the camera controller mode to use when processing user inputs.
     *
     * @param pCameraControllerMode <div class="const">in</div> The type of controller to use.
     * @return `true` if the controller mode has changed.
     */
    setCameraControllerMode(pCameraControllerMode: CameraControllerMode): boolean;
    /**
     * Changes the camera location to zoom in the defined 2D screen region rectangle.
     *
     * @param pRegion <div class="const">in</div> The screen region to zoom in.
     * @return `true` if the controller accepts to move to the region.
     */
    zoomToScreenRegion(pRegion: Rectangle): boolean;
    /**
     * Saves all camera manager data (such as location, orientation, coi and reference frame)
     * to a json object, as a string.
     *
     * @return The json representation of the current camera parameters.
     */
    toJson(): string;
    /**
     * Restores the CameraManagerInterface state to the one when {@link toJson} was called.
     *
     * `pJson` should be the result of a {@link toJson} call, and restores the CameraManagerInterface
     * data (such as location, orientation, coi and reference frame).
     *
     * @param pJson <div class="const">in</div> The result from a previous {@link toJson} call to restore the camera parameters.
     * @return `true` if the json is parsed and if its content is correct.
     */
    fromJson(pJson: string): boolean;
    /**
     * Sets if the CameraManagerInterface should prevent the camera from rotating around the up-axis (rolling).
     *
     * `setConstraintUpAxis` is `true` as default.
     *
     * @param pConstraint <div class="const">in</div> If <ode>true</code>, the camera cannot roll.
     * @return `true` if the call changed the constraint.
     */
    setConstraintUpAxis(pConstraint: boolean): boolean;
    /**
     * Tells if the CameraManagerInterface prevents the camera from rotating around the up-axis (rolling).
     *
     * `isConstraintUpAxis` is `true` as default.
     *
     * @return `true` if the CameraManagerInterface prevents the camera from rolling.
     */
    isConstraintUpAxis(): boolean;
    /**
     * (WebXR) Gets the InfiniteXRReferenceSpace that represents the current orientation / position of the camera.
     *
     * `getXRReferenceFrame` can be called with the WebXR api to make some computations on inputs.
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_webxr.ts]]
     * ```
     * <br>
     * Please cast InfiniteXRReferenceSpace to XRReferenceSpace.
     *
     * @return The current XRReferenceSpace of the camera, or undefined if no XRSession is started (please refer to {@link InfiniteEngineInterface.setView}).
     */
    getXRReferenceFrame(): InfiniteXRReferenceSpace | undefined;
    /**
     * (WebXR) Enables/Disables the default WebXR controllers behavior.
     *
     * @param pLeftControllerEnabled <div class="const">in</div> If true, the left controller is enabled.
     * @param pRightControllerEnabled <div class="const">in</div> If true, the right controller is enabled.
     */
    setXRControllersEnabled(pLeftControllerEnabled: boolean, pRightControllerEnabled: boolean): void;
    /**
     * Gets a Json object witch contains specific settings for the specified camera mode.
     *
     * @param pCameraMode <div class="const">in</div> The camera mode to get specific settings on.
     * @return A copy of the json based settings for the given controller.
     */
    getCustomSettings(pCameraMode: CameraControllerMode): Object;
    /**
     * Sets the new custom settings for the given controller mode.
     *
     * @param pCustomSettings <div class="const">in</div> The json based settings for the given controller.
     * @param pCameraMode <div class="const">in</div> The camera mode to change.
     * @return `true` if the settings object is valid and set.
     */
    setCustomSettings(pCustomSettings: Object, pCameraMode: CameraControllerMode): boolean;
    /**
     * Requests an asynchronous wait for animation ended.
     *
     * The promise is signaled when the animation is over, i.e. when {@link isCameraAnimationRunning} returns
     * `false`.
     *
     * An example with a animation procedure :
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_asyncScreenshot.ts]]
     * ```
     * <br>
     * Please make sure the destination browser supports promises before using async calls.
     *
     * @return A promise. The promise is resolved with the reason (success, disposed).
     */
    asyncWaitForAnimationEnded(): Promise<AsyncResultReason>;
}
export { CameraManagerInterface };
/**
 * The VisualStates enum tells how the elements of the DMU are rendered.
 *
 * [[include:geometric_instance_ids_conversion.md]]
 * <br>
 * Each `geometric Instance id` has a visibility (or visual) state, telling if such an object is hidden, selected, ghosted, or set to ignore cut planes.
 * The {@link VisualStates} enumeration tells about the graphical representation of a `geometric Instance id`. The state is a set of flags (with the "|" (or) operator)
 * to allow the combination of multiple states.
 *
 * Warning : the API uses internally other flags that should not be messed with. The developer should only change flags exposed here.
 *
 * The flag manipulation is set with the methods {@link InfiniteEngineInterface.updateGeometricState} and {@link InfiniteEngineInterface.updateGeometricStateForAll} that
 * takes a mask to show the bits that will be modified:<br>
 * ex: (VisualStates.S_Hidden | VisualStates.S_Ghost) to change the hidden and ghost states at once
 * <br>and the actual bits to set:<br>
 * ex : ((~VisualStates.S_Hidden) | VisualStates.S_Ghost) to set the objects visible and ghosted.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_change_states.ts]]
 * ```
 * <br>
 * VisualStates are used with an {@link InfiniteEngineInterface}.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link InfiniteEngineInterface.updateGeometricState}
 * @see {@link InfiniteEngineInterface.updateGeometricStateForAll}
 * @category 3D Rendering
 */
declare enum VisualStates {
    /**
     * The object is hidden.
     */
    S_Hidden = 1,
    /**
     * The object is selected. If there is an InfiniteEngineInterface, then the object is rendered with an orange pulse.
     * Any object with the S_Selected "flag" is visible, even if S_Hidden is set. Such an object is NOT rendered as ghosted
     * even is S_Ghost is set.
     */
    S_Selected = 2,
    /**
     * The object is ghosted. If there is an InfiniteEngineInterface, then the object is rendered semi transparent in a light blue color.
     */
    S_Ghost = 4,
    /**
     * The object is not impacted by cut planes. Cut plane will not be applied to this object.
     */
    S_IgnoreCutPlane = 128
}
export { VisualStates };
/**
 * The LoadingStates enum tells the state of the resource loading of the objects currently rendered.
 *
 * There may be cases when the rendering of the DMU cannot be done with all the objects rendered.
 * The InfiniteEngineInterface features a complex algorithm to render the maximum relevant triangles
 * given a budget of triangles and memory.
 * This algorithm is triggered to download and render some geometries and is referred as the "resource loader".
 *
 * The LoadingStates is the status of "resource loader".
 * It tells if the "resource loader" is loading, if all geometries have been
 * loaded, or if the loading was stopped due to memory/triangles limitations.
 * This information is retrieved on the InfiniteEngineInterface {@link InfiniteEngineInterface.getResourceLoadingState}.
 *
 * This is not to be confused with the loading of the DMU itself that is retrieved on the {@link DataSessionInterface}.
 * This value is only relevant when the DMU has been loaded, and `part instances` are being rendered ({@link DataSessionInterfaceSignal.DMULoadingSuccess} signal
 * has been received).
 *
 * ```typescript
 *  [[include:InfiniteEngineInterface_LoadingStates.ts]]
 * ```
 * <br>
 * If promises are available, you may wait for the data loading to complete :
 *
 * ```typescript
 *  [[include:InfiniteEngineInterface_asyncScreenshot.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link InfiniteEngineInterface.getResourceLoadingState}
 * @category 3D Rendering
 */
declare enum LoadingStates {
    /**
     * The "resource loader" is loading resources.
     */
    S_Loading = 1,
    /**
     * The "resource loader" has loaded all required resources.
     */
    S_AllLoaded = 2,
    /**
     * The "resource loader" could not load all resources, it will not do anything more.
     */
    S_OutOfBudget = 3
}
export { LoadingStates };
/**
 * The Build is included in the attachment of the {@link ConnectionData} in the {@link DirectorySessionInterfaceSignal.LoginSuccess} signal.
 *
 * It describes the information available for a build.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link ConnectionData}
 * @category Events
 */
interface Build {
    /**
     * The name of the build.
     */
    buildcomment: string;
    /**
     * The id of the build.
     */
    buildid: string;
    /**
     * The version of the build.
     */
    buildversion: string;
    /**
     * The generation date in ISO format.
     */
    generationdate: string;
    /**
     * `true` if an evojuump file is generated for this build.
     */
    haspack: boolean;
    /**
     * `true` if the build is compatible with the web client.
     */
    hasweb: boolean;
    /**
     * The name of the owner project.
     */
    projectcomment: string;
    /**
     * The id of the project this build belongs to.
     */
    projectid: string;
    /**
     * The list of tags attached to this build.
     */
    tags: string[];
}
export { Build };
/**
 * The BuildMap is included in the attachment of the [ConnectionData] in the {@link DirectorySessionInterfaceSignal.LoginSuccess} signal.
 *
 * It is a map of build id to :
 * * a {@link Build}.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link ConnectionData}
 * @see {@link Build}
 * @category Events
 */
interface BuildMap {
    [buildid: string]: Build;
}
export { BuildMap };
/**
 * The TeamMap is included in the attachment of the [ConnectionData] in the {@link DirectorySessionInterfaceSignal.LoginSuccess} signal.
 *
 * It is a map of team name to :
 * * an internal team id.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link ConnectionData}
 * @category Events
 */
interface TeamMap {
    [teamName: string]: string;
}
export { TeamMap };
/**
 * The ProjectList is included in the attachment of the {@link ConnectionData} in the {@link DirectorySessionInterfaceSignal.LoginSuccess} signal.
 *
 * It is a map of project id to :
 * * a map of builds {@link BuildMap}.
 * * The ordered list of builds, from most recent to oldest.
 * * A project name.
 * * The project id (same value as the key to the map).
 *
 * Please refer to the [ConnectionData]].
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link ConnectionData}
 * @see {@link BuildMap}
 * @category Events
 */
interface ProjectList {
    [projectid: string]: {
        /**
         * The map of {@link Build | Builds}.
         */
        builds: BuildMap;
        /**
         * The ordered list of builds, from most recent to oldest.
         */
        orderedBuilds: Array<Build>;
        /**
         * The project name.
         */
        projectname: string;
        /**
         * The project id (same value as the key to the map).
         */
        projectid: string;
    };
}
export { ProjectList };
/**
 * The UserData is included in the attachment of the [ConnectionData] in the {@link DirectorySessionInterfaceSignal.LoginSuccess} signal.
 *
 * It contains the name of the user and possibly a URL to the user picture.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link ConnectionData}
 * @category Events
 */
interface UserData {
    /**
     * The complete name of the user.
     */
    name: string;
    /**
     * The initials of the user, John doe may have JD or J.
     */
    initials: string;
    /**
     * The Url of the user thumbnail (may be empty).
     */
    picture: string;
    /**
     * The id of the user.
     */
    id: string;
}
export { UserData };
/**
 * The ConnectionData is the attachment of the {@link DirectorySessionInterfaceSignal.LoginSuccess} signal
 * sent by the DirectorySessionInterface in case of a successful login.
 * This interface also corresponds to the attachment of the {@link DirectorySessionInterfaceSignal.AccessRightsRefreshed}
 * signal sent by the DirectorySessionInterface in case of successful refresh of the access rights.
 *
 * This object stores information about the available builds to the user, projects, computed tags and teams rights
 * attached to the user, user information and server privacy policies.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @category Events
 */
interface ConnectionData {
    /**
     * The map of available builds for the logged in user.
     */
    builds: BuildMap;
    /**
     * The available builds sorted from most recent to oldest.
     */
    orderedBuilds: Array<Build>;
    /**
     * The available builds ordered by projects.
     */
    projects: ProjectList;
    /**
     * Tells if the user should send the hardware reports to the server.
     */
    client_hardwarereport: boolean;
    /**
     * Tells if the user should send the usage reports to the server.
     */
    client_usagereport: boolean;
    /**
     * The list of tags attached to the connected user (empty).
     *
     * @deprecated The tags are always empty.
     */
    tags: string[];
    /**
     * The map of teams the user belongs to.
     */
    teams: TeamMap;
    /**
     * Some specific user data.
     */
    user: UserData;
}
export { ConnectionData };
/**
 * The Unit enum tells about the available metric unit available in the API.
 *
 * All geometric values inside the DMU are expressed as DMU units.
 *
 * The DMU Unit is available through :
 *
 * ```typescript
 * [[include:MetadataManagerInterface_unit.ts]]
 * ```
 * <br>
 * Usual units are available.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @see {@link DataSessionInterface.getDmuBuildUnit}
 * @see {@link DataSessionInterface.convertUnitFactor}
 * @category Metadata
 */
declare enum Unit {
    /**
     * Invalid unit.
     */
    U_Invalid = -1,
    /**
     * Unit in millimeters.
     */
    U_Millimeter = 0,
    /**
     * Unit in centimeters.
     */
    U_Centimeter = 1,
    /**
     * Unit in decimeters.
     */
    U_Decimeter = 2,
    /**
     * Unit in meters.
     */
    U_Meter = 3,
    /**
     * Unit in inches. An inch is 25,4 millimeters.
     */
    U_Inch = 4
}
export { Unit };
/**
 * These are the possible reasons of a promise resolution.
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @category Main
 */
declare enum AsyncResultReason {
    /**
     * Result is a success, and the result should not be undefined.
     */
    ARR_Success = 0,
    /**
     * The call was cancelled due to a bad input.
     */
    ARR_BadInput = 1,
    /**
     * The call was cancelled due to an explicit call to cancel.
     */
    ARR_Cancelled = 2,
    /**
     * The call was cancelled because of the object dispose function was called.
     */
    ARR_Disposed = 3
}
export { AsyncResultReason };
/**
 * Represents a promise result, for functions expecting to return a UInt32Array.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @category Metadata
 */
interface AsyncUInt32ArrayResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     */
    value?: Uint32Array;
}
export { AsyncUInt32ArrayResult };
/**
 * The ConfContextInterfaceSignal lists all the signals that may be sent by the {@link ConfContextInterface}.
 *
 * These signals are emitted by the {@link ConfContextInterface} when some data is available.
 *
 * * ConfContextReady : the {@link ConfContextInterface} is ready, i.e. when it has
 * finished processing its `geometric Instance ids`.
 * * ConfContextCancelled : the update (running) of the {@link ConfContextInterface} has been cancelled.
 * * ConfContextChanged : the configurations of the {@link ConfContextInterface} has been changed.
 *
 * ```typescript
 * [[include:ConfContextInterface_signal.ts]]
 * ```
 * <br>
 * Please refer to the {@link ConfContextInterface} to know how to handle the ConfContextInterfaceSignal.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link ConfContextInterface}
 * @category Events
 */
declare enum ConfContextInterfaceSignal {
    /**
     * Signal sent by the {@link ConfContextInterface} when the {@link ConfContextInterface} is ready, i.e. {@link ConfContextInterface.getGeometricInstanceIds} can be called.
     *
     * Warning, the ConfContextReady signal is sent when {@link DataSessionInterface.cancelUpdate} is called and the {@link ConfContextInterface} is updating.
     *
     * No attachment for events of this type.
     */
    ConfContextReady = "ready",
    /**
     * Signal sent by the {@link ConfContextInterface} when the {@link ConfContextInterface} is requested to be cancelled.
     *
     * Such an event is fired if the {@link ConfContextInterface} was running and {@link DataSessionInterface.cancelUpdate} is called.
     * The {@link ConfContextReady} signal will be triggered shortly after.
     *
     * No attachment for events of this type.
     */
    ConfContextCancelled = "cancelled",
    /**
     * Signal sent by the {@link ConfContextInterface} when the {@link ConfContextInterface} is changed i.e. {@link ConfContextInterface.setActiveConfs}
     * has been called.
     *
     * The {@link ConfContextReady} signal will be triggered after.
     *
     * No attachment for events of this type.
     */
    ConfContextChanged = "changed"
}
export { ConfContextInterfaceSignal };
/**
 * The ConfContextInterface interface is used to get the available `part instances` inside the "Configurations" of
 * a DMU. It may be used to filter out a group of `part instances` depending on their presence in these "Configurations",
 * or to display or hide them.
 *
 * The ConfContextInterface interfaces are created through the {@link DataSessionInterface.createConfContext} method, and
 * there may be any number of different configurations inside the ConfContextInterface.
 *
 * The list of available configurations is accessible through the {@link DataSessionInterface.getConfigurationList} method.
 *
 * ```typescript
 * [[include:ConfContextInterface_setActiveConfs.ts]]
 * ```
 * <br>
 * The user may include any number of {@link ConfigurationInterface} inside the ConfContextInterface (by their id), the resulting set will be the union of all
 * the requested {@link ConfigurationInterface} : any `part instance` included in at least one {@link ConfigurationInterface} will be available through the resulting
 * ConfContextInterface.
 *
 * The ConfContextInterface has two major uses :
 * * if included in a {@link VisibilityContextInterface}, the objects used with the given {@link VisibilityContextInterface} will be limited to the union of the
 * {@link ConfigurationInterface} inside the given ConfContextInterface. Any `part instance` not inside the chosen {@link ConfigurationInterface} will be filtered out.
 * * If used as is, it may be used to compute the visibility of the DMU.
 *
 * For example, the following code hides all the `part instances` except the ones in the ConfContextInterface :
 *
 * ```typescript
 * [[include:ConfContextInterface_getGeometricInstanceIds.ts]]
 * ```
 * <br>
 * The list of signals the ConfContextInterface may trigger is available in the {@link ConfContextInterfaceSignal} enumeration.
 *
 * The active configurations are set through the {@link setActiveConfs} method. Each time the setActiveConfs method is called, the
 * {@link ConfContextInterfaceSignal.ConfContextChanged} signal is sent.
 * The {@link setActiveConfs} does not trigger a computation on the infinite servers, the user has to call the {@link DataSessionInterface.update}
 * method to trigger a computation (after the ConfContextInterface has been changed). The result is not available right away, but the
 * event {@link ConfContextInterfaceSignal.ConfContextReady} is triggered when the result of the ConfContextInterface is available.
 *
 * Warning : Creating a ConfContextInterface by {@link DataSessionInterface.createConfContext} will fire a {@link ConfContextInterfaceSignal.ConfContextReady}
 * with all the `part instances` of the DMU, this is equivalent to calling :
 *
 * ```typescript
 * [[include:ConfContextInterface_setActiveConfs.ts]]
 * ```
 * <br>
 * Or with async calls :
 *
 * ```typescript
 * [[include:ConfContextInterface_asyncSetActiveConfs.ts]]
 * ```
 * <br>
 * Warning : there may be cases when the {@link getGeometricInstanceIds} is not available such as when the ConfContextInterface is updating, i.e. when
 * {@link isRunning} returns true, or when the ConfContextInterface has been cancelled, i.e. when {@link isCancelled} returns true.
 *
 * A ConfContextInterface may be interrupted (cancelled) when the ConfContextInterface is running and {@link DataSessionInterface.cancelUpdate}
 * is called. In such cases, the {@link ConfContextInterfaceSignal.ConfContextCancelled} signal is fired, and shortly after, {@link ConfContextInterfaceSignal.ConfContextReady}
 * signal is fired, but {@link getGeometricInstanceIds} will return undefined. Just call {@link DataSessionInterface.update} and the computation of
 * the ConfContextInterface will resume.
 *
 * If you call multiple times {@link setActiveConfs} before receiving the {@link ConfContextInterfaceSignal.ConfContextReady}, only one {@link ConfContextInterfaceSignal.ConfContextReady}
 * will be sent with the content of the last call to {@link setActiveConfs}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @see {@link DataSessionInterface}
 * @see {@link ConfContextInterfaceSignal}
 * @category Metadata
 */
interface ConfContextInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Tells if the ConfContextInterface is updating (running).
     *
     * This is generally the case after calling {@link setActiveConfs}, or right after the ConfContextInterface creation
     * with {@link DataSessionInterface.createConfContext}.
     *
     * @return `true` if the ConfContextInterface is updating.
     */
    isRunning(): boolean;
    /**
     * Sets the identifier of the ConfContextInterface.
     * Make sure the id is unique.
     * A unique ConfContextInterface identifier is created if the identifier is not overridden.
     *
     * @param pConfCtxId <div class="const">in</div> The new identifier of the ConfContextInterface.
     */
    setConfContextId(pConfCtxId: string): void;
    /**
     * Gets the identifier of the ConfContextInterface.
     *
     * @return The identifier of the ConfContextInterface.
     */
    getConfContextId(): string;
    /**
     * Gets the last error returned by the update of the ConfContextInterface.
     *
     * @return The last error message (if any, or an empty string if no error occurred).
     */
    getLastError(): string;
    /**
     * Sets the ids of the active configurations that are included in this ConfContextInterface.
     *
     * An event {@link ConfContextInterfaceSignal.ConfContextChanged} signal is sent.
     *
     * The consecutive call to {@link DataSessionInterface.update} will trigger the
     * calculation of the `part instances` inside this union of {@link ConfigurationInterface.getConfigurationId}.
     *
     * If pActiveConfigurationIds is an empty array (the default case), the given ConfContextInterface is "unconfigured", i.e. it contains
     * all the `part instances` of the DMU.
     *
     * Only unique and valid configuration ids from pActiveConfigurationIds will be activated.
     * If pActiveConfigurationIds contains duplicates or invalid configuration ids they will be skipped which might results as activating an empty array (the default case).
     *
     * @param pActiveConfigurationIds <div class="const">in</div> The list of {@link ConfigurationInterface} ids to set.
     * @return `true` if the ConfContextInterface is changed and {@link DataSessionInterface.update} must be called.
     * @see {@link ConfigurationInterface.getConfigurationId}
     */
    setActiveConfs(pActiveConfigurationIds: Array<string>): boolean;
    /**
     * Gets the list of active {@link ConfigurationInterface} ids, as set previously by {@link setActiveConfs}.
     *
     * Modifying this array in place results in undefined behavior.
     *
     * DO NOT modify this array.
     *
     * @return <div class="const">const</div> The list of active {@link ConfigurationInterface} ids in this ConfContextInterface.
     * @see {@link ConfigurationInterface.getConfigurationId}
     */
    getActiveConfs(): ReadonlyArray<string>;
    /**
     * Gets the list of all the `geometric Instance ids` of the `part instances` represented by the union of these {@link ConfigurationInterface}.
     *
     * The list of `geometric Instance ids` is returned if the ConfContextInterface has finished computing.
     * Use {@link addEventListener} on the event {@link ConfContextInterfaceSignal.ConfContextReady} to
     * know when the ConfContextInterface is ready.
     *
     * DO NOT modify this array in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of <code>geometric instances ids</code> represented by this ConfContextInterface, or
     * undefined if the ConfContextInterface is computing or if the ConfContextInterface is in error or cancelled.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     */
    getGeometricInstanceIds(): Uint32Array | undefined;
    /**
     * Gets the list of all the `geometric Instance ids` of the `part instances` represented by the union of these {@link ConfigurationInterface}.
     *
     * The list of `geometric Instance ids` is returned once the ConfContextInterface has finished computing.
     * If it's been cancelled, or disposed, the returned promise contains this information.
     *
     * If pRunUpdate is true (by default), a {@link DataSessionInterface.update} is automatically called.
     *
     * DO NOT modify this array in place, this results in undefined behavior.
     *
     * @param pRunUpdate If set to true, a {@link DataSessionInterface.update} is called (defaults to true).
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input), or rejected in case of an
     * unexpected error. In case of success, the promise is resolved with the list of <code>geometric instances ids</code> represented by this ConfContextInterface.
     */
    asyncGetGeometricInstanceIds(pRunUpdate?: boolean): Promise<AsyncUInt32ArrayResult>;
    /**
     * Tells if the ConfContextInterface has been cancelled.
     *
     * This is generally the case after calling {@link DataSessionInterface.cancelUpdate} when the ConfContextInterface
     * is updating (ie .isRunning() is true).
     *
     * @return `true` If the ConfContextInterface is cancelled.
     * @see {@link DataSessionInterface.cancelUpdate}
     */
    isCancelled(): boolean;
}
export { ConfContextInterface };
/**
 * The {@link EffectivityItemInterface} is a requirement used for the configuration system.
 *
 * The configuration system is explained in the <a href="../index.html#configurations">configurations</a>.
 *
 * The {@link EffectivityItemInterface} is included in the {@link EffectivitiesInterface}, and tells a requirement that must
 * or must not be satisfied in order for a `part instance` to be included in a configuration.
 *
 * The {@link EffectivityItemInterface} tells the attribute name and the accepted/rejected
 * values that a document must feature in order to be accepted/rejected.
 *
 * ```typescript
 * [[include:ConfContextInterface_setActiveConfs.ts]]
 * ```
 * <br>
 * Please refer to the <a href="../index.html#configurations">Configuration System</a>.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @see {@link EffectivitiesInterface}
 * @see {@link ConfigurationInterface}
 * @category Metadata
 */
interface EffectivityItemInterface {
    /**
     * Gets the attribute name of the requirement.
     *
     * @return The attribute name of the requirement.
     */
    getKey(): string;
    /**
     * Gets the accepted/rejected values of the requirement.
     *
     * Such a list is sorted by values.
     *
     * DO NOT modify the returned string [].
     *
     * @return <div class="const">const</div> The accepted/rejected values of the requirement.
     */
    getSortedEffectivities(): Array<string>;
}
export { EffectivityItemInterface };
/**
 * The {@link EffectivitiesInterface} is a list of requirements used for the configuration system.
 *
 * The configuration system is explained in the <a href="../index.html#configurations">configurations</a>.
 *
 * The {@link EffectivitiesInterface} is included in the {@link ConfigurationInterface}, and tells the requirements that must
 * or must not be satisfied in order for a `part instance` to be included in a configuration.
 *
 * Such a requirement is expressed as a {@link EffectivityItemInterface} that tells the attribute name and the accepted/rejected
 * values that a document must feature in order to be accepted/rejected.
 *
 * ```typescript
 * [[include:ConfContextInterface_setActiveConfs.ts]]
 * ```
 * <br>
 * Please refer to the <a href="../index.html#configurations">Configuration System</a>.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @see {@link EffectivityItemInterface}
 * @see {@link ConfigurationInterface}
 * @category Metadata
 */
interface EffectivitiesInterface {
    /**
     * Gets the list of effectivities that must be satisfied/dissatisfied.
     *
     * Such a list is sorted by {@link EffectivityItemInterface.getKey}.
     *
     * DO NOT modify the returned {@link EffectivityItemInterface} [].
     *
     * @return <div class="const">const</div> the sorted list of requirements to satisfy/dissatisfy for the given configuration.
     */
    getSortedEffectivities(): Array<EffectivityItemInterface>;
}
export { EffectivitiesInterface };
/**
 * The {@link ConfigurationInterface} is a configuration.
 *
 * The DMU may feature a list of configurations. A configuration is created by
 * the project maintainer and made available to all users. In order to ease the possible dynamic creation
 * of configurations in the future versions of the API, the {@link ConfigurationInterface} object shows the same attributes (some are still stripped) as the configuration
 * object created by the project maintainer.
 *
 * The configuration system is explained in the <a href="../index.html#configurations">configurations</a>.
 *
 * A configuration elects always the root `part instance`. Then each child instance is included in the configuration if :
 * * it has no link documents attached to the parent/child relationship
 * * or the link document satisfies the requirements in the [[getNeededEffectivities] **AND** does not satisfy the requirements in {@link getExcludedEffectivities}.
 *
 * This algorithm is then recursively applied to all `part instances` to select a subset of all the `part instances`.
 *
 * ```typescript
 * [[include:ConfContextInterface_setActiveConfs.ts]]
 * ```
 * <br>
 * Please refer to the <a href="../index.html#configurations">Configuration System</a>.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @category Metadata
 */
interface ConfigurationInterface {
    /**
     * Gets the id of the configuration.
     *
     * The configuration id is used by the {@link ConfContextInterface} to set configurations (see {@link ConfContextInterface.setActiveConfs}).
     *
     * @return The configuration id of the configuration.
     */
    getConfigurationId(): string;
    /**
     * Gets the "GUI" name of the configuration (as created by the project maintainer).
     *
     * @return The name of the configuration.
     */
    getName(): string;
    /**
     * Gets the description of the configuration.
     *
     * The description is a short text explaining the configuration.
     *
     * @return The description for the configuration.
     */
    getDescription(): string;
    /**
     * Gets the requirements a link document must satisfy in order to be included in the configuration.
     *
     * Each parent instance / child instance relationship decorated with a document that satisfies the getNeededEffectivities
     * will imply the child will be included in the configuration if the parent is itself inside the given configuration.
     *
     * DO NOT modify the returned {@link EffectivitiesInterface}.
     *
     * @return <div class="const">const</div> The list of requirements to satisfy for the given configuration.
     */
    getNeededEffectivities(): EffectivitiesInterface;
    /**
     * Gets the requirements a link document must **NOT** satisfy in order to be included in the configuration.
     *
     * Each parent instance / child instance relationship decorated with a document that does not satisfy the getExcludedEffectivities
     * will imply the child will be included in the configuration if the parent is itself inside the given configuration.
     *
     * DO NOT modify the returned {@link EffectivitiesInterface}.
     *
     * @return <div class="const">const</div> The list of requirements to dissatisfy for the given configuration.
     */
    getExcludedEffectivities(): EffectivitiesInterface;
}
export { ConfigurationInterface };
/**
 * The AttachedDocumentInfoInterface interface is used to store properties about any attached document linked to a `part instance`.
 *
 * The AttachedDocumentInfoInterface is retrieved when using a {@link PartInstanceInfoInterface.getAttachedDocumentInfos} i.e. when the metadata
 * attached to a `part instance` are retrieved. The actual data (binary or not) will be retrieved by using the {@link DocumentContentGetterInterface}.
 *
 * NB : Any attached document with a URL will NOT be retrieved, as the developer should know how to handle URLs.
 *
 * The following code snippet allows a browser user to "download" the content of an AttachedDocumentInfoInterface by the {@link DocumentContentGetterInterface}.
 * Such a code may be triggered while receiving the {@link DocumentContentGetterInterfaceSignal.DocumentContentReady} of the {@link DocumentContentGetterInterface}.
 *
 * ```typescript
 * [[include:DocumentContentGetterInterface_download.ts]]
 * ```
 * <br>
 * Please refer to the {@link DocumentContentGetterInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link IdCardGetterInterface}
 * @see {@link DocumentContentGetterInterface}
 * @see {@link DocumentContentGetterInterfaceSignal}
 * @category Data Retrievers
 */
interface AttachedDocumentInfoInterface {
    /**
     * The id of the attached document.
     */
    mDocumentId: string;
    /**
     * The name of the attached document.
     */
    mDocumentName: string;
    /**
     * The description relative to the attached attached document content.
     */
    mDocumentDescription: string;
    /**
     * The mime type of the attached document.
     *
     * Available types are (not comprehensive):
     * * application/json.
     * * application/xml.
     * * application/x-qt-richtext.
     * * image/jpg.
     * * image/jpeg.
     * * image/png.
     * * text/csv.
     * * text/html.
     * * text/plain.
     * * text/xml.
     *
     * Depending on your DMU provider, more or less types may be available.
     *
     * @see {@link https://en.wikipedia.org/wiki/Media_type}
     */
    mDocumentMimeType: string | undefined;
    /**
     * The url of the attached document. If empty, then the attached document has no URL.
     *
     * Any attached document with a URL will NOT be retrieved.
     */
    mDocumentUrl: string;
}
export { AttachedDocumentInfoInterface };
/**
 * The DocumentContentGetterInterfaceSignal lists all the signals that may be sent by the {@link DocumentContentGetterInterface}.
 *
 * These signals are emitted by the {@link DocumentContentGetterInterface} when some data is available.
 *
 * * DocumentContentReady : the {@link DocumentContentGetterInterface} is ready, i.e. when it has
 * finished downloading the content of an attached document.
 * * DocumentContentCancelled : the retrieving of the content by the {@link DocumentContentGetterInterface} has been cancelled.
 *
 * ```typescript
 * [[include:DocumentContentGetterInterface_retrieveDocumentContent.ts]]
 * ```
 * <br>
 * Please refer to the {@link DocumentContentGetterInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link DocumentContentGetterInterface}
 * @see {@link IdCardGetterInterface}
 * @category Events
 */
declare enum DocumentContentGetterInterfaceSignal {
    /**
     * Signal sent by the {@link DocumentContentGetterInterface} when the {@link DocumentContentGetterInterface} is ready, i.e. {@link DocumentContentGetterInterface.getDocumentContent}
     * can be called.
     *
     * Warning, the {@link DocumentContentReady} signal is sent when {@link DocumentContentGetterInterface.cancel} is called and the {@link DocumentContentGetterInterface} is updating.
     *
     * The attachment is a string telling the request id of the call to {@link DocumentContentGetterInterface.retrieveDocumentContent} ({@link DocumentContentGetterInterface.getLastRequestId}).
     */
    DocumentContentReady = "ready",
    /**
     * Signal sent by the {@link DocumentContentGetterInterface} when the {@link DocumentContentGetterInterface} has been cancelled.
     *
     * Such an event is fired if the {@link DocumentContentGetterInterface} was updating and {@link DocumentContentGetterInterface.cancel} is called.
     * The {@link DocumentContentReady} signal will be triggered shortly after.
     *
     * No attachment for events of this type.
     */
    DocumentContentCancelled = "cancelled"
}
export { DocumentContentGetterInterfaceSignal };
/**
 * Represents an document content request promise result.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link DocumentContentGetterInterface.asyncRetrieveDocumentContent}
 * @category Data Retrievers
 */
interface AsyncDocumentContentGetterResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     */
    value?: string | Uint8Array;
}
export { AsyncDocumentContentGetterResult };
/**
 * The DocumentContentGetterInterface interface is used to get the content of an attached document knowing its properties (this cannot be used to get metadata documents).
 *
 * The attached document content can be retrieved by an AttachedDocumentInfoInterface obtained by a call to {@link PartInstanceInfoInterface.getAttachedDocumentInfos}.
 * If the user knows the `attached document id` and its mime type, then the attached document can also be retrieved using this information and using
 * {@link DataSessionInterface.createAttachedDocumentInfo}.
 * `attached document ids` are string based.
 *
 * Warning : any attached document with a URL will NOT be retrieved.
 *
 * The DocumentContentGetterInterface interfaces are created through the {@link DataSessionInterface.createDocumentContentGetter} method.
 *
 * ```typescript
 * [[include:DocumentContentGetterInterface_retrieveDocumentContent.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:DocumentContentGetterInterface_asyncRetrieveDocumentContent.ts]]
 * ```
 * <br>
 * The list of signals the DocumentContentGetterInterface may trigger is available in the {@link DocumentContentGetterInterfaceSignal} enumeration.
 *
 * The attached document retrieval procedure is triggered through the {@link retrieveDocumentContent} methods. The result is not available right away, but the
 * event {@link DocumentContentGetterInterfaceSignal.DocumentContentReady} is triggered when the result of the DocumentContentGetterInterface is available.
 * The result is available through the {@link getDocumentContent} function.
 *
 * Warning : there may be cases when the {@link getDocumentContent} is not available such as when the DocumentContentGetterInterface is updating, i.e. when
 * {@link isRunning} returns true, or when the DocumentContentGetterInterface has been cancelled, i.e. when {@link isCancelled} returns true.
 *
 * A DocumentContentGetterInterface may be interrupted (cancelled) when the DocumentContentGetterInterface is updating and {@link cancel}
 * is called. In such cases, the {@link DocumentContentGetterInterfaceSignal.DocumentContentCancelled} signal is fired, and shortly after, {@link DocumentContentGetterInterfaceSignal.DocumentContentReady}
 * signal is fired, but {@link getDocumentContent} will return undefined. Just call {@link retrieveDocumentContent} with another (or the same) attached document properties to trigger a new
 * retrieval.
 *
 * If you call multiple times {@link retrieveDocumentContent} before receiving the {@link DocumentContentGetterInterfaceSignal.DocumentContentReady}, only one {@link DocumentContentGetterInterfaceSignal.DocumentContentReady}
 * will be sent with the content of the last call to {@link retrieveDocumentContent}.
 *
 * The result may be undefined ({@link getDocumentContent}) if no such document was found, this is not considered as an error.
 *
 * The developer may trigger a download on the user browser by the following code :
 *
 * ```typescript
 * [[include:DocumentContentGetterInterface_download.ts]]
 * ```
 * <br>
 * Please refer to the {@link AttachedDocumentInfoInterface} for additional information.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link DocumentContentGetterInterfaceSignal}
 * @see {@link AttachedDocumentInfoInterface}
 * @category Data Retrievers
 */
interface DocumentContentGetterInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Gets the content of the specified attached document.
     *
     * The attached document must not have a URL, any AttachedDocumentInfoInterface with a non empty mDocumentURL will return false.
     *
     * The mime type sets the return type of getDocumentContent, string if the attached document is a text one, Uint8Array if binary.
     *
     * The AttachedDocumentInfoInterface is retrieved when using a {@link PartInstanceInfoInterface.getAttachedDocumentInfos}.<br>
     * The event DocumentContentGetterInterfaceSignal.DocumentContentReady is fired when the
     * attached document content result is ready.
     *
     * Returns true if the attached document procedure retrieval is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed.
     *
     * The result may be undefined ({@link getDocumentContent}) if no such document was found.
     *
     * @param pAttachedDocumentInfo <div class="const">in</div> The information about the attached document to get.
     * @return `true` if the retrieval procedure has begun.
     * @see {@link DocumentContentGetterInterfaceSignal}
     */
    retrieveDocumentContent(pAttachedDocumentInfo: AttachedDocumentInfoInterface): boolean;
    /**
     * Asynchronously gets the content of the specified attached document.
     *
     * The attached document must not have a URL, any AttachedDocumentInfoInterface with a non empty mDocumentURL will return a promise
     * with a bad input result.
     *
     * The mime type sets the return type of the promise content, string if the attached document is a text one, Uint8Array if binary.
     *
     * The AttachedDocumentInfoInterface is retrieved when using a {@link PartInstanceInfoInterface.getAttachedDocumentInfos}.
     *
     * Returns a promise.
     *
     * The result may be undefined ({@link getDocumentContent}, {@link AsyncDocumentContentGetterResult.value}) if no such document was found,
     * this is not an error, and {@link AsyncDocumentContentGetterResult.reason} is equal to {@link AsyncResultReason.ARR_Success}.
     *
     * Please note that in case of multiple promises running at the same time on the same DocumentContentGetterInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getDocumentContent}
     * after awaiting will return the same value.
     *
     * @param pAttachedDocumentInfo <div class="const">in</div> The information about the attached document to get.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the document content.
     */
    asyncRetrieveDocumentContent(pAttachedDocumentInfo: AttachedDocumentInfoInterface): Promise<AsyncDocumentContentGetterResult>;
    /**
     * Gets the content of the specified attached document by its id and mime type.
     *
     * The mime type sets the return type of getDocumentContent, string if the attached document is a text one, Uint8Array if binary.
     *
     * The event DocumentContentGetterInterfaceSignal.DocumentContentReady is fired when the
     * attached document content result is ready.
     *
     * Returns true if the document procedure retrieval is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed.
     *
     * The result may be undefined ({@link getDocumentContent}) if no such document was found.
     *
     * @param pDocumentId <div class="const">in</div> The id of the attached document to get.
     * @param pMimeType <div class="const">in</div> The optional mime type of the attached document to get.
     * @return `true` if the retrieval procedure has begun.
     * @see {@link DocumentContentGetterInterfaceSignal}
     */
    retrieveDocumentContent(pDocumentId: string, pMimeType?: string | undefined): boolean;
    /**
     * Asynchronously gets the content of the specified attached document by its id and mime type.
     *
     * The mime type sets the return type of the promise, string if the attached document is a text one, Uint8Array if binary.
     *
     * Returns a promise.
     *
     * The result may be undefined ({@link getDocumentContent}, {@link AsyncDocumentContentGetterResult.value}) if no such document was found,
     * this is not an error, and {@link AsyncDocumentContentGetterResult.reason} is equal to {@link AsyncResultReason.ARR_Success}.
     *
     * Please note that in case of multiple promises running at the same time on the same DocumentContentGetterInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getDocumentContent}
     * after awaiting will return the same value.
     *
     * @param pDocumentId <div class="const">in</div> The id of the attached document to get.
     * @param pMimeType <div class="const">in</div> The optional mime type of the attached document to get.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the document content.
     */
    asyncRetrieveDocumentContent(pDocumentId: string, pMimeType?: string | undefined): Promise<AsyncDocumentContentGetterResult>;
    /**
     * Each call to {@link retrieveDocumentContent} is assigned a request id.
     *
     * This call tels the id of the last call to {@link retrieveDocumentContent}.
     *
     * @return The id of the last call to {@link retrieveDocumentContent}.
     */
    getLastRequestId(): string;
    /**
     * Cancels the computation of the attached document retrieving process (if any).
     *
     * A {@link DocumentContentGetterInterfaceSignal.DocumentContentCancelled} signal is emitted if the
     * DocumentContentGetterInterface is retrieving data.
     *
     * @return `true` if the document retrieval procedure was running.
     * @see {@link DocumentContentGetterInterfaceSignal.DocumentContentCancelled}
     */
    cancel(): boolean;
    /**
     * Tells if the DocumentContentGetterInterface is updating.
     *
     * This is the case after calling {@link retrieveDocumentContent}.
     *
     * @return `true` if the DocumentContentGetterInterface is updating.
     */
    isRunning(): boolean;
    /**
     * Tells if the DocumentContentGetterInterface has been cancelled.
     *
     * This is generally the case after calling {@link cancel} when the DocumentContentGetterInterface
     * is retrieving data.
     *
     * @return `true` if the DocumentContentGetterInterface is cancelled.
     */
    isCancelled(): boolean;
    /**
     * Gets the last error returned by the update of the DocumentContentGetterInterface.
     *
     * @return The last error message (if any, or an empty string if no error occurred).
     */
    getLastError(): string;
    /**
     * Gets the content of the requested attached document.
     *
     * The attached document content is returned if the DocumentContentGetterInterface has finished computing.
     * Use {@link addEventListener} on the event {@link DocumentContentGetterInterfaceSignal.DocumentContentReady} to
     * know when the DocumentContentGetterInterface is ready.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * The type of the data returned by the function depends on the mime type of the attached document.
     *
     * @return <div class="const">const</div> The data of the attached document, in string or binary data given its mime-type, or
     * undefined if the DocumentContentGetterInterface is computing or if the DocumentContentGetterInterface is in error or cancelled.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     */
    getDocumentContent(): string | Uint8Array | undefined;
    /**
     * Gets the attached document information that was requested by a previous call to {@link retrieveDocumentContent}.
     *
     * If {@link retrieveDocumentContent} was called with a `attached document id`, a "dummy" {@link AttachedDocumentInfoInterface} is created with the document id
     * and mime-type, and with empty other properties.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> the {@link AttachedDocumentInfoInterface} that was requested.
     * @see {@link retrieveDocumentContent}
     */
    getAttachedDocumentInfo(): AttachedDocumentInfoInterface | undefined;
}
export { DocumentContentGetterInterface };
/**
 * The CutPlaneManagerInterfaceSignal lists all the signals that may be sent by the {@link CutPlaneManagerInterface}.
 *
 * These signals are emitted by the CutPlaneManagerInterface when a cut plane is modified.
 * Two types of signals are emitted :
 *
 * * CutPlaneGeometryChanged when the cut plane geometry (normal and/or location) is changed.
 * * CutPlaneEnabledChanged when a cut plane is enabled/disabled.
 *
 * ```typescript
 * [[include:CutPlaneManagerInterface_signal.ts]]
 * ```
 * <br>
 * Please refer to the {@link CutPlaneManagerInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link CutPlaneManagerInterface}
 * @category Events
 */
declare enum CutPlaneManagerInterfaceSignal {
    /**
     * Signal sent by the CutPlaneManagerInterface when the cut plane geometry (normal and/or location) is changed.
     *
     * The attachment for events of this type is a number, the id of the modified cut plane.
     */
    CutPlaneGeometryChanged = "cutPlaneGeometryChanged",
    /**
     * Signal sent by the CutPlaneManagerInterface when a cut plane is enabled/disabled.
     *
     * The attachment for events of this type is a number, the id of the modified cut plane.
     */
    CutPlaneEnabledChanged = "cutPlaneEnabledChanged"
}
export { CutPlaneManagerInterfaceSignal };
/**
 * The CutPlaneManagerInterface interface is used to create cut planes, that are used to hide half of the scene.
 *
 * The CutPlaneManagerInterface is included in the {@link InfiniteEngineInterface} and accessed through {@link InfiniteEngineInterface.getCutPlaneManager}.
 *
 * There are no javascript cut plane objects, cut planes are modified through the {@link CutPlaneManagerInterface} by their ids. Indeed, each cut plane is assigned an integer id upon its creation,
 * 0 is an invalid id, and is the expected result if the cut plan creation fails. Ids are recycled upon cut planes deletion, so there may be cases when a new cut plane is assigned an id
 * that belonged to a deleted cut plane.
 *
 * There is a limited number of cut planes that may be created by an application (shader limitations) : only 16 different cut planes may be created / manipulated.
 *
 * A cut plane is defined by a position `P` that belong to the plane and a normal `N : (Nx,Ny,Nz)`. This defines a plane (hyperplane) in 3D dimension. Such a plane
 * has the equation : &#123;x,y,z&#125; / Nx.x + Ny.y + Nz.z = dot(N,P). Each vertex on the side of the normal `N` will be discarded, so every vertex V : &#123;Vx,Vy,Vz&#125; /
 * Nx.Vx + Ny.Vy + Nz.Vz > dot(N,P) will be discarded.
 *
 * A CutPlaneManipulatorInterface may be created and bound to a cut plane to display a GUI element that allows the end user to change the cut plane orientation / position through
 * {@link CutPlaneManagerInterface.createCutPlaneManipulator}.
 *
 * When a cut plane is modified or disabled/enabled, {@link CutPlaneManagerInterfaceSignal} signals are sent.
 *
 * Suppose that we want to hide all items whose "altitude" (z) is superior to 10.
 *
 * ```typescript
 * [[include:CutPlaneManagerInterface_set_cut_plane.ts]]
 * ```
 * <br>
 * Please refer to the {@link CutPlaneManipulatorInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link CutPlaneManagerInterfaceSignal}
 * @see {@link CutPlaneManipulatorInterface}
 * @see {@link InfiniteEngineInterface.getCutPlaneManager}
 * @category 3D Rendering
 */
interface CutPlaneManagerInterface extends EventDispatcherInterface {
    /**
     * Creates a new plane given a point position and a normal vector.
     *
     * The normal does not
     * need to be unit length, but a zero vector is invalid.
     * Each vertex on the side of the normal `pNormal` will be discarded.
     *
     * Returns the id (non zero) of the newly created cut plane, or 0 if the maximum number of
     * cut planes have already been reached or if pNormal is equal to (0,0,0).
     *
     * The newly created cut plane is enabled by default.
     *
     * @param pPoint <div class="const">in</div> The 3D position of a point that belongs to the plane.
     * @param pNormal <div class="const">in</div> The 3D normal vector of the plane.
     * @return The id of the created plane. If the maximum number of available cut planes slots has been reached or if pNormal is equal to (0,0,0), returns 0.
     */
    createCutPlane(pPoint: Vector3, pNormal: Vector3): number;
    /**
     * Instantiates a cut plane manipulator.
     *
     * The CutPlaneManipulatorInterface is a GUI widget that allows a user to change the position/orientation of the cut plane.
     *
     * @return A cut plane manipulator object.
     * @see {@link CutPlaneManipulatorInterface}
     */
    createCutPlaneManipulator(): CutPlaneManipulatorInterface;
    /**
     * Gets the actual number of cut planes.
     *
     * @return The number of created cut planes.
     */
    getNbCutPlanes(): number;
    /**
     * Gets the maximum number of cut planes that can be created.
     *
     * This number is constant, it returns 16 at the moment.
     *
     * @return The maximum number of planes that can be created.
     */
    getNbMaxCutPlanes(): number;
    /**
     * Gets the actual cut plane ids that have been created.
     *
     * @return <div class="const">const</div> The ids of the cut planes in use.
     */
    getCutPlanesIds(): Array<number>;
    /**
     * Removes the cut plane with id pCutPlaneId.
     *
     * @param pCutPlaneId <div class="const">in</div> The id of the cut plane to remove.
     * @return `true` if the cut plane exists and therefore has been removed.
     */
    removeCutPlane(pCutPlaneId: number): boolean;
    /**
     * Removes all previously created cut planes.
     */
    removeAllCutPlanes(): void;
    /**
     * Gets the cut plane equation of the given plane.
     *
     * Be N : (Nx,Ny, Nz) the normal of the cut plane, Be P a point of the plane,
     * we get the equation : &#123;x,y,z&#125; / Nx&#46;x + Ny&#46;y + Nz&#46;z = dot(N,P).
     * `pCutPlaneEquation` will be set such as <br>
     * `pCutPlaneEquation`&#46;x = Nx<br>
     * `pCutPlaneEquation`&#46;y = Ny<br>
     * `pCutPlaneEquation`&#46;z = Nz<br>
     * `pCutPlaneEquation`&#46;w = dot(N,P).
     *
     * Returns false if the cut plane was not found.
     *
     * @param pCutPlaneId <div class="const">in</div> The id of the plane to query.
     * @param pCutPlaneEquation <div class="const">out</div> The cut plane equation used to determine the plane.
     * @return `true` if pCutPlaneId refers to a valid cut plane.
     */
    getCutPlaneEquation(pCutPlaneId: number, pCutPlaneEquation: Vector4): boolean;
    /**
     * Gets the point used to determine the given plane position.
     *
     * Returns false if the cut plane was not found.
     *
     * @param pCutPlaneId <div class="const">in</div> The id of the plane to query.
     * @param pCutPlanePoint <div class="const">out</div> The cut plane point used to determine the plane.
     * @return `true` if pCutPlaneId refers to a valid cut plane.
     */
    getCutPlanePoint(pCutPlaneId: number, pCutPlanePoint: Vector3): boolean;
    /**
     * Gets the normal used to determine the given plane normal.
     * The cut plane normal is unit length.
     *
     * Returns false if the plane was not found.
     *
     * @param pCutPlaneId <div class="const">in</div> The id of the plane to query.
     * @param pCutPlaneNormal <div class="const">out</div> The cut plane normal used to determine the plane.
     * @return `true` if pCutPlaneId refers to a valid cut plane.
     */
    getCutPlaneNormal(pCutPlaneId: number, pCutPlaneNormal: Vector3): boolean;
    /**
     * Sets the enabled status of the cut plane.
     *
     * A disabled cut plane does not cut anything.
     *
     * Returns false if the cut plane was not found.
     *
     * @param pCutPlaneId <div class="const">in</div> The id of the plane to query.
     * @param pEnabled <div class="const">in</div> The enabled status of the cutPlane.
     * @return `true` if pCutPlaneId refers to a valid cut plane.
     */
    setCutPlaneEnabled(pCutPlaneId: number, pEnabled: boolean): boolean;
    /**
     * Tells if the given cut plane is enabled.
     *
     * Returns false if the cut plane was not found or if the cut plane is disabled.
     *
     * @param pCutPlaneId <div class="const">in</div> The id of the plane to query.
     * @return `true` if the given cut plane exists and is enabled.
     */
    isCutPlaneEnabled(pCutPlaneId: number): boolean;
    /**
     * Updates the shape of the plane pCutPlaneId, changes the point position and the normal vector of the given plane.
     *
     * Returns false if the plane was not found or if pNormal is (0,0,0).
     *
     * @param pCutPlaneId <div class="const">in</div> The id of the plane to modify.
     * @param pPoint <div class="const">in</div> The new 3D position of a point of the plane.
     * @param pNormal <div class="const">in</div> The new 3D normal vector of the plane.
     * @return `true` if the plane was found, pNormal is not (0,0,0) and therefore the plane orientation/position is updated.
     */
    updateCutPlaneGeometry(pCutPlaneId: number, pPoint: Vector3, pNormal: Vector3): boolean;
}
export { CutPlaneManagerInterface };
/**
 * The CutPlaneManipulatorInterface interface is used to manipulate cut planes.
 *
 * This is a GUI element that allows an end-user to modify the orientation/position of
 * a cut plane.
 *
 * The usage is very simple :
 * * Instantiate/recycle a CutPlaneManipulatorInterface.
 * * Bind the CutPlaneManipulatorInterface to a cut plane.
 * * Show the CutPlaneManipulatorInterface.
 * * The user changes the cut plane through the interactions with the GUI element.
 *
 * ```typescript
 * [[include:CutPlaneManagerInterface_set_cut_plane.ts]]
 * ```
 * <br>
 * The user may know when the bound cut plane is modified through the {@link CutPlaneManagerInterfaceSignal} from the {@link CutPlaneManagerInterface}.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link CutPlaneManagerInterface}
 * @category 3D Rendering
 */
interface CutPlaneManipulatorInterface extends InfiniteObjectInterface {
    /**
     * Sets the cut plane id to manipulate.
     *
     * Setting the id 0 is valid, it effectively unbinds the cut plane manipulator from a cut plane.
     *
     * Setting an invalid id to the cut plane manipulator has the same affect as unbinding the
     * manipulator from a cut plane.
     *
     * Setting the same id has no effect.
     *
     * Returns true if the cut plane id exists or if 0 is set.
     *
     * @param pCutPlaneId <div class="const">in</div> The id of the cut plane to manipulate.
     * @return `true` if the cut plane corresponding to the id is valid or 0.
     */
    setCutPlaneId(pCutPlaneId: number): boolean;
    /**
     * Gets the manipulated cut plane id.
     *
     * @return The manipulated cut plane id. Return 0 if no cut plane is bound.
     */
    getCutPlaneId(): number;
    /**
     * Sets the manipulator visibility.
     *
     * @param pVisible <div class="const">in</div> Shows the manipulator if true else hide it.
     * @return 'true' if the the cut plane manipulator has changed his visibility.
     */
    setVisible(pVisible: boolean): boolean;
    /**
     * Gets the visibility state of the manipulator.
     *
     * @return `true` if the manipulator is visible.
     */
    isVisible(): boolean;
    /**
     * Gets the Cut plane manipulator handler.
     *
     * @return The HTMLelement representing the cut plane manipulator.
     */
    getCutPlaneHandler(): HTMLElement | undefined;
}
export { CutPlaneManipulatorInterface };
/**
 * The DataSessionInterfaceSignal lists all the signals that may be sent by the {@link DataSessionInterface}.
 *
 * These signals are emitted by the {@link DataSessionInterface} when
 * * the data session has finished loading (DMULoadingFailed and DMULoadingSuccess).
 * * A message from the server has been received (MessageReceived).
 * * The data session is actually closed (after a call to {@link DataSessionInterface.closeDataSession}) (DataSessionClosed)
 * * The idle system has changed its state (DataSessionReactivated,IdleWarningDataSession, IdleDataSession).
 *
 * The data session is opened following the given sequence :
 *
 * <img src="media://opendatasession.svg" alt="open data session sequence">
 * <br>
 * The idle system is in charge to close the database after a while when no user interactions have been detected.
 * It is composed of three pseudo-states :
 * * Normal : the user is interacting regularly with the system.
 * * Warning : the user has not interacted with the system from some time (2/3 of the idle duration as configured in the architecture).
 * * Idle : the server resources have been cleaned up, the session is closed with no possibility to resume or recover. This state
 * happens after the "warning" state.
 *
 * <br><img src="media://idle_system.svg" alt="idle system">
 * <br>The idle system.
 *
 * ```typescript
 * [[include:DataSessionInterface_openDataSession.ts]]
 * ```
 * <br>
 * Please refer to the {@link DataSessionInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link DataSessionInterface}
 * @see {@link MessageInterface}
 * @category Events
 */
declare enum DataSessionInterfaceSignal {
    /**
     * Signal sent by the {@link DataSessionInterface} when the user has been idle for a long time and
     * the data session will be closed shortly after. If the user moves shortly after, nothing
     * is done. You may cancel the idle behavior by calling the {@link DataSessionInterface.restartIdleTimer} function.
     *
     * No attachment for events of this type.
     */
    IdleWarningDataSession = "idleWarningDataSession",
    /**
     * Signal sent by the {@link DataSessionInterface} when the data session has already sent the {@link IdleWarningDataSession} event, and the user has interacted with the application (performing a metadata filter request)
     * or {@link DataSessionInterface.restartIdleTimer} has been called.
     *
     * No attachment for events of this type.
     */
    DataSessionReactivated = "dataSessionReactivated",
    /**
     * Signal sent by the {@link DataSessionInterface} when the data session has already sent the {@link IdleWarningDataSession} event and the {@link DataSessionInterface} has been closed
     * on the servers. There is no possible recovery when the {@link IdleDataSession} event is fired.
     *
     * Warning :  at this point no cleanup has been done
     * on the client side (only server side cleanup has already been done). You MUST call {@link DataSessionInterface.closeDataSession} to cleanup up the client resources.
     *
     * This behavior is done to allow the application to show a message on top of the 3D before the actual 3D cleanup (if relevant).
     *
     * No attachment for events of this type.
     */
    IdleDataSession = "idleDataSession",
    /**
     * Signal sent by the {@link DataSessionInterface} when the {@link DataSessionInterface} has finished loading the required DMU, i.e. after
     * {@link DataSessionInterface.openDataSession} has been called.
     *
     * A this point, the user can create filters, search in the DMU, etc....
     *
     * No attachment for events of this type.
     */
    DMULoadingSuccess = "dmuLoadingSuccess",
    /**
     * Signal sent by the {@link DataSessionInterface} when the {@link DataSessionInterface} has encountered an error while loading the DMU,  i.e. after
     * {@link DataSessionInterface.openDataSession} has been called.
     *
     * The attachment is a string describing the reason for the error.
     */
    DMULoadingFailed = "dmuLoadingFailed",
    /**
     * Signal sent by the {@link DataSessionInterface} when the {@link DataSessionInterface} has finished cleaning the client and server resources.
     * It may happen that the server resources have already been cleaned (by an idle time for example), in this case, the signal is emitted right
     * after the client resources have been cleaned.
     *
     * No attachment for events of this type.
     */
    DataSessionClosed = "closeDataSession",
    /**
     * Signal sent by the {@link DataSessionInterface} when the {@link DataSessionInterface} has received an informational message from the server.
     *
     * The attachment is a {@link MessageInterface} that contains the message.
     */
    MessageReceived = "messageReceived"
}
export { DataSessionInterfaceSignal };
/**
 * Represents a promise result, for {@link DataSessionInterface.asyncOpenDataSession}.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @category Sessions
 */
declare enum AsyncDataSessionInterfaceOpenResult {
    /**
     * The {@link DataSessionInterface.asyncOpenDataSession} succeeded and the data session is ready to be used.
     * This may be the case if the data session was already loading or loaded at the time of the call.
     */
    OpenResult_Success = 1,
    /**
     * The {@link DataSessionInterface.asyncOpenDataSession} failed, either due to some misconfiguration or a close request has been sent.
     */
    OpenResult_Failed = 2,
    /**
     * The {@link DataSessionInterface.asyncOpenDataSession} failed because the data session was already closed at the time of the call.
     */
    OpenResult_AlreadyClosed = 0
}
export { AsyncDataSessionInterfaceOpenResult };
/**
 * The {@link DataSessionInterface} is the formalization of the connection between a 3djuump infinite proxy and the client.
 *
 * The {@link DataSessionInterface} is created from a connected {@link DirectorySessionInterface} and a proxy by the use of the
 * {@link DirectorySessionInterface.createDataSession} function. This function also allows the user to choose a specific proxy by the use of
 * a {@link tChooseProxyCallback}.
 *
 * ```typescript
 * [[include:DirectorySessionInterface_choose_proxy.ts]]
 * ```
 * <br>
 * The open data session procedure is detailed below :
 *
 * <img src="media://opendatasession.svg" alt="open data session sequence">
 * <br>
 * The DataSession can only be opened once, when closed, there is no way to reuse this object.
 * You will have to call again the {@link DirectorySessionInterface.createDataSession} function.
 *
 * ```typescript
 * [[include:DataSessionInterface_openDataSession.ts]]
 * ```
 * <br>
 * Idle System.
 *
 * In order to save resources on the proxy, the api features an automatic data session closing mechanism
 * when the user is not interacting with the application. The {@link DataSessionInterface} will be closed if
 * {@link DataSessionInterface.restartIdleTimer} is not called on user interactions.
 *
 * It is composed of three pseudo-states :
 * * Normal : the user is interacting regularly with the system.
 * * Warning : the user has not interacted with the system from some time (2/3 of the idle duration as configured in the architecture).
 * * Idle : the server resources have been cleaned up, the session is closed with no possibility to resume or recover. This state
 * happens after the "warning" state.
 *
 * <img src="media://idle_system.svg" alt="idle system">
 * <br>The idle system.
 *
 * ```typescript
 * [[include:DataSessionInterface_handle_idle.ts]]
 * ```
 * <br>
 * Please see {@link DirectorySessionInterface} for more explanations about sessions.
 *
 * All metadata filters, search procedure, data retrieval procedures are created from this interface.
 *
 * The DataSessionInterface, once successfully connected to a 3djuump infinite build
 * (see {@link DataSessionInterfaceSignal.DMULoadingSuccess}) provides access to
 * * Filtering interface creation : {@link FilterItemInterface | FilterItemInterfaces} / {@link VisibilityContextInterface} / {@link FilterSolverInterface} /
 * {@link ConfContextInterface}.
 * * Id converters creation :{@link DocumentIdConverterInterface}, {@link PartInstanceConverterInterface}, {@link GeometricInstanceConverterInterface}.
 * * data retrieval interfaces : {@link IdCardGetterInterface}, {@link DocumentContentGetterInterface}, {@link SearchInterface}.
 *
 * Id converters and data retrieval interfaces are somewhat autonomous. They only rely on a
 * <a href="../index.html#filtering-context">filtering Context</a> to be used.
 *
 * However, filtering interfaces are linked together, making difficult to find the correct order to update them and when. For these
 * reasons, and to avoid too many requests being sent to the 3djuump infinite server, the update is triggered by the
 * DataSessionInterface that handles the dependency graph of all these items, and updates only the required interfaces with
 * {@link update}.
 *
 * All these interfaces (filtering, id conversion, data retrieval) work asynchronously : they all feature some
 * kind of "ready" signal telling the result is available, just call
 * {@link EventDispatcherInterface.addEventListener} with the correct signal on the required interfaces.
 *
 * These interfaces has a {@link InfiniteObjectInterface.dispose} function to get rid of them. Do not forget to call these functions to save memory and
 * CPU usage.
 *
 * The only way to create sets of `part instances` is to use a {@link FilterSolverInterface} that allows computing the
 * result of the intersection of filtering queries with some <a href="../index.html#configurations">infinite configurations</a>
 * (see {@link ConfigurationInterface}, {@link ConfContextInterface}) and optionally other
 * {@link FilterSolverInterface | FilterSolverInterface(s)}. Available configurations (created by the 3djuump infinite maintainer)
 * are accessed through {@link getConfigurationList}, they are then "included" by id in a {@link ConfContextInterface}.
 *
 * DMU statistics (Axis Aligned Bounding Box ({@link AABB}) of each `geometric instance`,
 * diagonal length of each `geometric instance`, min/max of diagonal length, DMU {@link AABB}, DMU {@link Unit}, maximum `geometric instance id`,
 * maximum `part instance id`) and the metadata dictionary ({@link AttributesDictionaryInterface}) are accessed through the DataSessionInterface.
 *
 * ```typescript
 * [[include:FilterSolverInterface_colorize.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:FilterSolverInterface_asyncColorize.ts]]
 * ```
 * <br>
 * The DataSessionInterface now offers the same functionalities than the former {@link MetadataManagerInterface}.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DataSessionInterfaceSignal}
 * @category Sessions
 */
interface DataSessionInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Starts the open data session procedure.
     *
     * A connection is established between the client and the 3djuump infinite proxy.
     * Security tokens are exchanged and initialization data is retrieved from the proxy and the
     * cache if relevant. This operation may take from a few seconds to several minutes depending on the size
     * of the build to work with.
     *
     * When data is ready, the [[DataSessionInterfaceSignal.DMULoadingSuccess] signal is sent.
     *
     * If an error occurs, then the [[DataSessionInterfaceSignal.DMULoadingFailed] signal is sent.
     *
     * @return `true` if the data session open procedure can be started. Returns false in case the {@link DataSessionInterface} has already been opened.
     */
    openDataSession(): boolean;
    /**
     * Asynchronous - starts the open data session procedure.
     *
     * A connection is established between the client and the 3djuump infinite proxy.
     * Security tokens are exchanged and initialization data is retrieved from the proxy and the
     * cache if relevant. This operation may take from a few seconds to several minutes depending on the size
     * of the build to work with.
     *
     * The promise is then resolved with the correct {@link AsyncDataSessionInterfaceOpenResult} value.
     *
     * @return A promise. The promise is resolved with the reason (success, failed, already closed).
     */
    asyncOpenDataSession(): Promise<AsyncDataSessionInterfaceOpenResult>;
    /**
     * Sends a close request concerning the current data session on the proxy.
     *
     * This actually cleans up the data retained by the proxy and cleans up the resources allocated to
     * the {@link InfiniteEngineInterface}.
     *
     * When the data session is effectively closed, then the {@link DataSessionInterfaceSignal.DataSessionClosed} signal is sent.
     *
     * @param _pDoNotCareAboutResult <div class="const">in</div> Ignored, just here for compatibility with prior versions.
     */
    closeDataSession(_pDoNotCareAboutResult?: boolean): void;
    /**
     * Asynchronous - Sends a close request concerning the current data session on the proxy.
     *
     * This actually cleans up the data retained by the proxy and cleans up the resources allocated to
     * the {@link InfiniteEngineInterface}.
     *
     * When the data session is effectively closed, the promise is resolved.
     *
     * @return A promise.
     */
    asyncCloseDataSession(): Promise<void>;
    /**
     * Tells if the data session is connected to a proxy.
     *
     * If the proxy has not yet received the {@link DataSessionInterfaceSignal.DMULoadingSuccess} or {@link DataSessionInterfaceSignal.DMULoadingFailed} signal, returns false.
     *
     * Returns true if {@link openDataSession} has been called and {@link closeDataSession} has not yet been called.
     *
     * @return `true` if all initialization data have been parsed.
     */
    isConnected(): boolean;
    /**
     * Relaunches the idle timer.
     *
     * For security reasons, the data session is closed when no event has been received for a long time, this duration
     * depends on the server configuration. You may reset the idle timer at some time to avoid the user from being disconnected.
     *
     * When the user will be soon disconnected, then the {@link DataSessionInterfaceSignal.IdleWarningDataSession} signal is sent, if the user interacts with the system or
     * restartIdleTimer is called, then the DataSessionReactivated signal is sent.<br>
     * If no user interaction takes place, then the data session is closed, the {@link DataSessionInterfaceSignal.IdleDataSession} signal is sent (no {@link DataSessionInterfaceSignal.DataSessionClosed} signal will be sent).
     * You will have to call {@link closeDataSession} to clean up the resources.
     *
     */
    restartIdleTimer(): void;
    /**
     * Gets the data session token that may be used in other requests to prove the authentication to the 3djuump infinite architecture.
     *
     * Any http request should have the header `x-infinite-bearer : <DataSessionInterface.getDataSessionBearer()>`.
     * If the data session is not connected, returns an empty string.
     *
     * @return The data session token used to prove the authentication.
     */
    getDataSessionBearer(): string;
    /**
     * Gets the extended data session token that may be used in other requests to prove the authentication to the 3djuump infinite architecture.
     *
     * The extended bearer is heavier than {@link getDataSessionBearer} and is available only if the directory session was connected with an
     * extended bearer request (Please refer to {@link DirectorySessionInterface.getSamePageAuthenticationUrl} and {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl}).
     *
     * Any http request may have the header `x-infinite-bearer : <DataSessionInterface.getExtendedDataSessionBearer()>`.
     * If the data session is not connected, returns an empty string.
     *
     * @return The extended data session token used to prove the authentication.
     */
    getExtendedDataSessionBearer(): string;
    /**
     * Gets the attributes dictionary of the DMU.
     *
     * The {@link AttributesDictionaryInterface} is a read only interface to know all
     * the available attributes and their types.
     *
     * @return <div class="const">const</div> The attribute dictionary of the DMU.
     */
    getAttributesDictionary(): AttributesDictionaryInterface;
    /**
     * Gets the list of configurations.
     *
     * Please see <a href="../index.html#configurations">configurations</a> for more
     * explanations about configurations.
     *
     * Modifying this array in place results in undefined behavior.
     *
     * DO NOT modify this array.
     *
     * @return <div class="const">const</div> The list of configurations.
     */
    getConfigurationList(): Array<ConfigurationInterface>;
    /**
     * Gets the list of available annotation types.
     *
     * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
     *
     * Modifying this array in place results in undefined behavior.
     *
     * DO NOT modify this array.
     *
     * @return <div class="const">const</div> The list of annotation types.
     */
    getAnnotationTypes(): Array<string>;
    /**
     * Gets a project document by its id.
     *
     * Available project documents are :
     * * <span style="text-decoration: line-through">com.3djuump:scripts (idcard customization script) </span> (deprecated).
     * * com.3djuump:defaultsettings (field of view, orientation, etc).
     * * com.3djuump:indexinfo (elasticsearch index information) : this information is accessible through the {@link AttributesDictionaryInterface}.
     *
     * Please refer to the integration manual to have more information about these
     * documents.
     *
     * It is very unlikely any developer will use these versioned documents.
     *
     * Example (**illustration purpose only**):
     *
     * ```javascript
     * [[include:project_documents.js]]
     * ```
     * <br>
     * These documents are json documents.
     *
     * Returns the json document as a string if the given document is available in the 3djuump infinite project.
     *
     * It is very unlikely any developer will use
     * {@link getProjectDocument | getProjectDocument("com.3djuump:scripts")} as this script should be used for the
     * 3djuump infinite native client.
     *
     * @param pDocumentId <div class="const">in</div> The project document id to fetch.
     * @return The document as a string, or undefined if the document could not be found or the DMU is not loaded.
     */
    getProjectDocument(pDocumentId: string): string | undefined;
    /**
     * Gets the axis aligned bounding box of the given `geometric instance id`.
     *
     * Call this function after the DMU has been loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess})
     * with a previously created {@link AABB}.
     *
     * Returns true if the given `geometric instance id` is valid.
     *
     * @param pGeometricInstanceId <div class="const">in</div> The <code>geometric instance id</code> to query.
     * @param pAABBOut <div class="const">out</div> The resulting AABB of the <code>geometric instance</code>.
     * @return `true` if pAABBOut is updated.
     */
    getGeometricAABB(pGeometricInstanceId: number, pAABBOut: AABB): boolean;
    /**
     * Gets the diagonal **squared** length of the Oriented Bounding Box (OBB) of the given `geometric instance id`.
     *
     * Returns -1 if pGeometricInstanceId is an invalid  `geometric instance id`, or the diagonal squared length of the OBB if it is valid.
     *
     * @param pGeometricInstanceId <div class="const">in</div> The <code>geometric instance id</code> to query.
     * @return The diagonal **squared** length of the OBB of the <code>geometric instance</code> or -1 if invalid.
     */
    getGeometricOBBDiagonalSquaredLength(pGeometricInstanceId: number): number;
    /**
     * Gets the minimum diagonal **squared** length of all the Oriented Bounding Box (OBB) of all the `geometric instances`.
     *
     * Returns the minimum diagonal squared length of the current DMU, or -1 if a DMU has not been loaded ().
     *
     * @return The minimum diagonal squared length of the current DMU, or -1 if no DMU is loaded.
     */
    getGeometricOBBDiagonalSquaredMin(): number;
    /**
     * Gets the maximum diagonal **squared** length of all the Oriented Bounding Box (OBB) of all the `geometric instances`.
     *
     * Returns the maximum diagonal squared length of the current DMU, or -1 if a DMU has not been loaded
     * ({@link DataSessionInterfaceSignal.DMULoadingSuccess}).
     *
     * @return The maximum diagonal squared length of the current DMU, or -1 if no DMU is loaded.
     */
    getGeometricOBBDiagonalSquaredMax(): number;
    /**
     * Gets the {@link Unit} of the DMU currently loaded.
     *
     * Returns {@link Unit.U_Invalid} if no DMU is loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess}).
     *
     * @return The {@link Unit} of the DMU (millimeters, centimeters, meters, etc).
     */
    getDmuBuildUnit(): Unit;
    /**
     * Gets the Axis Aligned Bounding Box of the currently loaded DMU.
     *
     * Returns `false` if no DMU is loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess}).
     *
     * @param pDmuAABBOut <div class="const">out</div> AABB of the DMU.
     * @return `true` if a DMU is loaded and pDmuAABBOut is updated.
     */
    getDmuAABB(pDmuAABBOut: AABB): boolean;
    /**
     * Gets the maximum `geometric instance id` of the DMU.
     *
     * Valid `geometric instance ids` range from `1` to <code>{@link getMaximumGeometricId | getMaximumGeometricId()}</code> included.
     * Please refer to <a href="../index.html#main-id-types">Main ID Types</a> for more information.
     *
     * Returns the maximum `geometric instance id` if a DMU is loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess},
     * 0 else.
     *
     * @return The maximum <code>geometric instance id</code>, 0 if no DMU is loaded.
     */
    getMaximumGeometricId(): number;
    /**
     * Gets the maximum `part instance id` of the DMU.
     *
     * Valid `part instance ids` range from `1` to <code>{@link getMaximumPartInstanceId | getMaximumPartInstanceId()}</code> included.
     * Please refer to <a href="../index.html#main-id-types">Main ID Types</a> for more information.
     *
     * Returns the maximum `part instance id` if a DMU is loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess},
     * 0 else.
     *
     * @return The maximum <code>part instance id</code>, 0 if no DMU is loaded.
     */
    getMaximumPartInstanceId(): number;
    /**
     * Instantiates a filtering context.
     *
     * A filtering context ({@link VisibilityContextInterface}) is composed of a list of
     * {@link FilterSolverInterface | FilterSolverInterface(s)},
     * and a required {@link ConfContextInterface}. Filtering context may be used
     * to restrict the next filtering requests
     * to a set a specific `part instances`.
     *
     * The {@link VisibilityContextInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link VisibilityContextInterface.dispose}.
     *
     * @param pVisibilityCtxId <div class="const">in</div> if set, the {@link VisibilityContextInterface} will be set with this id.
     * @return A new filtering context bound to the currently loaded DMU.
     * @see {@link VisibilityContextInterface}
     */
    createVisibilityContext(pVisibilityCtxId?: string): VisibilityContextInterface;
    /**
     * Instantiates a {@link ConfContextInterface}.
     *
     * A {@link ConfContextInterface} MUST be used inside a {@link VisibilityContextInterface} to create a valid filtering context.
     * The {@link ConfContextInterface} is the expression of a selected <a href="../index.html#configurations">configuration</a>.
     *
     * The {@link ConfContextInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link ConfContextInterface.dispose}.
     *
     * @param pConfCtxId <div class="const">in</div> if set, the {@link ConfContextInterface} will be set with this id.
     * @return A new {@link ConfContextInterface}.
     * @see {@link ConfContextInterface}
     */
    createConfContext(pConfCtxId?: string): ConfContextInterface;
    /**
     * Instantiates a {@link FilterSolverInterface}.
     *
     * A {@link FilterSolverInterface} is the expression of a combination of multiple
     * {@link FilterItemInterface | FilterItemInterfaces} with a specific operator to create
     * `part instance ids` sets and `geometric instance ids` sets.
     *
     * The {@link FilterSolverInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterSolverInterface.dispose}.
     *
     * @param pFilterSolverId <div class="const">in</div> if set, the {@link FilterSolverInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterSolverInterface}.
     * @see {@link FilterSolverInterface}
     */
    createFilterSolver(pFilterSolverId?: string): FilterSolverInterface;
    /**
     * Instantiates an all `part instances` filter.
     *
     * A {@link FilterAllPartsInterface} is a {@link FilterItemInterface} that elects all `part instances`
     * of the DMU. It is the most optimized way to create a filter that contains all `part instances`.
     *
     * The {@link FilterAllPartsInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterAllPartsInterface} is bound to the currently {@link DataSessionInterface}.
     *
     * Get rid of the object with {@link FilterAllPartsInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterAllPartsInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterAllPartsInterface}.
     * @see {@link FilterAllPartsInterface}
     */
    createFilterAllParts(pFilterId?: string): FilterAllPartsInterface;
    /**
     * Instantiates a metadata text attribute value filter.
     *
     * A {@link FilterAttributeInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that contain in their `joined attribute set` a string attribute whose value matches with the ones contained inside
     * the {@link FilterAttributeInterface}.
     *
     * The {@link FilterAttributeInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface} or {@link FilterCompoundInterface}).
     *
     * The {@link FilterAttributeInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterAttributeInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterAttributeInterface} will be with to this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterAttributeInterface}.
     * @see {@link FilterAttributeInterface}
     */
    createFilterAttribute(pFilterId?: string): FilterAttributeInterface;
    /**
     * Instantiates a metadata numeric attribute value filter.
     *
     * A {@link FilterRangeInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that contain in their `joined attribute set` a numeric attribute (number, date, number range, date range) whose value
     * is contained within the ranges contained inside
     * the {@link FilterRangeInterface}.
     *
     * The {@link FilterRangeInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface} or {@link FilterCompoundInterface}).
     *
     * The {@link FilterRangeInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterRangeInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterRangeInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterRangeInterface}.
     * @see {@link FilterRangeInterface}
     */
    createFilterRange(pFilterId?: string): FilterRangeInterface;
    /**
     * Instantiates a metadata attribute filter.
     *
     * A {@link FilterHasFieldInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that contain in their `joined attribute set` an attribute whose name
     * is set by the {@link FilterHasFieldInterface}.
     *
     * The {@link FilterHasFieldInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface} or {@link FilterCompoundInterface}).
     *
     * The {@link FilterHasFieldInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterHasFieldInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterHasFieldInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterHasFieldInterface}.
     * @see {@link FilterHasFieldInterface}
     */
    createFilterHasField(pFilterId?: string): FilterHasFieldInterface;
    /**
     * Instantiates a metadata boolean attribute value filter.
     *
     * A {@link FilterBooleanInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that contain in their `joined attribute set` a boolean attribute (true, false) whose value
     * is set by
     * the {@link FilterBooleanInterface}.
     *
     * The {@link FilterBooleanInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface} or {@link FilterCompoundInterface}).
     *
     * The {@link FilterBooleanInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterBooleanInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterBooleanInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterBooleanInterface}.
     * @see {@link FilterBooleanInterface}
     */
    createBooleanFilter(pFilterId?: string): FilterBooleanInterface;
    /**
     * Instantiates a query string filter.
     *
     * A {@link FilterLiteralInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that have at least one individual metadata document that satisfies the string query set by the {@link FilterLiteralInterface}.
     *
     * Such a query should be written in the <a href="../index.html#3djuump-infinite-literal-and-search-query-language">
     * 3djuump infinite literal and search query language</a>.
     *
     * The {@link FilterLiteralInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterLiteralInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterLiteralInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterLiteralInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterLiteralInterface}.
     * @see {@link FilterLiteralInterface}
     */
    createLiteralFilter(pFilterId?: string): FilterLiteralInterface;
    /**
     * Instantiates a group metadata filter.
     *
     * A {@link FilterCompoundInterface} is a group filter that elects `documents`
     * whose metadata satisfy the clauses contained inside ({@link FilterAttributeInterface},
     * {@link FilterRangeInterface}, {@link FilterHasFieldInterface}, {@link FilterBooleanInterface}).
     *
     * When the set of documents that match the clauses are found, then `part instances` that
     * are referenced by these documents are elected.
     *
     * The {@link FilterCompoundInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterCompoundInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterCompoundInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterCompoundInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterCompoundInterface}.
     * @see {@link FilterCompoundInterface}
     */
    createCompoundFilter(pFilterId?: string): FilterCompoundInterface;
    /**
     * Instantiates an oriented bounding box diagonal **SQUARED** length filter.
     *
     * A {@link FilterDiagonalInterface} is a {@link FilterItemInterface} that elects `part instances`
     * whom oriented bounding box diagonal **SQUARED** length is contained within the ranges
     * specified in the {@link FilterDiagonalInterface}.
     *
     * The {@link FilterDiagonalInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterDiagonalInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterCompoundInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterDiagonalInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterDiagonalInterface}.
     * @see {@link FilterDiagonalInterface}
     */
    createDiagonalFilter(pFilterId?: string): FilterDiagonalInterface;
    /**
     * Instantiates an axis aligned bounding box filter.
     *
     * A {@link FilterAABBInterface} is a {@link FilterItemInterface} that elects `part instances`
     * whom triangles intersect the {@link FilterAABBInterface} {@link AABB}.
     *
     * The {@link FilterAABBInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterAABBInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterAABBInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterAABBInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterAABBInterface}.
     * @see {@link FilterAABBInterface}
     */
    createFilterAABB(pFilterId?: string): FilterAABBInterface;
    /**
     * Instantiates a `part instance ids` list filter.
     *
     * A {@link FilterPartInstanceListInterface} is a {@link FilterItemInterface} that elects `part instances`
     * whose `part instance id` is contained in the {@link FilterPartInstanceListInterface} `part instance ids` list.
     *
     * **WARNING** : using a {@link FilterPartInstanceListInterface} is very risky since `part instance ids` are recomputed at
     * each new 3djuump Infinite build process.
     *
     * The {@link FilterPartInstanceListInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterPartInstanceListInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterPartInstanceListInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterPartInstanceListInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterPartInstanceListInterface}.
     * @see {@link FilterPartInstanceListInterface}
     */
    createFilterPartInstanceList(pFilterId?: string): FilterPartInstanceListInterface;
    /**
     * Instantiates a "group" filter.
     *
     * A {@link FilterSetInterface} is a {@link FilterItemInterface} that allows grouping
     * [FilterItemInterface | FilterItemInterfaces] to introduce an operator precedence
     * between them. It is similar to a "parenthesis" system (see {@link FilterOperator}).
     *
     * The {@link FilterSetInterface} can be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}), but beware there is a hard coded limit on the number of
     * imbricated {@link FilterSetInterface} (a {@link FilterSetInterface} inside a {@link FilterSetInterface}
     * inside a {@link FilterSetInterface} etc).
     *
     * Please refer to {@link FilterSetInterface} and {@link FilterItemInterface.getDepthContribution}.
     *
     * The {@link FilterSetInterface} is bound to the given DataSessionInterface.
     *
     * Get rid of the object with {@link FilterSetInterface.dispose}.
     *
     * @param pFilterId <div class="const">in</div> if set, the {@link FilterSetInterface} will be set with this id.
     * In this case, the caller is responsible to set a unique id.
     * @return A new {@link FilterSetInterface}.
     * @see {@link FilterSetInterface}
     * @see {@link FilterOperator}
     */
    createFilterSet(pFilterId?: string): FilterSetInterface;
    /**
     * Instantiates a converter to translate a `document id` to its corresponding `part instance ids` and `geometric instance ids`.
     *
     * A {@link DocumentIdConverterInterface} may be used in a <a href="../index.html#search">search procedure</a>.
     *
     * @return A {@link DocumentIdConverterInterface}.
     * @see {@link DocumentIdConverterInterface}
     */
    createDocumentIdConverter(): DocumentIdConverterInterface;
    /**
     * Instantiates a converter to translate `part instances ids` to their corresponding `geometric instance ids`.
     *
     * This is the opposite of the {@link GeometricInstanceConverterInterface}.
     *
     * @return A new {@link PartInstanceConverterInterface}.
     * @see {@link PartInstanceConverterInterface}
     */
    createPartInstanceConverter(): PartInstanceConverterInterface;
    /**
     * Instantiates a converter to translate `geometric instance ids` to their corresponding `part instance ids`.
     *
     * This is the opposite of the {@link PartInstanceConverterInterface}.
     *
     * @return A new {@link GeometricInstanceConverterInterface}.
     * @see {@link GeometricInstanceConverterInterface}
     * @see {@link PartInstanceConverterInterface}
     */
    createGeometricInstanceConverter(): GeometricInstanceConverterInterface;
    /**
     * Instantiates a <a href="../index.html#search">search procedure</a>.
     *
     * A {@link SearchInterface} is used to trigger search query(ies) in the 3djuump infinite metadata
     * documents, and returns matching metadata documents. The {@link SearchInterface} relies on the
     * <a href="../index.html#3djuump-infinite-literal-and-search-query-language">
     * 3djuump infinite literal and search query language</a>.
     *
     * @return A new {@link SearchInterface}.
     * @see {@link SearchInterface}
     */
    createSearch(): SearchInterface;
    /**
     * Instantiates a data retrieval procedure to retrieve the metadata documents
     * associated to a list of `part instance ids` and their ancestors.
     *
     * Such a list of metadata documents can be merged and customized to
     * display an "idcard" of a `part instance` and its ancestors.
     *
     * @return A new {@link IdCardGetterInterface}.
     * @see {@link IdCardGetterInterface}
     */
    createIdCardGetter(): IdCardGetterInterface;
    /**
     * Instantiates a data retrieval procedure to retrieve the metadata documents
     * of the children of a list of `part instance ids`.
     *
     * Such a list of metadata documents can be merged and customized to
     * display an metadata information of the children of a `part instance`.
     *
     * @return A new {@link ChildrenIdCardGetterInterface}.
     * @see {@link ChildrenIdCardGetterInterface}
     */
    createChildrenIdCardGetter(): ChildrenIdCardGetterInterface;
    /**
     * Instantiates a data retrieval procedure to retrieve the content of an attached document.
     *
     * Attached document are documents (images, html pages, links, pdf documents, etc ...)
     * that may be linked to `part instances`. The list of available documents for a `part instance`
     * are retrieved by an {@link IdCardGetterInterface}, then individual documents may be retrieved by
     * a {@link DocumentContentGetterInterface}.
     *
     * @return A new {@link DocumentContentGetterInterface}.
     * @see {@link DocumentContentGetterInterface}
     */
    createDocumentContentGetter(): DocumentContentGetterInterface;
    /**
     * Instantiates an annotation retrieval procedure to retrieve the content of a view of annotations.
     *
     * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
     *
     * Annotation views may be linked to `part instances`. The list of annotation views for a `part instance`
     * are retrieved by an {@link IdCardGetterInterface}, then individual views may be retrieved by
     * a {@link AnnotationGetterInterface}.
     *
     * @return A new {@link AnnotationGetterInterface}.
     * @see {@link AnnotationGetterInterface}
     */
    createAnnotationGetter(): AnnotationGetterInterface;
    /**
     * Creates an {@link AttachedDocumentInfoInterface} to be used in a {@link DocumentContentGetterInterface}.
     *
     * If the user knows the `attached document id` and the mime type of an attached document, then a
     * {@link DocumentContentGetterInterface} can be used to retrieve the corresponding attached document.
     *
     * @return A new {@link AttachedDocumentInfoInterface}.
     * @see {@link DocumentContentGetterInterface}
     * @see {@link AttachedDocumentInfoInterface}
     */
    createAttachedDocumentInfo(): AttachedDocumentInfoInterface;
    /**
     * Triggers the calculation of all modified {@link FilterSolverInterface | FilterSolverInterfaces},
     * {@link FilterItemInterface | FilterItemInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces}
     * {@link ConfContextInterface | ConfContextInterfaces}.
     *
     * All the modified items are updated on the server according to their
     * inter-dependencies.
     *
     * The relevant interfaces will asynchronously trigger a "ready" signal when their calculation is over.
     */
    update(): void;
    /**
     * Cancels the calculation of all running {@link FilterSolverInterface | FilterSolverInterfaces},
     * {@link FilterItemInterface | FilterItemInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces}
     * {@link ConfContextInterface | ConfContextInterfaces}.
     *
     * The relevant interfaces will asynchronously trigger a "cancel" and "ready" signal when the
     * canceling is done.
     */
    cancelUpdate(): void;
    /**
     * Computes the axis aligned bounding box of the given `geometric instances`.
     *
     * This consists in the union of all the {@link AABB} of the `geometric instances` expressed by their
     * `geometric instance ids`.
     *
     * Call this function once the DMU has been loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess})
     * with a previously created {@link AABB}.
     *
     * Returns true if at least one `geometric instance id` inside pGeometricInstanceIds is valid. Invalid `geometric instance ids`
     * are silently discarded.
     *
     * @param pGeometricInstanceIds <div class="const">in</div> the list of <code>geometric instance ids</code> to query.
     * @param pAABBOut <div class="const">out</div> the union of all the {@link AABB} o the <code>geometric instance</code>.
     * @return `true` if at least  one `geometric instance id` inside pGeometricInstanceIds is valid and pAABBOut is therefore updated.
     */
    getUnitedAABB(pGeometricInstanceIds: Uint32Array | number[], pAABBOut: AABB): boolean;
    /**
     * Gets the conversion ratio between two units.
     *
     * @param pFrom <div class="const">in</div> The reference {@link Unit} where the conversion starts from.
     * @param pTo <div class="const">in</div> The {@link Unit} to convert to.
     * @return A floating number corresponding to the {@link Unit} conversion ratio.
     */
    convertUnitFactor(pFrom: Unit, pTo: Unit): number;
}
export { DataSessionInterface };
/**
 * The {@link DirectorySessionFactory} factory allows to create a {@link DirectorySessionInterface} to authenticate onto a 3djuump Infinite directory.
 *
 * It also provides also a way to decode authentication hashes that are retrieved on the authentication hashes in case of popup based authentication.
 * The authentication system is based on <a href="https://openid.net/connect/" target="_blank">OpenID Connect layer</a>.
 *
 * The directory session will try to connect to the given Host, and is bound to a MetadataManagerInterface.
 *
 * ```typescript
 * [[include:DirectorySessionInterface_create.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#authentication-samples">Authentication Samples</a> for more explanations about Sessions.
 *
 * [[include:constStyle.md]]<br><div class="const">Factories</div><br>
 * @see {@link DirectorySessionInterface}
 * @category Factories
 */
declare class DirectorySessionFactory {
    /**
     * Creates a directory session.
     *
     * Warning : creating a directory session from Host / Path / Port is now deprecated, please use the url signature.
     *
     * @param pMetadataManager <div class="const">in</div> The {@link MetadataManagerInterface} the resulting {@link DirectorySessionInterface} will be bound to.
     * @param pDirectoryUrlOrHost <div class="const">in</div> The url of 3djuump Infinite directory (https://host:443/directory) <br><div class="const">deprecated</div>the host of the 3djuump Infinite directory, be it a name or an ip address associated with pDirectoryPath and pDirectoryPort.
     * @param pDirectoryPath <div class="const">deprecated</div> The path to the directory api (defined by the directory configuration, defaults to "/directory").
     * @param pDirectoryPort <div class="const">deprecated</div> The port to the https directory api (defaults to 443).
     * @return The directory session object, that will try the authentication upon calling the authenticate functions.
     */
    static CreateDirectorySession(pMetadataManager: MetadataManagerInterface, pDirectoryUrlOrHost: string, pDirectoryPath?: string, pDirectoryPort?: number): DirectorySessionInterface;
    /**
     * Decodes the hash content and sends the token to the caller of {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl} in case of popup based authentications.
     *
     * This will trigger the {@link DirectorySessionInterfaceSignal.LoginSuccess} signal on the tab that called {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl}.
     *
     * This function makes use of the LocalStorage of the browser.
     *
     * ```typescript
     * [[include:DirectorySessionFactory_authenticate.ts]]
     * ```
     * <br>
     * Please see <a href="../index.html#authentication-samples">Authentication Samples</a> for more explanations about Sessions.
     *
     * @param pHash <div class="const">in</div> The hash to decode (usually window.location.hash).
     * @return `true` if the hash was correctly decoded.
     */
    static DecodeAuthenticationHash(pHash: string): boolean;
}
export { DirectorySessionFactory };
/**
 * The {@link DirectorySessionInterfaceSignal} lists all the signals that may be sent by the {@link DirectorySessionInterface}.
 *
 * These signals are emitted by the {@link DirectorySessionInterface} when
 * * The authentication mechanism has succeeded (LoginSuccess).
 * * The authentication mechanism has failed (LoginFailed).
 * * The {@link DirectorySessionInterface} has been disconnected (LoggedOff).
 * * The build list available from the directory has been refreshed (after a call to {@link DirectorySessionInterface.refreshAccessRightsInfo}) (AccessRightsRefreshed).
 * * The build list available from the directory could not be refreshed (after a call to {@link DirectorySessionInterface.refreshAccessRightsInfo}) (AccessRightsRefreshFailed).
 *
 * ```typescript
 * [[include:DirectorySessionInterface_signal.ts]]
 * ```
 * <br>
 * Please refer to {@link DirectorySessionInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link DirectorySessionInterface}
 * @category Events
 */
declare enum DirectorySessionInterfaceSignal {
    /**
     * Signal sent by the {@link DirectorySessionInterface} when the user has been logged in after a call to {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl}.
     *
     * The attachment for events of this type is a {@link ConnectionData}.
     */
    LoginSuccess = "loginSuccess",
    /**
     * Signal sent by the {@link DirectorySessionInterface} when the login procedure has failed after a call to {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl}.
     *
     * The attachment for events of this type is an object that contains the field message ( ```typescript { "message": "description of the error"}```).
     */
    LoginFailed = "loginFailed",
    /**
     * Signal sent by the {@link DirectorySessionInterface} when the user has effectively logged off i.e. after a call to {@link DirectorySessionInterface.closeDirectorySession}.
     *
     * No attachment for events of this type.
     */
    LoggedOff = "loggedOff",
    /**
     * Signal sent by the {@link DirectorySessionInterface} when the list of available builds and projects have been refreshed i.e. after a call to {@link DirectorySessionInterface.refreshAccessRightsInfo}.
     *
     * The attachment for events of this type is a {@link ConnectionData}.
     */
    AccessRightsRefreshed = "accessRightsRefreshed",
    /**
     * Signal sent by the {@link DirectorySessionInterface} when the refresh access rights procedure has failed after a call to {@link DirectorySessionInterface.refreshAccessRightsInfo}.
     *
     * The attachment for events of this type is an object that contains the field message ( ```typescript { "message": "description of the error"}```).
     */
    AccessRightsRefreshFailed = "accessRightsRefreshFailed"
}
export { DirectorySessionInterfaceSignal };
/**
 * This enumeration is the result from a call to {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl},
 * {@link DirectorySessionInterface.getSamePageAuthenticationUrl}, and {@link DirectorySessionInterface.setTokenFromHash} functions.
 *
 * It tells if the authentication mechanism could be initiated/completed. The authentication mechanism to the 3djuump infinite architecture is delegated
 * to an external authentication server by the use of an openid connect protocol.
 * The {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl} and {@link DirectorySessionInterface.getSamePageAuthenticationUrl} may return {@link DoNotReAuthenticate}
 * {@link InvalidSession}, and {@link EmptyCallbackURI}.
 * The {@link DirectorySessionInterface.setTokenFromHash} may return {@link AuthenticationPending}, {@link DoNotReAuthenticate}, {@link InvalidSession}, {@link InvalidHash}.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DirectorySessionInterface}
 * @category Sessions
 */
declare enum AuthenticationGetURLResult {
    /**
     * The user is already authenticated or the authentication procedure is already running at the moment.
     *
     * Returned from {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl}, {@link DirectorySessionInterface.getSamePageAuthenticationUrl}.
     */
    DoNotReAuthenticate = 0,
    /**
     * The authentication procedure is running (likely the {@link DirectorySessionInterfaceSignal.LoginSuccess} will be sent later).
     *
     * Returned from {@link DirectorySessionInterface.setTokenFromHash}.
     */
    AuthenticationPending = 1,
    /**
     * An internal error led to the failure of the procedure.
     *
     * Returned from {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl}, {@link DirectorySessionInterface.getSamePageAuthenticationUrl},
     * {@link DirectorySessionInterface.setTokenFromHash}.
     */
    InvalidSession = -1,
    /**
     * The given hash for the authentication is invalid.
     *
     * Returned from {@link DirectorySessionInterface.setTokenFromHash}.
     */
    InvalidHash = -2,
    /**
     * Empty callback URI ("") is not allowed.
     *
     * Returned from {@link DirectorySessionInterface.getPopupBasedAuthenticationUrl}, {@link DirectorySessionInterface.getSamePageAuthenticationUrl}.
     */
    EmptyCallbackURI = -3
}
export { AuthenticationGetURLResult };
/**
 * The ProxyChoiceInterface interface allows the developer to create an heuristic to elect the best proxy to
 * connect to.
 *
 * Future versions of this interface will also include load and number of users heuristics.
 *
 * ```typescript
 * [[include:DirectorySessionInterface_choose_proxy.ts]]
 * ```
 * <br>
 * Please refer to {@link DirectorySessionInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DirectorySessionInterface}
 * @category Sessions
 */
interface ProxyChoiceInterface {
    /**
     * Gets the url of the proxy.
     *
     * @return The url of the given proxy.
     */
    getUrl(): string;
    /**
     * Gets the label of the proxy.
     *
     * @return The label of the given proxy.
     */
    getLabel(): string;
}
export { ProxyChoiceInterface };
/**
 * This type is a callback that must be called inside or after the user defined tChooseProxyCallback.
 *
 * This is used in the open data session procedure, initiated by {@link DirectorySessionInterface.createDataSession} and
 * {@link DataSessionInterface.openDataSession}.
 *
 * This callback actually continues the open data session procedure. If not called, the
 * open data session procedure will never terminate.
 *
 * ```typescript
 * [[include:DirectorySessionInterface_choose_proxy.ts]]
 * ```
 * <br>
 * Please refer to {@link DirectorySessionInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DirectorySessionInterface.createDataSession}
 * @see {@link DataSessionInterface.openDataSession}
 * @category Sessions
 */
type tSetProxyCallback = 
/**
 * @param pUrl <div class="const">in</div> the url of the proxy to elect for the connection. If not found or undefined,
 * a random proxy will be elected.
 */
(pUrl: string | undefined) => void;
export { tSetProxyCallback };
/**
 * This type is a user defined callback that is set in {@link DirectorySessionInterface.createDataSession} function.
 *
 * When {@link DataSessionInterface.openDataSession} is called, this callback will be called asynchronously. The callback
 * may process the list of input proxies to choose the most appropriate one and call pSetProxyCallback to continue
 * the open data session procedure.
 *
 * ```typescript
 * [[include:DirectorySessionInterface_choose_proxy.ts]]
 * ```
 * <br>
 * Please refer to {@link DirectorySessionInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DirectorySessionInterface.createDataSession}
 * @see {@link DataSessionInterface.openDataSession}
 * @category Sessions
 */
type tChooseProxyCallback = 
/**
 * @param pProxies <div class="const">in</div> the list of available proxies.
 * @param pSetProxyCallback <div class="const">in</div> the callback to call when a proxy is elected, call it with
 * {@link ProxyChoiceInterface.getUrl} as parameter.
 */
(pProxies: Array<ProxyChoiceInterface>, pSetProxyCallback: tSetProxyCallback) => void;
export { tChooseProxyCallback };
/**
 * Represents a promise result reason with {@link DirectorySessionInterface.asyncWaitForPopupAuthentication}.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DirectorySessionInterface}
 * @category Sessions
 */
declare enum AsyncDirectorySessionWaitForLoadedResultReason {
    /**
     * The call was cancelled because of the object dispose function was called.
     */
    OpenResult_Disposed = 0,
    /**
     * The {@link DirectorySessionInterface.asyncWaitForPopupAuthentication} succeeded and the data session is ready to be used.
     */
    OpenResult_LoginSuccess = 1,
    /**
     * The {@link DirectorySessionInterface.asyncWaitForPopupAuthentication} succeeded and the data session is ready to be used,
     * but the Directory session was already open.
     */
    OpenResult_AlreadyOpen = 2,
    /**
     * The {@link DirectorySessionInterface.asyncWaitForPopupAuthentication} failed because the popup authentication was not started and
     * the {@link DirectorySessionInterface} is not connected.
     */
    OpenResult_PopupNotStarted = 3,
    /**
     * The {@link DirectorySessionInterface.asyncWaitForPopupAuthentication} failed and the error field contains the error.
     */
    OpenResult_LoginFailed = 4,
    /**
     * The {@link DirectorySessionInterface.asyncWaitForPopupAuthentication} failed because the user logged off before waiting for the
     * result of the connection (less likely to happen).
     */
    OpenResult_LoggedOff = 5
}
export { AsyncDirectorySessionWaitForLoadedResultReason };
/**
 * Represents a promise result reason with {@link DirectorySessionInterface.asyncRefreshAccessRightsInfo}.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DirectorySessionInterface}
 * @category Sessions
 */
declare enum AsyncDirectorySessionRefreshAccessRightsReason {
    /**
     * The {@link DirectorySessionInterface.asyncRefreshAccessRightsInfo} succeeded and the data session is ready to be used.
     */
    Refresh_Success = 1,
    /**
     * The {@link DirectorySessionInterface.asyncRefreshAccessRightsInfo} failed because the Directory session was not open.
     */
    Refresh_NotConnected = 0,
    /**
     * The {@link DirectorySessionInterface.asyncRefreshAccessRightsInfo} failed for some reason.
     */
    Refresh_Error = 3
}
export { AsyncDirectorySessionRefreshAccessRightsReason };
/**
 * Represents a Wait for popup authentication promise result.
 *
 * This value is used with {@link DirectorySessionInterface.asyncWaitForPopupAuthentication}.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DirectorySessionInterface}
 * @see {@link AsyncDirectorySessionWaitForLoadedResultReason}
 * @category Sessions
 */
interface AsyncDirectorySessionWaitForLoadedResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncDirectorySessionWaitForLoadedResultReason;
    /**
     * The connection data in case of success (reason === AsyncDirectorySessionWaitForLoadedResultReason.OpenResult_LoginSuccess).
     */
    value?: ConnectionData;
    /**
     * The error message if any.
     *
     * This is only relevant for {@link AsyncDirectorySessionWaitForLoadedResultReason.OpenResult_LoginFailed}.
     *
     * This value is the same than the attachment of {@link DirectorySessionInterfaceSignal.LoginFailed}.
     *
     * @see {@link DirectorySessionInterfaceSignal.LoginFailed}
     */
    error?: string;
}
export { AsyncDirectorySessionWaitForLoadedResult };
/**
 * Represents a Wait for refresh Access Rights  promise result.
 *
 * This value is used with {@link DirectorySessionInterface.asyncRefreshAccessRightsInfo}.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DirectorySessionInterface}
 * @see {@link AsyncDirectorySessionRefreshAccessRightsReason}
 * @category Sessions
 */
interface AsyncDirectorySessionWaitForRefreshAccessRightsResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncDirectorySessionRefreshAccessRightsReason;
    /**
     * The connection data in case of success (reason === {@link AsyncDirectorySessionRefreshAccessRightsReason.Refresh_Success}).
     */
    value?: ConnectionData;
    /**
     * The error message if any.
     *
     * This is only relevant for {@link AsyncDirectorySessionRefreshAccessRightsReason.Refresh_Error}.
     *
     * This value is the same than the attachment of {@link DirectorySessionInterfaceSignal.LoginFailed}.
     *
     * @see {@link DirectorySessionInterfaceSignal.LoginFailed}
     */
    error?: string;
}
export { AsyncDirectorySessionWaitForRefreshAccessRightsResult };
/**
 * The {@link DirectorySessionInterface} is in charge of the authentication of a user to the 3djuump infinite architecture, and the initiation of a connection
 * to a 3djuump infinite proxy.
 *
 * The {@link DirectorySessionInterface} is the formalization of the session to a 3djuump infinite directory.
 *
 * The 3djuump infinite architecture does not feature an internal authentication procedure, but delegates the authentication
 * to an external authentication server by the use of an openid connect protocol.
 *
 * There is at the moment only 2 ways to authenticate to the 3djuump infinite directory, by the use of a popup window ({@link getPopupBasedAuthenticationUrl}, {@link DirectorySessionFactory.DecodeAuthenticationHash}) or
 * a same page authentication ({@link getSamePageAuthenticationUrl}, {@link setTokenFromHash}).
 *
 * 1. With the popup based authentication.
 * The user will see a new tab opened to the authentication server, asking for a user/password, and at the end a redirection to
 * a custom authentication page (e.g. `authenticate.html`) with a script that calls {@link DirectorySessionFactory.DecodeAuthenticationHash} (see <a href="../index.html">the main page</a> for more information).
 * Depending on your openid configuration, this procedure may be silent for the end-user, with the use of SSO and automatic login. The user may then need to close the authentication tab (or perhaps closed
 * automatically).
 * It is your responsibility to open a new tab (see the code below).
 *
 * ```typescript
 * [[include:api_popup_authentication.ts]]
 * ```
 * <br>
 * 2. With the same page authentication.
 * The user will see multiple redirections (the number depending on your openid connect server and its configuration) from the current tab and will be finally redirected to the start page of your application.
 * The last redirect will be done with a hash parameter that contains authentication information. The developer may need to store url parameters in the browser storage to be redirected to the final url
 * (see <a href="../index.html">the main page</a> for more information). The authentication procedure is initiated with {@link getSamePageAuthenticationUrl} which should return
 * the redirection url to visit, and then {@link setTokenFromHash} on the last redirect. The {@link setTokenFromHash} will normally send {@link DirectorySessionInterfaceSignal.LoginSuccess}
 * asynchronously.
 *
 * ```typescript
 * [[include:api_samepage_authentication.ts]]
 * ```
 * <br>
 * Upon a successful connection, user access rights are retrieved asynchronously with the connection to the {@link DirectorySessionInterfaceSignal.LoginSuccess} signal.
 * User access rights are formalized with the {@link ConnectionData} object, that contains the list of tags, teams attached to this user, and also the
 * available builds/projects for this user. You can force the refreshing of this information with a call to {@link refreshAccessRightsInfo}.
 *
 * The {@link DirectorySessionInterface} is the factory to a {@link DataSessionInterface} that allows to choose a specific proxy to connect to, and begin
 * downloading / initializing data.
 *
 * ```typescript
 * [[include:DirectorySessionInterface_choose_proxy.ts]]
 * ```
 * <br>
 * Please see {@link DataSessionInterface} for more explanations about sessions.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DirectorySessionInterfaceSignal}
 * @see {@link tChooseProxyCallback}
 * @see {@link tSetProxyCallback}
 * @see {@link ProxyChoiceInterface}
 * @see {@link AuthenticationGetURLResult}
 * @see {@link UserData}
 * @category Sessions
 */
interface DirectorySessionInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Gets the URL to the 3djuump infinite directory API.
     *
     * It is `https://pDirectoryHost:pDirectoryPort/pDirectoryPath/` which was set by the factory {@link DirectorySessionFactory.CreateDirectorySession}.
     *
     * @return The URL string of the directory API.
     */
    getApiUrl(): string;
    /**
     * Gets the infinite token of the current directory session authentication.
     *
     * Any http request to the directory should have the header `x-infinite-bearer : <DirectorySessionInterface.getAuthenticationBearer()>`.
     *
     * If the directory session is not connected, returns an empty string.
     *
     * @return The infinite token.
     */
    getAuthenticationBearer(): string;
    /**
     * Gets the Bearer of the current http directory session authentication.
     *
     * Any http request to the directory should have the header `Authorization : Bearer <DirectorySessionInterface.getHttpBearer()>`.
     *
     * If the directory session is not connected, returns an empty string.
     *
     * @return The Bearer of the http authentication.
     */
    getHttpBearer(): string;
    /**
     * Creates a data session concerning the given build id, with the given cache and the choose proxy callback.
     *
     * <img src="media://opendatasession.svg" alt="open data session sequence">
     * <br>
     * Please refer to {@link DataSessionInterface} for more information about sessions.
     *
     * @param pBuildId <div class="const">in</div> The build id to connect to.
     * @param pCache <div class="const">in</div> The optional cache to use to speed up data retrieving (if relevant), may be undefined.
     * @param pCallback <div class="const">in</div> The choose proxy callback to use to elect a specific proxy.
     * @return The newly data session or undefined if the {@link DirectorySessionInterface} is not authenticated to the directory.
     * @see {@link DataSessionInterface}
     */
    createDataSession(pBuildId: string, pCache: InfiniteCacheInterface | undefined, pCallback: tChooseProxyCallback | undefined): DataSessionInterface | undefined;
    /**
     * Creates a data session concerning the given build id, with the given cache.
     *
     * <img src="media://opendatasession.svg" alt="open data session sequence">
     * <br>
     * The first proxy will be elected for the connection since no tChooseProxyCallback has been specified.
     *
     * @param pBuildId <div class="const">in</div> The build id to connect to.
     * @param pCache <div class="const">in</div> The optional cache to use to speed up data retrieving (if relevant), may be undefined.
     * @return The newly data session or undefined if the {@link DirectorySessionInterface} is not authenticated to the directory.
     * @see {@link DataSessionInterface}
     */
    createDataSession(pBuildId: string, pCache: InfiniteCacheInterface | undefined): DataSessionInterface | undefined;
    /**
     * Sends a close request concerning the current authentication session on the directory.
     *
     * This will also revoke all the {@link DataSessionInterface} that were created from this directory session.
     * The {@link DataSessionInterfaceSignal.DataSessionClosed} signal may be sent after a while on the given {@link DataSessionInterface},
     * since {@link DataSessionInterface} objects are autonomous and will discover the closing of their session on their own.
     *
     * If pForceAuthentication is set to true, SSO configuration will be flushed, forcing the user to reconnect on the next call to {@link getPopupBasedAuthenticationUrl} or {@link getSamePageAuthenticationUrl}.
     * If pForceAuthentication is set to false, SSO configuration will be kept leading to, depending on your openid configuration, a silent and automatic connection on
     * the next call to {@link getPopupBasedAuthenticationUrl} or {@link getSamePageAuthenticationUrl}.
     *
     * If the user is not authenticated, this function does nothing.
     *
     * @param pForceAuthentication <div class="const">in</div> If true the next call to {@link getPopupBasedAuthenticationUrl} or {@link getSamePageAuthenticationUrl} will ask the user to re-authenticate.
     * @see {@link DataSessionInterface}
     */
    closeDirectorySession(pForceAuthentication: boolean): void;
    /**
     * Asynchronous - Sends a close request concerning the current authentication session on the directory.
     *
     * This will also revoke all the {@link DataSessionInterface} that were created from this directory session.
     * The {@link DataSessionInterfaceSignal.DataSessionClosed} signal may be sent after a while on the given {@link DataSessionInterface},
     * since {@link DataSessionInterface} objects are autonomous and will discover the closing of their session on their own.
     *
     * If pForceAuthentication is set to true, SSO configuration will be flushed, forcing the user to reconnect on the next call to {@link getPopupBasedAuthenticationUrl} or {@link getSamePageAuthenticationUrl}.
     * If pForceAuthentication is set to false, SSO configuration will be kept leading to, depending on your openid configuration, a silent and automatic connection on
     * the next call to {@link getPopupBasedAuthenticationUrl} or {@link getSamePageAuthenticationUrl}.
     *
     * If the user is not authenticated, this function does nothing and the waiting time is 0.
     *
     * When the directory session is effectively closed, the promise is resolved.
     *
     * @param pForceAuthentication <div class="const">in</div> If true the next call to {@link getPopupBasedAuthenticationUrl} or {@link getSamePageAuthenticationUrl} will ask the user to re-authenticate.
     * @return A promise.
     */
    asyncCloseDirectorySession(pForceAuthentication: boolean): Promise<void>;
    /**
     * Tells if the user is authenticated.
     *
     * @return `true` if the user is authenticated.
     */
    isAuthenticated(): boolean;
    /**
     * Requests an asynchronous refresh of the access rights on the current directory session (build list, projects, report info, tags, teams).
     *
     * Access rights information is available after a successful login (from attachments of {@link DirectorySessionInterfaceSignal.LoginSuccess}) but they can be updated during the session life.
     *
     * The asynchronous request will trigger the signals {@link DirectorySessionInterfaceSignal.AccessRightsRefreshed} or {@link DirectorySessionInterfaceSignal.AccessRightsRefreshFailed}
     * according to the request resolution.
     *
     * @return `false` if the user is not logged-in.
     */
    refreshAccessRightsInfo(): boolean;
    /**
     * Asynchronous - Requests a refresh of the access rights on the current directory session (build list, projects, report info, tags, teams).
     *
     * Access rights information is available after a successful login (from attachments of {@link DirectorySessionInterfaceSignal.LoginSuccess}) but they can be updated during the session life.
     *
     * The {@link AsyncDirectorySessionRefreshAccessRightsReason} of the {@link AsyncDirectorySessionWaitForRefreshAccessRightsResult} may be :
     * * If the refresh was correct : {@link AsyncDirectorySessionRefreshAccessRightsReason.Refresh_Success} (see {@link DirectorySessionInterfaceSignal.AccessRightsRefreshed}).
     * * If the DirectorySessionInterface is not connected : {@link AsyncDirectorySessionRefreshAccessRightsReason.Refresh_NotConnected}.
     * * If the refresh failed for some reason : {@link AsyncDirectorySessionRefreshAccessRightsReason.Refresh_Error} (see {@link DirectorySessionInterfaceSignal.AccessRightsRefreshFailed}).
     *
     * @return A promise with the result of the access rights reason.
     */
    asyncRefreshAccessRightsInfo(): Promise<AsyncDirectorySessionWaitForRefreshAccessRightsResult>;
    /**
     * Opens a new authentication session on the directory with the pop up mechanism, asking the user to authenticate.
     *
     * A new tab is not opened by this function, the user will have to call window.open(getPopupBasedAuthenticationUrl()).
     * Depending on your openid connect configuration, this may be a silent procedure for the user.
     * This function makes use of the localStorage of the browser internally.
     *
     * This function may return a string if successful, {@link AuthenticationGetURLResult.DoNotReAuthenticate} if the {@link DirectorySessionInterface}
     * is already connected, {@link AuthenticationGetURLResult.InvalidSession} if an internal error occurred (unlikely) and {@link AuthenticationGetURLResult.EmptyCallbackURI}
     * if pRedirectURI is an empty string.
     *
     * ```typescript
     * [[include:api_popup_authentication.ts]]
     * ```
     * <br>
     * See <a href="../index.html">the main page</a> for more information.
     *
     * @param pRedirectURI <div class="const">in</div> The redirect url where the authentication server should redirect on the completion of the
     * authentication procedure (successful or not). An empty string will result in an error.
     * @param pAppClaims <div class="const">in</div> An array of application claims that may be used to validate queries on third party servers. Third party servers
     * may validate some requests and look for the application claims to satisfy or reject the request.
     * @param pNeedExtendedBearer <div class="const">in</div> If set to `true`, the access token will contain additional information (subid, teams, etc ...) (defaults to false).
     * @return The authentication URL to visit to start the authentication procedure if successful, {@link AuthenticationGetURLResult.DoNotReAuthenticate} if the {@link DirectorySessionInterface}
     * is already connected, {@link AuthenticationGetURLResult.InvalidSession} if an internal error occurred and {@link AuthenticationGetURLResult.EmptyCallbackURI} if pRedirectURI is an empty string.
     */
    getPopupBasedAuthenticationUrl(pRedirectURI: string, pAppClaims?: Array<string> | undefined, pNeedExtendedBearer?: boolean | undefined): string | AuthenticationGetURLResult;
    /**
     * Waits for the popup authentication to complete.
     *
     * The {@link AsyncDirectorySessionWaitForLoadedResultReason} of the {@link AsyncDirectorySessionWaitForLoadedResult} may be :
     * * If no popup authentication procedure is running, and the DirectorySessionInterface is connected : {@link AsyncDirectorySessionWaitForLoadedResultReason.OpenResult_AlreadyOpen}.
     * * If no popup authentication procedure is running, and the DirectorySessionInterface is not connected : {@link AsyncDirectorySessionWaitForLoadedResultReason.OpenResult_PopupNotStarted}.
     * * If a popup authentication procedure is running, and is completed : {@link AsyncDirectorySessionWaitForLoadedResultReason.OpenResult_LoginSuccess}.
     * * If a popup authentication procedure is running, but failed : {@link AsyncDirectorySessionWaitForLoadedResultReason.OpenResult_LoginFailed}, the {@link AsyncDirectorySessionWaitForLoadedResult.error}
     * field contains the error message.
     * * If the DirectorySessionInterface was disposed during the call : {@link AsyncDirectorySessionWaitForLoadedResultReason.OpenResult_Disposed}.
     * * Finally, If a popup authentication procedure is running, but the user logged off : {@link AsyncDirectorySessionWaitForLoadedResultReason.OpenResult_LoggedOff}.
     *
     * @return A promise.
     */
    asyncWaitForPopupAuthentication(): Promise<AsyncDirectorySessionWaitForLoadedResult>;
    /**
     * Opens a new authentication session on the directory with the same page mechanism, asking the user to authenticate.
     *
     * Depending on your openid connect configuration, this may be a silent procedure for the user.
     *
     * This function may return a string if successful, {@link AuthenticationGetURLResult.DoNotReAuthenticate} if the {@link DirectorySessionInterface}
     * is already connected, {@link AuthenticationGetURLResult.InvalidSession} if an internal error occurred (unlikely) and {@link AuthenticationGetURLResult.EmptyCallbackURI}
     * if pRedirectURI is an empty string.
     *
     * ```typescript
     * [[include:api_samepage_authentication.ts]]
     * ```
     * <br>
     * See <a href="../index.html">the main page</a> for more information.
     *
     * @param pRedirectURI <div class="const">in</div> The redirect url where the authentication server should redirect on the completion of the
     * authentication procedure (successful or not). An empty string will result in an error.
     * @param pAppClaims <div class="const">in</div> An array of application claims that may be used to validate queries on third party servers. Third party servers
     * may validate some requests and look for the application claims to satisfy or reject the request.
     * @param pNeedExtendedBearer <div class="const">in</div> If set to `true`, the access token will contain additional information (subid, teams, etc ...) (defaults to false).
     * @return The authentication URL to visit to start the authentication procedure if successful, {@link AuthenticationGetURLResult.DoNotReAuthenticate} if the {@link DirectorySessionInterface}
     * is already connected, {@link AuthenticationGetURLResult.InvalidSession} if an internal error occurred and {@link AuthenticationGetURLResult.EmptyCallbackURI} if pRedirectURI is an empty string.
     */
    getSamePageAuthenticationUrl(pRedirectURI: string, pAppClaims?: Array<string> | undefined, pNeedExtendedBearer?: boolean | undefined): string | AuthenticationGetURLResult;
    /**
     * Gets the current token hash that may be used to reconnect the directory in case of a page refresh, skipping authentication.
     *
     * Such a hash may be stored within the storage api of the browser and used with {@link setTokenFromHash}.
     *
     * If the {@link DirectorySessionInterface} is not connected, an empty string is returned. If the given {@link DirectorySessionInterface} is closed
     * and the hash is used on another {@link DirectorySessionInterface}, the {@link setTokenFromHash} procedure will fail and {@link DirectorySessionInterfaceSignal.LoginFailed} will
     * be called.
     *
     * @return The current token hash.
     */
    getCurrentTokenHash(): string;
    /**
     * Decodes the hash content and sets the decoded content as session information. If the session content is valid, the life of the session is resumed. This function will
     * fire {@link DirectorySessionInterfaceSignal.LoginSuccess} or {@link DirectorySessionInterfaceSignal.LoginFailed} depending on the success of the procedure.
     *
     * Please note that time information is included in the hash, preventing it from being used after some time. The validity period is set by the 3djuump infinite directory.
     *
     * This function may return {@link AuthenticationGetURLResult.AuthenticationPending} if the authentication mechanism is pending, {@link AuthenticationGetURLResult.DoNotReAuthenticate} if the {@link DirectorySessionInterface} is already connected,
     * {@link AuthenticationGetURLResult.InvalidSession} is an internal error occurred and {@link AuthenticationGetURLResult.InvalidHash} is the hash is invalid.
     *
     * @param pHash <div class="const">in</div> The hash to decode (taken from window.location.hash or {@link getCurrentTokenHash}).
     * @return The result of the procedure {@link AuthenticationGetURLResult.AuthenticationPending} if the authentication mechanism is pending, {@link AuthenticationGetURLResult.DoNotReAuthenticate} if the {@link DirectorySessionInterface} is already connected,
     * {@link AuthenticationGetURLResult.InvalidSession} is an internal error occurred and {@link AuthenticationGetURLResult.InvalidHash} is the hash is invalid.
     */
    setTokenFromHash(pHash: string): AuthenticationGetURLResult;
    /**
     * Gets a copy of the user information (name, initials and picture URL).
     *
     * If the authentication has not been performed, all fields are set to empty strings.
     *
     * @return The {@link UserData} containing the name (string) the initials (string) and the picture url (string) of the logged-in user.
     * @see {@link UserData}
     */
    getUserInfo(): UserData;
}
export { DirectorySessionInterface };
/**
 * The DocumentIdResultInterface interface stores a correspondence `part instance id` <=> `geometric instance ids`.
 * This correspondence is contained in the result of a {@link DocumentIdConverterInterface.convert} call. Each `part instance id`
 * refers to a `part instance` that is linked to the metadata document identified by the `metadata document id` in the
 * result of a {@link DocumentIdConverterInterface.convert} call (actually a {@link SearchDocumentResultInterface}, {@link SearchDocumentResultInterface.getDocumentId}).
 * Such a `part instance id` is in the filtering context that was used when
 * calling {@link DocumentIdConverterInterface.convert}.
 *
 * `metadata document ids` are retrieved after a search request (see <a href="../index.html#search">Search</a>) from a {@link SearchDocumentResultInterface.getDocumentId} call.
 *
 * ```typescript
 * [[include:Search_example.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:Search_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Converters</div><br>
 * @see {@link DocumentIdConverterResultInterface}
 * @see {@link DocumentIdConverterInterface}
 * @category Converters
 */
interface DocumentIdResultInterface {
    /**
     * Gets the `part instance id` of the result.
     *
     * A number <=0 corresponds to an invalid id.
     *
     * @return The <code>part instance id</code> of the result.
     */
    getPartInstanceId(): number;
    /**
     * Gets the `geometric instance ids` corresponding to the `part instance id` in the context of the {@link DocumentIdConverterInterface.convert} call.
     *
     * DO NOT MODIFY this array in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The <code>geometric instance ids</code> represented by the <code>part instance id</code>.
     */
    getGeometricInstanceIds(): Uint32Array;
    /**
     * Gets the rendering `flag` of the part instance result.
     *
     * If the {@link DocumentIdConverterInterface.convert} was called with the pRetrieveStatusFlags as true,
     * then getPartInstanceFlag returns the status flag of the part instance, else 0 is returned.
     *
     * @return The {@link PartInstanceInfoStatusFlag} of the given part instance.
     */
    getPartInstanceFlag(): PartInstanceInfoStatusFlag;
}
export { DocumentIdResultInterface };
/**
 * The {@link DocumentIdConverterResultInterface} interface stores the result of a {@link DocumentIdConverterInterface.convert} call.
 *
 * It represents a list of {@link DocumentIdResultInterface} i.e. all `part instances ids` (and their corresponding `geometric instance ids`) that takes part
 * in the conversion request ({@link DocumentIdConverterInterface.convert}) in the given filtering context.
 * This therefore lists all the `part instance ids` included in the filtering context that is linked to the given document (referenced by a `metadata document id`).
 * The result contains also the union of all the `geometric instance ids` contained in the list of {@link DocumentIdResultInterface} in order to avoid another conversion
 * call.
 *
 * `metadata document ids` are retrieved after a search request (see <a href="../index.html#search">Search</a>), from a {@link SearchDocumentResultInterface.getDocumentId} call.
 *
 * ```typescript
 * [[include:Search_example.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:Search_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Converters</div><br>
 * @see {@link DocumentIdConverterResultInterface}
 * @see {@link DocumentIdConverterInterface}
 * @see {@link DocumentIdResultInterface}
 * @category Converters
 */
interface DocumentIdConverterResultInterface {
    /**
     * Gets the result of {@link DocumentIdConverterInterface.convert} in a list of {@link DocumentIdResultInterface}.
     *
     * DO NOT MODIFY this array in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of <code>part instance ids</code> (and their corresponding <code>geometric instance ids</code>).
     */
    getConvertedInstances(): Array<DocumentIdResultInterface>;
    /**
     * Gets the result of {@link DocumentIdConverterInterface.convert} in a list of `geometric instance ids`.
     *
     * This list is the union of all the `geometric instance ids` that are contained in the {@link getConvertedInstances} result.
     *
     * DO NOT MODIFY this array in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of all the <code>geometric instance ids</code> concerned by the {@link DocumentIdConverterInterface.convert} call.
     */
    getAllGeometricInstanceIds(): Uint32Array;
}
export { DocumentIdConverterResultInterface };
/**
 * The {@link DocumentIdConverterInterfaceSignal} lists all the signals that may be sent by the {@link DocumentIdConverterInterface}.
 *
 * These signals are emitted by the {@link DocumentIdConverterInterface} when some data is available.
 *
 * * {@link DocumentIdConverterReady} : the {@link DocumentIdConverterInterface} is ready, i.e. when it has
 * finished getting the `part instance ids` that contain the given document, {@link DocumentIdConverterInterface.getConversionResult} can be called.
 * * {@link DocumentIdConverterCancelled} : the retrieving of the content by the {@link DocumentIdConverterInterface} has been cancelled.
 *
 * ```typescript
 * [[include:Search_example.ts]]
 * ```
 * <br>
 * Please refer to {@link DocumentIdConverterInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link DocumentIdConverterInterface}
 * @category Events
 */
declare enum DocumentIdConverterInterfaceSignal {
    /**
     * Signal sent by the {@link DocumentIdConverterInterface} when {@link DocumentIdConverterInterface.convert} is ready, i.e.
     * {@link DocumentIdConverterInterface.getConversionResult} can be called.
     *
     * Such a signal does not tell if the call was successful or not but that the result is ready to be fetched.
     *
     * Warning, the DocumentIdConverterReady signal is sent later when {@link DocumentIdConverterInterface.cancel} is called and the {@link DocumentIdConverterInterface} is fetching data.
     *
     * The attachment is a string telling the request id of the call to {@link DocumentIdConverterInterface.convert} ({@link DocumentIdConverterInterface.getLastRequestId}).
     */
    DocumentIdConverterReady = "ready",
    /**
     * Signal sent by the {@link DocumentIdConverterInterface} when the {@link DocumentIdConverterInterface} has been cancelled.
     *
     * Such an event is fired if the {@link DocumentIdConverterInterface} was  fetching data and {@link DocumentIdConverterInterface.cancel} is called.
     * The {@link DocumentIdConverterReady} signal will be triggered shortly after.
     *
     * No attachment for events of this type.
     */
    DocumentIdConverterCancelled = "cancelled"
}
export { DocumentIdConverterInterfaceSignal };
/**
 * Represents a document id conversion promise result.
 *
 * [[include:constStyle.md]]<br><div class="const">Converters</div><br>
 * @see {@link DocumentIdConverterInterface.asyncConvert}
 * @category Converters
 */
interface AsyncDocumentIdConverterResultInterfaceResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     */
    value?: DocumentIdConverterResultInterface;
}
export { AsyncDocumentIdConverterResultInterfaceResult };
/**
 * The DocumentIdConverterInterface interface is used to get the `part instance ids` and `geometric instance ids` that are linked to
 * a given document (referenced by its `metadata document id`).
 *
 * The `metadata document id` is retrieved from a {@link SearchDocumentResultInterface} after a successful call to a search request
 * (see <a href="../index.html#search">Search</a>).
 *
 * The DocumentIdConverterInterface interfaces are created through the {@link DataSessionInterface.createDocumentIdConverter} method.
 *
 * The list of signals the DocumentIdConverterInterface may trigger is available in the {@link DocumentIdConverterInterfaceSignal} enumeration.
 *
 * The conversion mechanism is triggered through the {@link convert} method. The result is not available right away, but the
 * event {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterReady} is triggered when the result of the DocumentIdConverterInterface is available.
 * The result is available through the {@link getConversionResult} function.
 *
 * Warning : there may be cases when the {@link getConversionResult} is not available such as when the DocumentIdConverterInterface is fetching data, i.e. when
 * {@link isRunning} returns true, or when the DocumentIdConverterInterface has been cancelled, i.e. when {@link isCancelled} returns true.
 *
 * A DocumentIdConverterInterface may be interrupted (cancelled) when the DocumentIdConverterInterface is fetching data and {@link cancel}
 * is called. In such cases, the {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterCancelled} signal is fired, and shortly after,
 * {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterReady}
 * signal is fired, but {@link getConversionResult} will return undefined. Just call {@link convert} with another (or the same) `metadata document id` to trigger a new
 * retrieval.
 *
 * If you call multiple times {@link convert} before receiving the {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterReady},
 * only one {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterReady} will be sent with the content of the last call to {@link convert}.
 *
 * ```typescript
 * [[include:Search_example.ts]]
 * ```
 * <br>
 * Or asynchronously :
 *
 * ```typescript
 * [[include:Search_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Converters</div><br>
 * @see {@link DocumentIdConverterInterfaceSignal}
 * @see {@link DocumentIdConverterResultInterface}
 * @see {@link DocumentIdResultInterface}
 * @category Converters
 */
interface DocumentIdConverterInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Triggers a request to "translate" the given `metadata document id` to the corresponding couple &#123;`part instance id`,`geometric instance id`&#125;.
     *
     * The server will try to find all `part instances` that are linked to the given document in the given filtering context.
     * An event {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterReady} is fired when the translation is finished, use {@link getLastError | getLastError()} to check if it was correctly performed.
     * Due to performance limitation, if the number of `part instances` exceeded 1000, the converter return an error.
     *
     * `pRetrieveStatusFlags` may be set to true to retrieve also the status flags of the resulting part instances.
     * If omitted, `pRetrieveStatusFlags` defaults to `false`.
     *
     * Returns true if the "conversion" is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed.
     *
     * If pVisibilityContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pMetadataDocumentId <div class="const">in</div> The <code>metadata document id</code> to fetch data from.
     * @param pVisibilityContext <div class="const">in</div> The filtering context to use when "converting" to a list of <code>part instance ids</code>.
     * @param pRetrieveStatusFlags <div class="const">in</div> Optional boolean to tell to get the part instances status flags when converting document ids. Defaults to false.
     * @return `true` if the conversion process has started, just wait for {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterReady}.
     */
    convert(pMetadataDocumentId: number, pVisibilityContext: VisibilityContextInterface, pRetrieveStatusFlags?: boolean): boolean;
    /**
     * Asynchronously triggers a request to "translate" the given `metadata document id` to the corresponding couple &#123;`part instance id`,`geometric instance id`&#125;.
     *
     * The server will try to find all `part instances` that are linked to the given document in the given filtering context.
     * A promised is returned, that is either rejected in case of an error, or resolved with the corresponding DocumentIdConverterResultInterface.
     * Due to performance limitation, if the number of `part instances` exceeded 1000, the converter returns an error.
     *
     * `pRetrieveStatusFlags` may be set to true to retrieve also the status flags of the resulting part instances.
     * If omitted, `pRetrieveStatusFlags` defaults to false.
     *
     * Returns a promise.
     *
     * Please note that in case of multiple promises running at the same time on the same DocumentIdConverterInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getConversionResult}
     * after awaiting will return the same value.
     *
     * If pVisibilityContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pMetadataDocumentId <div class="const">in</div> The <code>metadata document id</code> to fetch data from.
     * @param pVisibilityContext <div class="const">in</div> The filtering context to use when "converting" to a list of <code>part instance ids</code>.
     * @param pRetrieveStatusFlags <div class="const">in</div> Optional boolean to tell to get the part instances status flags when converting document ids. Defaults to false.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the conversion result.
     */
    asyncConvert(pMetadataDocumentId: number, pVisibilityContext: VisibilityContextInterface, pRetrieveStatusFlags?: boolean): Promise<AsyncDocumentIdConverterResultInterfaceResult>;
    /**
     * Each call to {@link convert} is assigned a request id.
     *
     * This call tels the id of the last call to {@link convert}.
     *
     * @return The id of the last call to {@link convert}.
     */
    getLastRequestId(): string;
    /**
     * Cancels the computation of the conversion process (if any).
     *
     * A {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterCancelled} signal is emitted if the
     * DocumentIdConverterInterface is retrieving data.
     *
     * @return `true` if the DocumentIdConverterInterface was running, else `false`.
     * @see {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterCancelled}
     */
    cancel(): boolean;
    /**
     * Tells if the DocumentIdConverterInterface is converting data.
     *
     * This is the case after calling {@link convert}.
     *
     * @return `true` if the DocumentIdConverterInterface is converting.
     */
    isRunning(): boolean;
    /**
     * Tells if the DocumentIdConverterInterface has been cancelled.
     *
     * This is generally the case after calling {@link cancel} when the DocumentIdConverterInterface
     * is retrieving data.
     *
     * @return `true` if the DocumentIdConverterInterface is cancelled.
     */
    isCancelled(): boolean;
    /**
     * Gets the last error returned by the {@link convert} call of the DocumentIdConverterInterface.
     *
     * @return The last error message (if any, or an empty string if no error occurred).
     */
    getLastError(): string;
    /**
     * Gets the result of the last {@link convert} call of the `metadata document id`.
     *
     * A {@link DocumentIdConverterResultInterface} is returned if the DocumentIdConverterInterface has finished computing.
     * Use {@link addEventListener} on the event {@link DocumentIdConverterInterfaceSignal.DocumentIdConverterReady} to
     * know when the DocumentIdConverterInterface is ready.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The result of the conversion as a {@link DocumentIdConverterResultInterface}, or
     * undefined if the DocumentIdConverterInterface is converting or if the DocumentIdConverterInterface is in error or cancelled.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     */
    getConversionResult(): DocumentIdConverterResultInterface | undefined;
}
export { DocumentIdConverterInterface };
/**
 * The PickingAttachmentItem interface defines some of the result of a pick request.
 *
 * This interface defines the given geometry/box/line/point picked, its closest pick point and the number of pixels inside the selection
 * (if rectangular) that hit the geometry/box/line/point.
 * It is included in the {@link PickingAttachment} interface of the {@link InfiniteEngineInterfaceSignal.Picked} signal event.
 * It is obtained from a {@link InfiniteEngineInterface.pickAt}, {@link InfiniteEngineInterface.pickRect} or {@link InfiniteEngineInterface.pickFromRay} calls.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_picking.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_asyncPicking.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link PickingAttachment}
 * @see {@link InfiniteEngineInterface}
 * @see {@link InfiniteEngineInterfaceSignal}
 * @category Events
 */
interface PickingAttachmentItem {
    /**
     * The `geometric instance id` / `instance id` of the picked element (geometry/box/line/point).
     */
    instanceId: number;
    /**
     * The closest 3D position of the picked element.
     *
     * Indeed, in case of a rectangle pick, multiple pixels may pick the same element (geometry/box/line/point).
     * The position is the one of the closest pixel.
     */
    position: Vector3;
    /**
     * The number of pixels which hit the given element (geometry/box/line/point).
     *
     * In case of single pick, this is `1`, this may be more in case of a rectangular pick.
     */
    hits: number;
}
export { PickingAttachmentItem };
/**
 * The picking result origin as called from {@link InfiniteEngineInterface.pickAt} or {@link InfiniteEngineInterface.pickRect}.
 *
 * The picking result origin is included in the {@link PickingAttachment}.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @category Events
 */
declare enum PickingAttachmentOrigin {
    /**
     * The picking was triggered from {@link InfiniteEngineInterface.pickAt}.
     */
    PAO_PickAt = 0,
    /**
     * The picking was triggered from {@link InfiniteEngineInterface.pickRect}.
     */
    PAO_PickRect = 1,
    /**
     * The picking was triggered from {@link InfiniteEngineInterface.pickFromRay}.
     */
    PAO_PickFromRay = 2,
    /**
     * The picking is invalid (should not happen).
     */
    PAO_Invalid = -1
}
export { PickingAttachmentOrigin };
/**
 * The PickingAttachment interface defines the result of a pick request.
 *
 * This is the actual attachment of a {@link InfiniteEngineInterfaceSignal.Picked} signal event.
 *
 * A pick request aks to know the content (i.e. geometries, lines, boxes, points) at
 * specific positions in the 3D display. The picking request may concern a specific position ({@link InfiniteEngineInterface.pickAt})
 * or a rectangular area ({@link InfiniteEngineInterface.pickRect}) inside the 3d view.
 *
 * The actual picked point follows the following convention :
 * <br>
 * <img src="media://picking.svg" alt="picking convention">
 * <br>
 * As multiple elements may be picked, this interface gathers a list of {@link PickingAttachmentItem} for geometries, a list of {@link PickingAttachmentItem} for points,
 * a list of {@link PickingAttachmentItem} for lines and a list of {@link PickingAttachmentItem} for boxes.
 *
 * Warning : this is a `geometric instance id` and not a `part instance id` !!! (see <a href="../index.html#main-id-types">Main ID Types</a> for more information).
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_picking.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_asyncPicking.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link PickingAttachmentItem}
 * @see {@link InfiniteEngineInterface}
 * @see {@link InfiniteEngineInterfaceSignal}
 * @category Events
 */
interface PickingAttachment {
    /**
     * The picking request id as returned by {@link InfiniteEngineInterface.getLastPickingRequestId}.
     */
    pickingRequestId: string;
    /**
     * The picking origin type (pickAt, pickRect, pickFromRay).
     */
    pickingOrigin: PickingAttachmentOrigin;
    /**
     * The picking origin data (pickAt is a Vector2, pickRect is a Rectangle, pickFromRay a Ray3).
     */
    pickingOriginData: Vector2 | Rectangle | Ray3;
    /**
     * The list of geometries that were picked, or undefined if no geometry was picked.
     */
    geometric: Array<PickingAttachmentItem> | undefined;
    /**
     * The list of points that were picked, or undefined if no point was picked.
     */
    point: Array<PickingAttachmentItem> | undefined;
    /**
     * The list of lines that were picked, or undefined if no line was picked.
     */
    line: Array<PickingAttachmentItem> | undefined;
    /**
     * The list of boxes that were picked, or undefined if no box was picked.
     */
    box: Array<PickingAttachmentItem> | undefined;
    /**
     * The list of annotations that were picked, or undefined if no annotation was picked.
     */
    annotation: Array<PickingAttachmentItem> | undefined;
}
export { PickingAttachment };
/**
 * The InfiniteEvent interface is the base interface of all the events of the 3djuump infinite signal system.
 *
 * The 3djuump Infinite API uses a system very similar to DOM events with the {@link EventDispatcherInterface.addEventListener}
 * and {@link EventDispatcherInterface.removeEventListener} functions (observer pattern).
 *
 * An event stores the type (name of event), the object the event fired on (the "emitter") and an optional attachment depending
 * on the type of the event. Each interface that may send signals (i.e. extends {@link EventDispatcherInterface}) will send events
 * with the same name of the interface appended with `Signal` e.g. `CameraManagerInterface` may trigger a `CameraManagerInterfaceSignal` event.
 *
 * **WARNING** : NO NOT store events internally, but make a deep copy of the relevant data since events may be reused internally by the library.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @category Events
 */
interface InfiniteEvent {
    /**
     * The object the event fired on.
     */
    emitter: Object;
    /**
     * The type of the event.
     */
    type: string;
    /**
     * Additional info of the event. Depends on the type of the event.
     */
    attachments?: PickingAttachment | any;
}
export { InfiniteEvent };
/**
 * The tListenerCallback type is the type of a callback that is called when an event is fired.
 *
 * When event are fired, registered callbacks are called. Callbacks are installed on objects implementing the {@link EventDispatcherInterface}.
 *
 * ```typescript
 * [[include:EventDispatcherInterface_listener.ts]]
 * ```
 * <br>
 * The syntax is very close to the DOM addEventlistener functions.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link EventDispatcherInterface}
 * @category Events
 */
type tListenerCallback = 
/**
 * @param pEvent <div class="const">in</div> The event that fired.
 * @param pCallbackData <div class="const">in</div> The optional callback data that was registered when {@link EventDispatcherInterface.addEventListener} was called.
 */
(pEvent: InfiniteEvent, pCallbackData: Object | undefined) => void;
export { tListenerCallback };
/**
 * The EventDispatcherInterface interface is the 3djuump infinite API implementation of the observer pattern, very similar to the DOM event system of
 * javascript.
 *
 * Objects implementing this interface can install/remove listeners to be notified when some
 * events fire with the use of the {@link addEventListener} and {@link removeEventListener} functions.
 *
 * ```typescript
 * [[include:EventDispatcherInterface_listener.ts]]
 * ```
 * <br>
 * The syntax is very close to the DOM addEventlistener functions.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @category Events
 */
interface EventDispatcherInterface {
    /**
     * Adds a listener to an event type.
     *
     * When an event of the type `pType` fires, the callback pListener will be called. This function
     * returns a unique string id that may be used in {@link removeEventListenerById} to allow simple
     * listener removal.
     * It is possible to add an object that will be included in the callback to avoid creating too many closures.
     *
     * @param pType <div class="const">in</div> The type of the event pListener will be called upon.
     * @param pListener <div class="const">in</div> The listener function that fires when the given event type occurs.
     * @param pObject <div class="const">in</div> The optional object the callback will be called with when the given event fires.
     * @return The id of the inserted callback (actually an UUID).
     */
    addEventListener(pType: string, pListener: tListenerCallback, pObject: Object | undefined): string;
    /**
     * Adds a listener to an event type.
     *
     * When an event of the type `pType` fires, the callback pListener will be called. This function
     * returns a unique string id that may be used in {@link removeEventListenerById} to allow simple
     * listener removal.
     *
     * @param pType <div class="const">in</div> The type of the event pListener will be called upon.
     * @param pListener <div class="const">in</div> The listener function that fires when the given event type occurs.
     * @return The id of the inserted callback (actually an UUID).
     */
    addEventListener(pType: string, pListener: tListenerCallback): string;
    /**
     * Tells if the EventDispatcher has such a callback registered for the given event type.
     *
     * @param pType <div class="const">in</div> The type of the event to test.
     * @param pListener <div class="const">in</div> The listener function that gets tested.
     * @return `true` if such a listener is installed for the given type of event.
     */
    hasEventListener(pType: string, pListener: tListenerCallback): boolean;
    /**
     * Removes a listener from an event type.
     *
     * If no such listener is found, then the function returns false and does nothing.
     * You must use the exact parameters that were used in {@link addEventListener} to actually remove the listener.
     *
     * @param pType <div class="const">in</div> The type of the listener that gets removed.
     * @param pListener The listener function that gets removed.
     * @param pObject The listener object that was used when {@link addEventListener} was called.
     * @return `true` if the callback was removed else false.
     */
    removeEventListener(pType: string, pListener: tListenerCallback, pObject: Object | undefined): boolean;
    /**
     * Removes a listener from an event type.
     *
     * If no such listener is found, then the function returns false and does nothing.
     * You must use the exact parameters that were used in {@link addEventListener} to actually remove the listener.
     *
     * @param pType <div class="const">in</div> The type of the listener that gets removed.
     * @param pListener The listener function that gets removed.
     * @return `true` if the callback was removed else false.
     */
    removeEventListener(pType: string, pListener: tListenerCallback): boolean;
    /**
     * Removes all the event listeners.
     *
     * @return `true` if all the callbacks were removed else false.
     */
    removeAllEventListeners(): boolean;
    /**
     * Removes a listener from an event type by its id.
     *
     * If no such listener is found, then the function returns false and does nothing.
     * You must use the return value of {@link addEventListener} to actually remove the listener.
     * There is no more checks on the type of the event, since ids are unique whatever the used type.
     *
     * @param pType <div class="const">in</div> The type of the listener that gets removed.
     * @param pId <div class="const">in</div> The id returned by the call to {@link addEventListener} that you want to remove.
     * @return `true` if the callback was removed else false.
     * @deprecated Use removeEventListenerById(pId: string) instead.
     * @internal
     */
    removeEventListenerById(pType: string, pId: string): boolean;
    /**
     * Removes a listener by its id.
     *
     * If no such listener is found, then the function returns false and does nothing.
     * You must use the return value of {@link addEventListener} to actually remove the listener.
     *
     * @param pId <div class="const">in</div> The id returned by the call to {@link addEventListener} that you want to remove.
     * @return `true` if the callback was removed else false.
     */
    removeEventListenerById(pId: string): boolean;
}
export { EventDispatcherInterface };
/**
 * The FilterRangeInterface interface is a {@link FilterItemInterface} to elect `part instances` that have a
 * numeric/date metadata inside or that intersects a fixed set of ranges.
 *
 * The filter range interface works on metadata of type :
 * * {@link AttributeType.ATTR_NUMBER} : number.
 * * {@link AttributeType.ATTR_DATE} : date.
 * * {@link AttributeType.ATTR_NUMBER_RANGE} : number range.
 * * {@link AttributeType.ATTR_DATE_RANGE} : date range.
 *
 * The type of the metadata is given by the {@link AttributeInfoInterface.getAttributeType} from the {@link AttributesDictionaryInterface} once the DMU is loaded.
 *
 * This interface allows to create range items {@link FilterRangeItemInterface} to allow the creation of an union of ranges, to be tested against a value or another range (be
 * it numeric or a date).
 *
 * The API only exposes numbers. Indeed, date types must be converted to a number : dates must be expressed as **number of milliseconds since January 1, 1970, 00:00:00 UTC**,
 * dates must be converted with the use of the `getTime` functions, and created or modified with the Date constructor from a number (the number of milliseconds since January 1, 1970, 00:00:00 UTC) or
 * with the `setTime` function. Negative values are accepted.
 *
 * Suppose the user wants to get all `part instances` that are created between yesterday and the current date time. The
 * relevant `part instances` have the metadata `CreationDate` with the {@link AttributeType.ATTR_DATE} type :
 *
 * ```typescript
 * [[include:FilterRangeInterface_Creation.ts]]
 * ```
 * <br>
 * This interface MUST be used in a {@link FilterSolverInterface}, which in turn should refer to a valid {@link VisibilityContextInterface} (see <a href="../index.html#filtering-context">Filtering Context</a>).
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterRangeItemInterface}
 * @category Metadata/Filters
 */
interface FilterRangeInterface extends FilterItemInterface {
    /**
     * Sets the name of the attribute to filter with.
     *
     * Using an unknown attribute name or an invalid attribute type (only {@link AttributeType.ATTR_NUMBER}, {@link AttributeType.ATTR_DATE} {@link AttributeType.ATTR_NUMBER_RANGE} and {@link AttributeType.ATTR_DATE_RANGE} types are accepted) will return
     * false and leave the FilterRangeInterface unchanged.
     *
     * Setting the same attribute name will return true and leave the FilterRangeInterface unchanged.
     *
     * Setting a valid and new attribute name will clear the ranges that may have previously created by {@link createFilterRangeItem}.
     *
     * Use {@link EventDispatcherInterface.addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged} to know when the filter data is changed.
     *
     * @param pAttributeName <div class="const">in</div> The name of the attribute to filter.
     * @return `true` if pAttributeName refers to an existing and valid attribute.
     */
    setAttributeName(pAttributeName: string): boolean;
    /**
     * Gets the name of the attribute to filter with.
     *
     * @return The name of the attribute to filter with.
     */
    getAttributeName(): string;
    /**
     * Creates and adds a new range item into the list of range items of the object.
     *
     * @return The newly created filter range item.
     * @see {@link FilterRangeItemInterface}
     */
    createFilterRangeItem(): FilterRangeItemInterface;
    /**
     * Removes the range item at the position `pRangeIndex`.
     *
     * If pRangeIndex is negative or superior or equal to the number of range items, this
     * function does nothing and returns false. Returns true if the item is effectively removed.
     *
     * @param pRangeIndex <div class="const">in</div> Index of the range item to remove.
     * @return `true` if the item was removed else return false.
     */
    removeRangeItem(pRangeIndex: number): boolean;
    /**
     * Gets the range item at index `pRangeIndex`.
     *
     * If pRangeIndex is negative or superior or equal to the number of range items, this
     * function does nothing and returns undefined.
     *
     * @param pRangeIndex <div class="const">in</div> Index of the range item to get.
     * @return The given {@link FilterRangeItemInterface} (this is the actual data held by the FilterRangeInterface, and not a copy) if pRangeIndex is valid else return undefined.
     */
    getRangeItem(pRangeIndex: number): FilterRangeItemInterface | undefined;
    /**
     * Gets the list of all range items.
     *
     * This function should not be used to add or remove range items, the FilterRangeInterface will
     * not see the modifications if the array is modified.
     *
     * However, modifying items inside the array is allowed.
     *
     * @return <div class="const">warning</div> all range items inside the FilterRangeInterface.
     */
    getRangeItems(): Array<FilterRangeItemInterface>;
}
export { FilterRangeInterface };
/**
 * The FilterDiagonalInterface interface is a {@link FilterItemInterface} to elect <code>part instances **leaves**</code>
 * that have an oriented bounding box diagonal **SQUARED** length contained in one of the range specified in the filter.
 *
 * This interface allows to create range items {@link FilterRangeItemInterface} to allow the creation of an union of ranges, to be tested against the diagonal
 * **SQUARED** length of the `part instances leaves`.
 *
 * Suppose the user wants to get all `part instances leaves` which have a diagonal **SQUARED** length contained within 2 and 3 mm.
 *
 * ```typescript
 * [[include:FilterDiagonalInterface_Creation.ts]]
 * ```
 * <br>
 * This interface MUST be used in a {@link FilterSolverInterface}, which in turn should refer to a valid {@link VisibilityContextInterface} (see <a href="../index.html#filtering-context">Filtering Context</a>).
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterRangeItemInterface}
 * @category Metadata/Filters
 */
interface FilterDiagonalInterface extends FilterItemInterface {
    /**
     * Creates and adds a new range item into the list of range items of the object.
     *
     * @return The newly created filter range item.
     * @see {@link FilterRangeItemInterface}
     */
    createFilterRangeItem(): FilterRangeItemInterface;
    /**
     * Removes the range item at the position `pRangeIndex`.
     *
     * If pRangeIndex is negative or superior or equal to the number of range items, this
     * function does nothing and returns false. Returns true if the item is effectively removed.
     *
     * @param pRangeIndex <div class="const">in</div> Index of the range item to remove.
     * @return `true` if the item was removed else return false.
     */
    removeRangeItem(pRangeIndex: number): boolean;
    /**
     * Gets the range item at index `pRangeIndex`.
     *
     * If pRangeIndex is negative or superior or equal to the number of range items, this
     * function does nothing and returns undefined.
     *
     * @param pRangeIndex <div class="const">in</div> Index of the range item to get.
     * @return The given {@link FilterRangeItemInterface} (this is the actual data held by the FilterDiagonalInterface, and not a copy) if pRangeIndex is valid else return undefined.
     */
    getRangeItem(pRangeIndex: number): FilterRangeItemInterface | undefined;
    /**
     * Gets the list of all range items.
     *
     * This function should not be used to add or remove range items, the FilterDiagonalInterface will
     * not see the modifications if the array is modified.
     *
     * However, modifying items inside the array is allowed.
     *
     * @return <div class="const">warning</div> All range items inside the FilterDiagonalInterface.
     */
    getRangeItems(): Array<FilterRangeItemInterface>;
}
export { FilterDiagonalInterface };
/**
 * The FilterMoveInfoInterface is the attachment of the {@link FilterSolverInterfaceSignal.SolverFilterMoved} signal.
 *
 * It describes the {@link FilterItemInterface} that was moved and its new position.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link FilterSolverInterfaceSignal}
 * @category Events
 */
interface FilterMoveInfoInterface {
    /**
     * The id of the {@link FilterItemInterface} that was moved.
     */
    filter: string;
    /**
     * The new position in the {@link FilterSolverInterface}.
     */
    to: number;
}
export { FilterMoveInfoInterface };
/**
 * The FilterSolverInterfaceSignal lists all the signals that may be sent by the abstract {@link FilterSolverInterface}.
 *
 * These signals are emitted by the {@link FilterSolverInterface} when it is modified and more importantly ready.
 *
 * A {@link FilterSolverInterface} is ready when it has computed the result of its set of {@link FilterItemInterface | FilterItemInterfaces}, such a result being
 * a list of `geometric instance ids` and `part instance ids`.
 *
 * The {@link FilterSolverInterface} may fire the following signals :
 * * {@link SolverReady} : the {@link FilterSolverInterface} result can be retrieved.
 * * {@link SolverChanged} : any {@link FilterItemInterface} inside the {@link FilterSolverInterface} has changed, a {@link FilterItemInterface}
 * has been added or removed, or the {@link VisibilityContextInterface} inside the {@link FilterSolverInterface} has changed.
 * * {@link SolverFilterAdded} : a {@link FilterItemInterface} has been added in the {@link FilterSolverInterface}.
 * * {@link SolverFilterRemoved} : a {@link FilterItemInterface} has been removed from the {@link FilterSolverInterface}.
 * * {@link SolverFilterMoved} : a {@link FilterItemInterface} has been moved inside the {@link FilterSolverInterface}.
 * * {@link SolverFilterBulkChanged} : multiple changes have been done at once inside the {@link FilterSolverInterface}.
 * * {@link SolverVisibilityContextChanged} : the {@link FilterSolverInterface} has been set a new [VisibilityContextInterface]].
 *
 * ```typescript
 * [[include:FilterDiagonalInterface_Creation.ts]]
 * ```
 * <br>
 * Please refer to the {@link FilterSolverInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link FilterSolverInterface}
 * @see {@link FilterItemInterface}
 * @category Events
 */
declare enum FilterSolverInterfaceSignal {
    /**
     * Signal sent by the {@link FilterSolverInterface} when the {@link FilterSolverInterface} is ready, i.e. {@link FilterSolverInterface.getGeometricInstanceIds},
     * {@link FilterSolverInterface.getPartInstanceIds} and {@link FilterSolverInterface.getTrianglesCount} can be called.
     *
     * No attachment for events of this type.
     */
    SolverReady = "ready",
    /**
     * Signal sent by the {@link FilterSolverInterface} when the {@link FilterSolverInterface} is cancelled, i.e. when the {@link FilterSolverInterface} is running and
     * the content of the solver is modified.
     *
     * No attachment for events of this type.
     */
    SolverCancelled = "cancelled",
    /**
     * Signal sent by the {@link FilterSolverInterface} when any {@link FilterItemInterface} has changed, a {@link FilterItemInterface}
     * has been added or removed, or the {@link VisibilityContextInterface} inside the {@link FilterSolverInterface} has changed.
     *
     * No attachment for events of this type.
     */
    SolverChanged = "changed",
    /**
     * Signal sent by the {@link FilterSolverInterface} when a {@link FilterItemInterface} has been added.
     *
     * The attachment is the {@link FilterItemInterface} that was added.
     */
    SolverFilterAdded = "added",
    /**
     * Signal sent by the {@link FilterSolverInterface} when a {@link FilterItemInterface} has been removed.
     *
     * The attachment is the id of {@link FilterItemInterface} that was removed.
     */
    SolverFilterRemoved = "removed",
    /**
     * Signal sent by the {@link FilterSolverInterface} when a {@link FilterItemInterface} has been moved.
     *
     * The attachment is a {@link FilterMoveInfoInterface} :
     *
     * ```json
     * {
     * 		"filter": "the id of the {@link FilterItemInterface} that changed",
     * 		"to": pOrder // the new position of the {@link FilterItemInterface} in the list of children of the container
     * }
     * ```
     * <br>
     * Please refer to {@link FilterSetInterface.moveFilter}.
     *
     * @see {@link FilterMoveInfoInterface}
     * @see {@link FilterSetInterface.moveFilter}
     */
    SolverFilterMoved = "moved",
    /**
     * Signal sent by the {@link FilterSolverInterface} when a lot of changes occur inside.
     *
     * No attachment for events of this type.
     */
    SolverFilterBulkChanged = "bulkChanged",
    /**
     * Signal sent by the {@link FilterSolverInterface} a new [VisibilityContextInterface]] has been set or cleared.
     *
     * The attachment is the {@link VisibilityContextInterface} that was set, or undefined if the {@link VisibilityContextInterface} has been cleared.
     */
    SolverVisibilityContextChanged = "visibilityContextChanged"
}
export { FilterSolverInterfaceSignal };
/**
 * The FilterSolverInterface interface is used to create sets of `part instances`.
 *
 * The {@link FilterItemInterface} serves to elect a list of `part instances` based on a criterion (or a list of criteria), for
 * example (others can be found in <a href="../index.html#filters-examples">Filters examples</a>):
 * * Find all `part instances` whose size is lower than 3 cms
 * * Find all `part instances` that overlaps a given 3D cube
 * * Find all `part instances` that have a metadata "type" equal to "electrical" (electrical `part instances`)
 * * Find all `part instances` that have a metadata "mounted" to true (mounted `part instances`).
 *
 * A FilterSolverInterface is the expression of a combination of multiple {@link FilterItemInterface | FilterItemInterfaces} with a specific operator
 * (union/intersection/exclusion), a {@link FilterOperator}.<br>
 * The order of {@link FilterItemInterface} in the FilterSolverInterface is relevant and changing the order of the {@link FilterItemInterface | FilterItemInterfaces} may (or not) change the final result.
 *
 * The {@link FilterSolverInterface} MUST be intersected with a filtering context : {@link VisibilityContextInterface}
 * (see <a href="../index.html#filtering-context">Filtering Context</a>). Such a filtering context will intersect the {@link FilterSolverInterface}
 * with a specific configuration and optionally a union of other FilterSolverInterfaces ({@link FilterSolverInterface.setVisibilityContext}).
 *
 * The FilterSolverInterface result is gotten asynchronously when triggered by a {@link DataSessionInterface.update} call.
 * The FilterSolverInterface sends the event {@link FilterSolverInterfaceSignal.SolverReady} when it the result is ready to be fetched.
 * The result of a FilterSolverInterface is represented as a set of `{geometric instance ids}` and optionally `{part instance ids}`.
 *
 * When a {@link FilterItemInterface} is modified, there is no automatic update of the content of a FilterSolverInterface : indeed, you may want to
 * modify a lot of {@link FilterItemInterface}, {@link ConfContextInterface | ConfContextInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces}
 * before getting the expected result. Moreover, {@link VisibilityContextInterface | VisibilityContextInterfaces} may depend on
 * FilterSolverInterfaces
 * that could lead to massive dependency graphs. For this reason, {@link VisibilityContextInterface | VisibilityContextInterfaces},
 * FilterSolverInterfaces
 * and {@link FilterItemInterface | FilterItemInterfaces} are not updated through a dedicated function on each interface but with the {@link DataSessionInterface.update} function.<br>
 * The dependency graph of {@link ConfContextInterface | ConfContextInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces} and {@link FilterItemInterface | FilterItemInterfaces}
 * is handled by the {@link DataSessionInterface}, this also prevent too many requests to be sent to the server for nothing.<br>
 * **To sum up** : modify your {@link FilterItemInterface | FilterItemInterfaces}, {@link ConfContextInterface | ConfContextInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces} and
 * FilterSolverInterfaces without taking into account any dependency that may have between them, and when you want the server
 * to compute the result,
 * remember to call {@link DataSessionInterface.update}.
 *
 * The FilterSolverInterface interface is created through the {@link DataSessionInterface.createFilterSolver} and is bound to this given {@link DataSessionInterface}.
 *
 * ```typescript
 * [[include:FilterSolverInterface_colorize.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:FilterSolverInterface_asyncColorize.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @see {@link FilterSolverInterfaceSignal}
 * @see {@link FilterItemInterface}
 * @category Metadata
 */
interface FilterSolverInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Gets the identifier of the FilterSolverInterface.
     *
     * @return The identifier of the FilterSolverInterface.
     * @deprecated Please use {@link getFilterSolverId}.
     */
    getFilterId(): string;
    /**
     * Gets the identifier of the FilterSolverInterface.
     *
     * @return The identifier of the FilterSolverInterface.
     */
    getFilterSolverId(): string;
    /**
     * Sets the identifier of the FilterSolverInterface.
     * Make sure the id is unique.
     * A unique FilterSolverInterface identifier is created if the identifier is not overridden.
     *
     * @param pFilterSolverId <div class="const">in</div> The new identifier of the FilterSolverInterface.
     */
    setFilterSolverId(pFilterSolverId: string): void;
    /**
     * Removes all the {@link FilterItemInterface} contained in this FilterSolverInterface.
     *
     * Use {@link addEventListener} on the event {@link FilterSolverInterfaceSignal.SolverChanged} to know
     * when a {@link FilterItemInterface} is changed.
     */
    removeAllFilters(): void;
    /**
     * Inserts a {@link FilterItemInterface} at the given position.
     *
     * Any invalid position will result in the {@link FilterItemInterface} being pushed at the end of
     * the list of {@link FilterItemInterface | FilterItemInterfaces}.
     * Use {@link addEventListener} on the event {@link FilterSolverInterfaceSignal.SolverFilterAdded} to know
     * when a {@link FilterItemInterface} is added to the FilterSolverInterface.
     *
     * Returns true if the {@link FilterItemInterface} is added, false if the {@link FilterItemInterface}
     * has been cleaned up, pOrder is not a number, or if the {@link FilterItemInterface} is already inside
     * a container.
     *
     * @param pOrder <div class="const">in</div> The position to insert the {@link FilterItemInterface} to (between <code>0</code> and <code>{@link getFilterCount | getFilterCount()}</code> included).
     * @param pFilter <div class="const">in</div> The {@link FilterItemInterface} to add.
     * @return `true` if the {@link FilterItemInterface} is added.
     */
    insertFilter(pOrder: number, pFilter: FilterItemInterface): boolean;
    /**
     * Gets a {@link FilterItemInterface} previously included by {@link insertFilter} given its identifier.
     *
     * Each {@link FilterItemInterface}, upon creation, is given a unique identifier.
     *
     * @param pFilterId <div class="const">in</div> The {@link FilterItemInterface} identifier to query for.
     * @return The {@link FilterItemInterface} with the given id, or undefined if not found.
     */
    getFilterById(pFilterId: string): FilterItemInterface | undefined;
    /**
     * Gets a {@link FilterItemInterface} previously included by {@link insertFilter} given its order in the container list.
     *
     * @param pOrder <div class="const">in</div> The order in the FilterSolverInterface of the {@link FilterItemInterface} to get.
     * @return The {@link FilterItemInterface} at the given order, or undefined if pOrder is out of range <code>[0,{@link getFilterCount | getFilterCount()}-1]</code>.
     */
    getFilterByOrder(pOrder: number): FilterItemInterface | undefined;
    /**
     * Gets the number of {@link FilterItemInterface} in the FilterSolverInterface.
     *
     * @return The {@link FilterItemInterface | FilterItemInterfaces} count.
     */
    getFilterCount(): number;
    /**
     * Removes a {@link FilterItemInterface} previously included by {@link insertFilter} given its identifier
     * from the container list.
     *
     * Use {@link addEventListener} on the event {@link FilterSolverInterfaceSignal.SolverFilterRemoved} to know
     * when a {@link FilterItemInterface} is removed.
     *
     * @param pFilterId <div class="const">in</div> The identifier of the {@link FilterItemInterface} to remove.
     * @return `true` if the {@link FilterItemInterface} was found and removed.
     */
    removeFilter(pFilterId: string): boolean;
    /**
     * Gets the order of the {@link FilterItemInterface} represented by the given identifier.
     *
     * @param pFilterId <div class="const">in</div> The identifier of the {@link FilterItemInterface} to query.
     * @return The order of the {@link FilterItemInterface} in the FilterSolverInterface, or -1 if not found.
     */
    getFilterOrder(pFilterId: string): number;
    /**
     * Changes the order of the given {@link FilterItemInterface} in the container list by its filter id.
     *
     * Orders number must be in the range of existing orders i.e. <code>[0,{@link getFilterCount | getFilterCount()}-1]</code>.
     *
     * @param pFilterId <div class="const">in</div> The identifier of the {@link FilterItemInterface} to move.
     * @param pOrder <div class="const">in</div> The new order of the {@link FilterItemInterface}.
     * @return `true` if the {@link FilterItemInterface} was found and changed.
     */
    moveFilter(pFilterId: string, pOrder: number): boolean;
    /**
     * Sets/Clears the filtering context ({@link VisibilityContextInterface}) to be intersected with.
     *
     * A valid {@link VisibilityContextInterface} is required for the FilterSolverInterface to be considered
     * valid. The {@link VisibilityContextInterface} MUST contain a valid {@link ConfContextInterface} (be it unconfigured).
     *
     * If the pFilteringContext is the same as {@link getVisibilityContext}, nothing is changed.
     *
     * Use {@link addEventListener} on the event {@link FilterSolverInterfaceSignal.SolverVisibilityContextChanged} to know
     * when a {@link VisibilityContextInterface} is changed.
     *
     * @param pFilteringContext <div class="const">in</div> The filtering context to be intersected with.
     * @return `true` if the filtering context has been set (i.e. the FilterSolverInterface has not been destroyed by
     * {@link FilterSolverInterface.dispose}, or the {@link VisibilityContextInterface.dispose} has been called on `pFilteringContext`).
     */
    setVisibilityContext(pFilteringContext: VisibilityContextInterface | undefined): boolean;
    /**
     * Gets the filtering context to be intersected with.
     *
     * A valid {@link VisibilityContextInterface} is required for the FilterSolverInterface to be considered
     * valid. The {@link VisibilityContextInterface} MUST contain a valid {@link ConfContextInterface} (be it unconfigured).
     *
     * @return The current {@link VisibilityContextInterface} to be intersected with (may be undefined).
     */
    getVisibilityContext(): VisibilityContextInterface | undefined;
    /**
     * Gets the list of all the `geometric instance ids` represented by this FilterSolverInterface.
     *
     * These `geometric instance ids` are the resulting set from the list of {@link FilterItemInterface | FilterItemInterface(s)}
     * with their operators intersected with the `geometric instance ids` represented by the {@link VisibilityContextInterface} set by
     * {@link setVisibilityContext}.<br>
     * The value is returned if the FilterSolverInterface has finished computing.<br>
     * Use {@link addEventListener} on the event {@link FilterSolverInterfaceSignal.SolverReady} to
     * know when the FilterSolverInterface is ready ({@link isRunning | isRunning()} is `false`).
     *
     * DO NOT modify this array in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of <code>geometric instances ids</code> represented by this FilterSolverInterface, or undefined if the FilterSolverInterface is computing
     * (({@link isRunning | isRunning()} is `false`)) or cancelled ({@link isCancelled | isCancelled()} is true) .
     */
    getGeometricInstanceIds(): Uint32Array | undefined;
    /**
     * Gets the list of all the `geometric Instance ids` represented by this FilterSolverInterface.
     *
     * These `geometric instance ids` are the resulting set from the list of {@link FilterItemInterface | FilterItemInterface(s)}
     * with their operators intersected with the `geometric instance ids` represented by the {@link VisibilityContextInterface} set by
     * {@link setVisibilityContext}.
     *
     * <br>if pRunUpdate is true (by default), a {@link DataSessionInterface.update} is automatically called.
     *
     * @param pRunUpdate <div class="const">in</div> If set to true, a {@link DataSessionInterface.update} is called (defaults to true).
     * @return A promise. The promise is resolved with the reason (success, cancelled, reset, bad input), or rejected in case of an
     * unexpected error. In case of success, the promise is resolved with the list of <code>geometric instances ids</code> represented by this FilterSolverInterface.
     */
    asyncGetGeometricInstanceIds(pRunUpdate?: boolean): Promise<AsyncUInt32ArrayResult>;
    /**
     * Gets the list of all the `part instance ids` of the `part instances` represented by this FilterSolverInterface.
     *
     * These `part instance ids` are the resulting set from the list of {@link FilterItemInterface | FilterItemInterface(s)}
     * with their operators intersected with the `part instance ids` represented by the {@link VisibilityContextInterface} set by
     * {@link setVisibilityContext}.<br>
     * The value is returned if the FilterSolverInterface has finished computing and if
     * {@link setRetrievePartInstanceIds | setRetrievePartInstanceIds(true)} has been called.<br>
     * Use {@link addEventListener} on the event {@link FilterSolverInterfaceSignal.SolverReady} to know when the FilterSolverInterface is ready.
     *
     * @return <div class="const">const</div> The list of <code>part instances ids</code> represented by this FilterSolverInterface,
     * or undefined if it is computing ({@link isRunning | isRunning()} is false) or cancelled ({@link isCancelled | isCancelled()} is true)
     * or if it should not retrieve <code>part instance ids</code> ({@link setRetrievePartInstanceIds | setRetrievePartInstanceIds(false)}).
     */
    getPartInstanceIds(): Uint32Array | undefined;
    /**
     * Gets the number of triangles of all the `geometric instances` represented by this FilterSolverInterface.
     *
     * This should be called after the FilterSolverInterface has finished its computation.<br>
     * Use {@link addEventListener} on the event {@link FilterSolverInterfaceSignal.SolverReady} to know when the
     * FilterSolverInterface is ready.<br>
     * getTrianglesCount should return -1 if the FilterSolverInterface is running or has never been computed.
     *
     * @return The number of triangles of the <code>geometric instances</code> of this FilterSolverInterface.
     */
    getTrianglesCount(): number;
    /**
     * Tells the FilterSolverInterface to compute the result as a list of `part instance ids`.
     *
     * You may choose not to get the `part instance ids` to save memory and network bandwidth.<br>
     * {@link isRetrievePartInstanceIds | isRetrievePartInstanceIds()} is `false` by default.
     *
     * @param pRetrievePartInstanceIds <div class="const">in</div> If set to true, <code>part instance ids</code> will also be retrieved
     * and available when {@link FilterSolverInterfaceSignal.SolverReady} is triggered.
     */
    setRetrievePartInstanceIds(pRetrievePartInstanceIds: boolean): void;
    /**
     * Tells if the FilterSolverInterface should retrieve the `part instance ids`.
     *
     * The {@link isRetrievePartInstanceIds | isRetrievePartInstanceIds()} is `false` by default.
     *
     * @return `true` if the FilterSolverInterface should also retrieve <code>part instance ids</code>.
     */
    isRetrievePartInstanceIds(): boolean;
    /**
     * Gets the last error returned by the update of the FilterSolverInterface.
     *
     * The error message is updated :
     * * if the FilterSolverInterface is modified ({@link FilterSolverInterfaceSignal.SolverChanged} signal).
     * * {@link DataSessionInterface.update} is called
     * * the {@link FilterSolverInterfaceSignal.SolverReady} signal has been triggered (({@link isRunning | isRunning()} is false).
     *
     * If the FilterSolverInterface is running, returns an empty string.
     *
     * @return A string containing the last error message (if any, or an empty string if no error occurred).
     */
    getLastError(): string;
    /**
     * Tells if the FilterSolverInterface is updating (running).
     *
     * This is the case after calling {@link DataSessionInterface.update}.
     *
     * @return `true` if the FilterSolverInterface is updating.
     * @see {@link DataSessionInterface.update}
     */
    isRunning(): boolean;
    /**
     * Tells if the FilterSolverInterface has been cancelled.
     *
     * This is the case after calling {@link DataSessionInterface.cancelUpdate} when the FilterSolverInterface
     * is updating (ie {@link isRunning | isRunning()} is true).
     * The {@link FilterSolverInterfaceSignal.SolverReady} signal will be sent shortly after {@link DataSessionInterface.cancelUpdate}
     * has been called.
     *
     * @return `true` if the FilterSolverInterface is cancelled.
     * @see {@link DataSessionInterface.cancelUpdate}
     */
    isCancelled(): boolean;
}
export { FilterSolverInterface };
/**
 * The FilterOperator enumeration tells how the{@link FilterItemInterface | FilterItemInterfaces} inside a {@link FilterSolverInterface} or a {@link FilterSetInterface} are combined.
 *
 * Usual set operators are available there, union, intersection and minus operator (exclusion).
 *
 * Operators between filters are set inside the `second` filter. The operator on the first filter
 * is always ignored but a warning os outputted if the filter operator is different that {@link FilterOperator.FO_UNION}. There are no operator precedence with the given operators, they are computed
 * from the left to the right. The only way to add parenthesis is to use a {@link FilterSetInterface} to group filters together.
 *
 * Example : creating a solver that computes the intersection of the `part instances` that have the field "CompletionStatus"
 * to "done" inside a specific box :
 *
 * ```typescript
 * [[include:FilterSolverInterface_createAABBandAttributeFilters.ts]]
 * ```
 * <br>
 * Please refer to the {@link FilterItemInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSetInterface}
 * @category Metadata/Filters
 */
declare enum FilterOperator {
    /**
     * The {@link FilterItemInterface | FilterItemInterfaces} are combined in unions.
     */
    FO_UNION = 0,
    /**
     * The {@link FilterItemInterface | FilterItemInterfaces} are combined in intersection.
     */
    FO_INTERSECTION = 1,
    /**
     * The {@link FilterItemInterface | FilterItemInterfaces} are combined in exclusion.
     */
    FO_EXCLUSION = 2
}
export { FilterOperator };
/**
 * The FilterType enumeration tells the available types of {@link FilterItemInterface}.
 *
 * It is the value obtained from {@link FilterItemInterface.getFilterType}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @category Metadata/Filters
 */
declare enum FilterType {
    /**
     * An invalid filter type.
     */
    FT_INVALID = -1,
    /**
     * A {@link FilterAllPartsInterface} type (all `part instances`).
     *
     * This metadata filter retrieves all the `part instances` of the DMU (nodes and leaves).
     *
     * @see {@link FilterAllPartsInterface}
     */
    FT_ALL = 0,
    /**
     * A {@link FilterAttributeInterface} type (attribute value).
     *
     * A {@link FilterAttributeInterface} is a filter that allows to get `part instances` that have a given attribute
     * in the following values : be them full text or exact.
     *
     * @see {@link FilterAttributeInterface}
     */
    FT_ATTRIBUTE = 1,
    /**
     * A {@link FilterPartInstanceListInterface} type (`part instances` list).
     *
     * A {@link FilterPartInstanceListInterface} is a non re-playable filter. This filter is used to get all the
     * `part instances` that have the given `part instance ids`. This filter is specific to a
     * 3djuump build. Any new build of a given project will trigger a full recomputation of all the ids, thus
     * invalidating the existing {@link FilterPartInstanceListInterface}.
     *
     * @see {@link FilterPartInstanceListInterface}
     */
    FT_PARTINSTANCE = 2,
    /**
     * A {@link FilterAABBInterface} type (axis aligned bounding box).
     *
     * An AABB filter is used to get all the `part instances leaves` that are included
     * in a given box ({@link AABB}).
     *
     * @see {@link FilterAABBInterface}
     * @see {@link AABB}
     */
    FT_AABB = 3,
    /**
     * A {@link FilterRangeInterface} type (numeric / date / numeric range / date range).
     *
     * A {@link FilterRangeInterface} is a metadata filter used to get all `part instances` that have a numeric metadata (number or date, ranges or not), that intersects
     * the union of ranges ({@link FilterRangeItemInterface}) inside the given {@link FilterRangeInterface}.
     *
     * @see {@link FilterRangeInterface}
     * @see {@link FilterRangeItemInterface}
     */
    FT_RANGE = 4,
    /**
     * A {@link FilterLiteralInterface} type (literal query in the
     * <a href="../index.html#3djuump-infinite-literal-and-search-query-language">3djuump infinite literal and search
     * query language</a>).
     *
     * A {@link FilterLiteralInterface} is a metadata filter defined by a character string that is the expression of a set of clauses with operators.
     *
     * @see {@link FilterLiteralInterface}
     */
    FT_LITERAL = 5,
    /**
     * A {@link FilterHasFieldInterface} type (has attribute name).
     *
     * A {@link FilterHasFieldInterface} is a metadata filter used to get all `part instances` that have a metadata with the given name regardless of its metadata
     * type.
     *
     * @see {@link FilterHasFieldInterface}
     */
    FT_HASFIELD = 6,
    /**
     * A {@link FilterBooleanInterface} type (boolean).
     *
     * A {@link FilterBooleanInterface} is a metadata filter used to get all `part instances` that have a boolean metadata
     * with a given value (true or false).
     *
     * @see {@link FilterBooleanInterface}
     */
    FT_BOOLEAN = 7,
    /**
     * A {@link FilterSetInterface} type (group).
     *
     * A {@link FilterSetInterface} allows to gather filters ({@link FilterItemInterface}) together and to create an operator precedence with {@link FilterOperator}.
     * It is a sort of "parenthesis" filter.
     *
     * @see {@link FilterSetInterface}
     */
    FT_SET = 8,
    /**
     * A {@link FilterCompoundInterface} type (compound).
     *
     * A {@link FilterCompoundInterface} is a combination of metadata filters (({@link FilterAttributeInterface}, {@link FilterRangeInterface},
     * {@link FilterHasFieldInterface}, {@link FilterBooleanInterface})) that works on
     * individual metadata documents and not on the `joined attribute set` (see <a href="../index.html#available-filters">Available Filters</a>).
     *
     * @see {@link FilterCompoundInterface}
     */
    FT_COMPOUND = 9,
    /**
     * A {@link FilterDiagonalInterface} type (diagonal).
     *
     * A {@link FilterDiagonalInterface} is used to get all `part instances` that have a diagonal whose
     * **square** length is inside the union of ranges ({@link FilterRangeItemInterface}) inside the given {@link FilterDiagonalInterface}.
     *
     * @see {@link FilterRangeItemInterface}
     * @see {@link FilterDiagonalInterface}
     */
    FT_DIAGONAL = 10
}
export { FilterType };
/**
 * The FilterItemInterfaceSignal lists all the signals that may be sent by the abstract {@link FilterItemInterface}.
 *
 * These signals are emitted by the abstract {@link FilterItemInterface} when it is modified.
 *
 * There is no way to know when the abstract {@link FilterItemInterface} is ready : the abstract {@link FilterItemInterface}
 * is contained inside a {@link FilterSolverInterface} that notifies when it is ready.
 * As such, there is no way to get the `part instances` or `geometric instances`
 * that are included in such a FilterItemInterface.
 *
 * The {@link FilterItemInterface} may fire the following signals :
 * * {@link FilterOperatorChanged} : the {@link FilterOperator} of the {@link FilterItemInterface} ({@link FilterItemInterface.setFilterOperator}) has changed.
 * * {@link FilterInvertedChanged} : the inverted status of the {@link FilterItemInterface} ({@link FilterItemInterface.setInverted}) has been changed.
 * * {@link FilterEnabledChanged} : the enabled state of the {@link FilterItemInterface} ({@link FilterItemInterface.setEnabled}) has been changed.
 * * {@link FilterDataChanged} : the specific data of the filter that inherits {@link FilterItemInterface} has been changed
 * (e.g. {@link FilterItemInterface.setFilterData}, {@link FilterAABBInterface.setAABB}, {@link FilterAttributeInterface.setAttributeName}, etc ...).
 * * {@link FilterChanged} : the abstract {@link FilterItemInterface} has changed, other signals may be sent before (e.g. {@link FilterOperatorChanged}), in this case
 * the {@link FilterChanged} signal is always the last to be sent.
 *
 * ```typescript
 * [[include:EventDispatcherInterface_listener.ts]]
 * ```
 * <br>
 * Please refer to {@link FilterItemInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterOperator}
 * @category Events
 */
declare enum FilterItemInterfaceSignal {
    /**
     * Signal sent by the abstract {@link FilterItemInterface} when its {@link FilterOperator} has changed.
     *
     * No attachment for events of this type.
     *
     * @see {@link FilterItemInterface.setFilterOperator}
     */
    FilterOperatorChanged = "operatorChanged",
    /**
     * Signal sent by the abstract {@link FilterItemInterface} when its inverted state has changed.
     *
     * No attachment for events of this type.
     *
     * @see {@link FilterItemInterface.setInverted}
     */
    FilterInvertedChanged = "invertedChanged",
    /**
     * Signal sent by the abstract {@link FilterItemInterface} when its enabled state has changed.
     *
     * No attachment for events of this type.
     *
     * @see {@link FilterItemInterface.setInverted}
     */
    FilterEnabledChanged = "enabledChanged",
    /**
     * Signal sent by the abstract {@link FilterItemInterface} when the specific data
     * of this filter has been changed (e.g. {@link FilterItemInterface.setFilterData},
     * {@link FilterAABBInterface.setAABB}, {@link FilterAttributeInterface.setAttributeName}, etc has been called).
     *
     * No attachment for events of this type.
     */
    FilterDataChanged = "dataChanged",
    /**
     * Signal sent by the abstract {@link FilterItemInterface} when it has changed.
     *
     * Such a signal is usually sent with other FilterItemInterfaceSignals, in this case, this signal is always the last
     * to be sent.
     *
     * No attachment for events of this type.
     */
    FilterChanged = "filterChanged"
}
export { FilterItemInterfaceSignal };
/**
 * The FilterItemInterface interface is the base interface used to filter `part instances`.
 *
 * NO FilterItemInterface are created per se. Instead, interfaces that **inherit** FilterItemInterface are created.
 * The FilterItemInterface is a kind of "abstract interface".
 *
 * The FilterItemInterface serves to elect a list of `part instances` based on a criterion (or a list of criteria), for
 * example (others can be found in <a href="../index.html#filters-examples">Filters examples</a>):
 * * Find all `part instances` whose size is lower than 3 cms
 * * Find all `part instances` that overlaps a given 3D cube
 * * Find all `part instances` that have a metadata "type" equal to "electrical" (electrical `part instances`)
 * * Find all `part instances` that have a metadata "mounted" to true (mounted `part instances`).
 *
 * Each example is made possible by the creation of a specific interface that inherits FilterItemInterface.
 *
 * A FilterItemInterface is not usable as is, it must be included in a "solver" object : the {@link FilterSolverInterface}. The {@link FilterSolverInterface}
 * interface allows the computation of the FilterItemInterfaces, and allows grouping them with set operators (intersection, union, exclusion).
 * See {@link FilterSolverInterface}
 * for more details, but keep in mind that {@link FilterSolverInterface | FilterSolverInterfaces} rely on filtering contexts to work : {@link VisibilityContextInterface}
 * (see <a href="../index.html#filtering-context">Filtering Context</a>).
 *
 * When a FilterItemInterface is modified, there is no automatic update of the content of a {@link FilterSolverInterface} : indeed, you may want to
 * modify a lot of FilterItemInterface, {@link ConfContextInterface | ConfContextInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces}
 * before getting the expected result. Moreover, {@link VisibilityContextInterface | VisibilityContextInterfaces} may depend on
 * {@link FilterSolverInterface | FilterSolverInterfaces}
 * that could lead to massive dependency graphs. For this reason, {@link VisibilityContextInterface | VisibilityContextInterfaces},
 * {@link FilterSolverInterface | FilterSolverInterfaces}
 * and FilterItemInterfaces are not updated through a dedicated function on each interface but with the {@link DataSessionInterface.update} function.
 * <br>The dependency graph of {@link ConfContextInterface | ConfContextInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces} and FilterItemInterfaces
 * is handled by the {@link DataSessionInterface}, this also prevent too many requests to be sent to the server for nothing.<br>
 * **To sum up** : modify your FilterItemInterfaces, {@link ConfContextInterface | ConfContextInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces} and
 * {@link FilterSolverInterface | FilterSolverInterfaces} without taking into account any dependency that may have between them, and when you want the server
 * to compute the result,
 * remember to call {@link DataSessionInterface.update}.
 *
 * **DO NOT** create interfaces that inherits FilterItemInterface before the DMU is loaded, such filters will be silently discarded.
 *
 * **Discard** existing interfaces that inherits FilterItemInterface before loading a new {@link DataSessionInterface} ({@link FilterItemInterface.dispose}).
 *
 * The same FilterItemInterface cannot be used at the same time in multiple {@link FilterSolverInterface}, {@link FilterSetInterface}, {@link FilterCompoundInterface} : calling
 * {@link FilterSolverInterface.insertFilter}, {@link FilterSetInterface.insertFilter} or {@link FilterCompoundInterface.insertFilter}
 * on a FilterItemInterface that is already included will return false.
 * <br>**Warning** : only {@link FilterAttributeInterface}, {@link FilterRangeInterface}, {@link FilterHasFieldInterface} and {@link FilterBooleanInterface}
 * can be included in a {@link FilterCompoundInterface}.
 *
 * Each FilterItemInterface is assigned a unique identifier upon creation.
 *
 * The combination of the FilterItemInterfaces in a {@link FilterSolverInterface}, {@link FilterSetInterface}, {@link FilterCompoundInterface} may be a union, intersection,
 * or exclusion ({@link FilterOperator}).
 * Each FilterItemInterface stores a combination mode that tells its way to combine itself with the first enabled **FORMER** FilterItemInterface in the parent list.<br>
 * The FilterItemInterface operator of the first enabled FilterItemInterface in a parent list ({@link FilterSolverInterface}, {@link FilterSetInterface}, {@link FilterCompoundInterface})
 * is always ignored, but a warning is
 * outputted in the console when the {@link FilterOperator} is different that {@link FilterOperator.FO_UNION}.<br>
 * The {@link FilterSetInterface} allows to gather filters together and to create an operator precedence with {@link FilterOperator}. It is a sort of "parenthesis"
 * FilterItemInterface.
 *
 * A FilterItemInterface may be :
 * * enabled/disabled : tells if this FilterItemInterface contributes to the filtering of the enclosing [FilterSolverInterface]], {@link FilterSetInterface}, {@link FilterCompoundInterface}.
 * Warning, changing the enabled status of a FilterItemInterface without knowing
 * its successor may result in unexpected results, since the successor {@link FilterOperator} will be used with the first enabled FilterItemInterface that precedes the
 * disabled one.
 * * normal/inverted : tells if the result of the FilterItemInterface may be "inverted". When "inverted", a FilterItemInterface "elects" all other
 * `part instances` leaves minus the one in the "normal" FilterItemInterface (in a {@link FilterCompoundInterface}, the meaning of the "inverted" state is different).
 *
 * The content of a FilterItemInterface is provided by {@link getFilterData} and can be restored by {@link setFilterData}.
 *
 * <br>Each filter has a `depth` contribution. For implementation reasons, the depth contribution of each FilterItemInterface included
 * inside a {@link FilterSolverInterface} is limited. The depth contribution is available through {@link getDepthContribution}.
 *
 * This `depth` contribution cannot be more than
 * @maxDepthContribution
 * Please see <a href="../index.html#available-filters">Available Filters</a> for a list of available FilterItemInterfaces.
 *
 * ```typescript
 * [[include:FilterSolverInterface_createAABBandAttributeFilters.ts]]
 * ```
 * <br>
 * Filters are created by the {@link DataSessionInterface}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterfaceSignal}
 * @see {@link FilterType}
 * @see {@link FilterOperator}
 * @category Metadata/Filters
 */
interface FilterItemInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Gets the identifier of the FilterItemInterface.
     *
     * @return The identifier of the FilterItemInterface.
     */
    getFilterId(): string;
    /**
     * Sets the identifier of the FilterItemInterface.
     * Make sure the id is unique.
     * A unique FilterItemInterface identifier is created if the identifier is not overridden.
     *
     * @param pFilterId <div class="const">in</div> The new identifier of the FilterItemInterface.
     */
    setFilterId(pFilterId: string): void;
    /**
     * Gets the identifier of the parent container that contains this FilterItemInterface (getFilterId()).
     *
     * Such a container may be a {@link FilterSolverInterface}, {@link FilterSetInterface} or a {@link FilterCompoundInterface}.
     * Returns an empty string if the FilterItemInterface is not included in a parent container.
     *
     * @return The identifier of the parent container.
     */
    getParentFilterId(): string;
    /**
     * Tells if the FilterItemInterface is enabled.
     *
     * If disabled, this FilterItemInterface is completely ignored during all the computations
     * (the behavior is the same as if it had not been created).
     *
     * A FilterItemInterface is enabled by default.
     *
     * @return `true` if the FilterItemInterface is enabled.
     */
    isEnabled(): boolean;
    /**
     * Sets the FilterItemInterface enabled/disabled status.
     *
     * If disabled, this FilterItemInterface is completely ignored during all the computations
     * (the behavior is the same as if it had not been created).
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterEnabledChanged}
     * to know when the FilterItemInterface has changed its enabled status.
     *
     * A FilterItemInterface is enabled by default.
     *
     * @param pEnabled <div class="const">in</div> If true, the given FilterItemInterface is enabled.
     */
    setEnabled(pEnabled: boolean): void;
    /**
     * Tells if the FilterItemInterface is "inverted".
     *
     * When "inverted", a FilterItemInterface elects all the `part instances` that were not selected if it was not inverted.
     *
     * A FilterItemInterface is not "inverted" by default.
     *
     * @return `true` if such a FilterItemInterface is "inverted".
     */
    isInverted(): boolean;
    /**
     * Sets the "inverted" status of the FilterItemInterface.
     *
     * When "inverted", a FilterItemInterface elects all the `part instances` that were not selected if it was not inverted.
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterInvertedChanged}
     * to know when the FilterItemInterface has changed its "inverted" status.
     *
     * A FilterItemInterface is not "inverted" by default.
     *
     * @param pInverted <div class="const">in</div> If true, such a FilterItemInterface will be "inverted".
     */
    setInverted(pInverted: boolean): void;
    /**
     * Sets the operator to apply with this FilterItemInterface and its closest enabled predecessor in its parent container.
     *
     * The FilterItemInterface operator of the first enabled FilterItemInterface in a parent list ({@link FilterSolverInterface},
     * {@link FilterSetInterface}, {@link FilterCompoundInterface})
     * is always ignored, but a warning is
     * outputted in the console when the {@link FilterOperator} is different that {@link FilterOperator.FO_UNION}.
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterInvertedChanged}  to know when the FilterItemInterface operator changed.
     *
     * A FilterItemInterface has the {@link FilterOperator.FO_UNION} operator by default.
     *
     * @param pFilterOperator <div class="const">in</div> The operator to combine this FilterItemInterface and its first enabled predecessor.
     * @see {@link FilterOperator}
     */
    setFilterOperator(pFilterOperator: FilterOperator): void;
    /**
     * Gets the operator to apply with this FilterItemInterface and its closest enabled predecessor in its parent container.
     *
     * The FilterItemInterface operator of the first enabled FilterItemInterface in a parent list ({@link FilterSolverInterface},
     * {@link FilterSetInterface}, {@link FilterCompoundInterface})
     * is always ignored, but a warning is
     * outputted in the console when the {@link FilterOperator} is different that {@link FilterOperator.FO_UNION}.
     *
     * A FilterItemInterface has the {@link FilterOperator.FO_UNION} operator by default.
     *
     * @return The operator to combine this FilterItemInterface and its first enabled predecessor.
     * @see {@link FilterOperator}
     */
    getFilterOperator(): FilterOperator;
    /**
     * Gets the type of the FilterItemInterface.
     *
     * @return The FilterItemInterface type.
     * @see {@link FilterType}
     */
    getFilterType(): FilterType;
    /**
     * Gets a deep copy of the internal data of the FilterItemInterface.
     *
     * @return The internal FilterItemInterface data.
     */
    getFilterData(): any;
    /**
     * Sets the content of the FilterItemInterface from a former call to {@link getFilterData}.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged} to know when the FilterItemInterface
     * internal data changed.
     *
     * @param pFilterData <div class="const">in</div> Internal FilterItemInterface data to set.
     * @return `true` if the data is set.
     */
    setFilterData(pFilterData: any): boolean;
    /**
     * Gets the depth contribution of the FilterItemInterface.
     *
     * This value is usually one.
     *
     * @return The depth contribution of the FilterItemInterface.
     */
    getDepthContribution(): number;
}
export { FilterItemInterface };
/**
 * The FilterAllPartsInterface interface is the easiest {@link FilterItemInterface}.
 *
 * This {@link FilterItemInterface} elects all the `part instances` of the DMU. A {@link FilterSolverInterface} that contains only a {@link FilterAllPartsInterface} will contain
 * all `part instance ids` between 1 and {@link DataSessionInterface.getMaximumPartInstanceId} included (thus a length of
 * {@link DataSessionInterface.getMaximumPartInstanceId}) and all `geometric instance ids` between 1 and {@link DataSessionInterface.getMaximumGeometricId}
 * included (thus a length of {@link DataSessionInterface.getMaximumGeometricId}).
 *
 * The FilterAllPartsInterface interface is created through the {@link DataSessionInterface.createFilterAllParts}.
 *
 * The FilterAllPartsInterface has the type {@link FilterType.FT_ALL} ({@link FilterItemInterface.getFilterType}).
 *
 * The FilterAllPartsInterface has a depth contribution of 1.
 *
 * ```typescript
 * [[include:FilterAllPartsInterface_example.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSolverInterface}
 * @see {@link FilterItemInterfaceSignal}
 * @category Metadata/Filters
 */
interface FilterAllPartsInterface extends FilterItemInterface {
}
export { FilterAllPartsInterface };
/**
 * The FilterAABBInterface interface is a {@link FilterItemInterface} to elect `part instances` intersecting an axis aligned box.
 *
 * The inclusion algorithm may be strict, i.e. all triangles
 * of a `part instance` must be strictly contained inside the box for the `part instance` to be elected, or loose, in this case, any
 * `part instance` with a triangle inside
 * the box or intersecting the box will be elected ({@link setOverlapped}).
 *
 * Upon creation a FilterAABBInterface is set to elect all `part instances` with a strict policy, i.e. all `part instances` having all triangles strictly inside the
 * box will be elected on a unit box centered on (0,0,0).
 *
 * The FilterAABBInterface interface is created through the {@link DataSessionInterface.createFilterAABB}.
 *
 * The FilterAABBInterface has the type {@link FilterType.FT_AABB} ({@link FilterItemInterface.getFilterType}).
 *
 * The FilterAABBInterface has a depth contribution of 1.
 *
 * ```typescript
 * [[include:FilterSolverInterface_createAABBandAttributeFilters.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSolverInterface}
 * @see {@link FilterItemInterfaceSignal}
 * @category Metadata/Filters
 */
interface FilterAABBInterface extends FilterItemInterface {
    /**
     * Sets the axis align bounding box.
     *
     * The FilterAABBInterface stores a copy of the data, the caller may change the box later
     * without worrying about the FilterAABBInterface being modified.
     *
     * Does nothing if the AABB is left unchanged.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged} to know when the FilterAABBInterface
     * data changed.
     *
     * A FilterAABBInterface has a unit box centered on (0,0,0) by default.
     *
     * Returns true if pAABB is set.
     *
     * @param pAABB <div class="const">in</div> The axis aligned box to set.
     * @return `true` if the AABB is set.
     */
    setAABB(pAABB: AABB): boolean;
    /**
     * Gets the axis align bounding box parameters.
     *
     * `pAABBOut` will be set with the content of the box.
     *
     * @param pAABBOut <div class="const">out</div> Upon completion, stores the current AABB of the FilterAABBInterface.
     */
    getAABB(pAABBOut: AABB): void;
    /**
     * Sets the loose/strict including policy.
     *
     * If overlapped (loose), any `part instance` with at least one triangle overlapping the box will be elected,
     * else any `part instance` should have all its triangles strictly included inside the box to be elected (strict).
     *
     * Upon creation, the FilterAABBInterface
     * is set to include all `part instances` strictly inside the box (false as default).
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged} to know when
     * the FilterAABBInterface data changed.
     *
     * @param pIncludeOverlappingTriangle <div class="const">in</div> If set, any <code>part instance</code> with at least one triangle overlapping
     * the box will be elected.
     */
    setOverlapped(pIncludeOverlappingTriangle: boolean): void;
    /**
     * Tells about the loose/strict including policy.
     *
     * If overlapped (loose), any `part instance` with at least one triangle overlapping the box will be elected,
     * else any `part instance` should have all its triangles strictly included inside the box to be elected (strict).
     *
     * Upon creation, the FilterAABBInterface
     * is set to include all `part instances` strictly inside the box (false as default).
     *
     * @return `true` if the FilterAABBInterface including policy is loose.
     */
    isOverlapped(): boolean;
}
export { FilterAABBInterface };
/**
 * The FilterPartInstanceListInterface interface is a {@link FilterItemInterface} to elect `part instances`
 * from their `part instance ids`.
 *
 * The `part instances` are known by their `part instance ids` (these are **NOT** their `geometric instance ids`).
 *
 * Such a {@link FilterItemInterface} may be created by first determining the `part instance ids` with a {@link FilterSolverInterface},
 * asking to retrieve their `part instance ids` ({@link FilterSolverInterface.setRetrievePartInstanceIds})
 * and then create a FilterPartInstanceListInterface with the given ids.
 *
 * **Warning** : using such a {@link FilterItemInterface} is very risky since `part instance ids` are recomputed at each new 3djuump Infinite build process.
 * That means that creating such a {@link FilterItemInterface} on a build will be invalidated if a new build is done on your project.
 * It is highly recommended to create your {@link FilterItemInterface | FilterItemInterfaces} based on geometric values and metadata rather than their
 * `part instance id`. Use extreme caution when using this type of {@link FilterItemInterface} and be sure you know what you are doing.
 * Moreover, such a {@link FilterItemInterface} may consume more memory than a combination of other {@link FilterItemInterface | FilterItemInterfaces}
 * if a large number of `part instance id`
 * is inside the FilterPartInstanceListInterface.
 *
 * The FilterPartInstanceListInterface interface is created through the {@link DataSessionInterface.createFilterPartInstanceList}.
 *
 * The FilterPartInstanceListInterface has the type {@link FilterType.FT_PARTINSTANCE} ({@link FilterItemInterface.getFilterType}).
 *
 * The FilterPartInstanceListInterface has a depth contribution of 1.
 *
 * ```typescript
 * [[include:FilterPartInstanceListInterface_example.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSolverInterface}
 * @see {@link FilterItemInterfaceSignal}
 * @category Metadata/Filters
 */
interface FilterPartInstanceListInterface extends FilterItemInterface {
    /**
     * Sets the list of `part instances` to elect.
     *
     * As a list of `part instance ids` is local to a build, and is VERY likely to change
     * between two builds of the same project, the buildid of the build to use must be set to avoid filtering on an incorrect
     * build. Using a pBuildId that does not correspond to the current open build leads to a no-op and returns false.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged} to know when
     * the FilterPartInstanceListInterface data changed.
     *
     * **Warning** : There is no validation that the input values are correct (i.e. integers strictly superior to 1 and inferior
     * to max `part instance id` {@link DataSessionInterface.getMaximumPartInstanceId}), or that the `part instances` list is the
     * same as previously. The invalid values will be discarded, and setting the same values as previously will trigger a recompute
     * on the server, and the function will return true.
     *
     * @param pPartInstanceList <div class="const">in</div> The list of <code>part instances</code> to elect.
     * @param pBuildId <div class="const">in</div> Id of the build the FilterPartInstanceListInterface belongs to.
     * @return `true` if pBuildId is equal to the 3djuump build actually open in the datasession.
     */
    setPartInstanceList(pPartInstanceList: Uint32Array | Array<number>, pBuildId: string): boolean;
    /**
     * Gets a copy of the list of `part instance ids` that were set by {@link setPartInstanceList}.
     *
     * @return The list of `part instance ids` to elect.
     */
    getPartInstanceList(): Uint32Array;
}
export { FilterPartInstanceListInterface };
/**
 * The FilterAttributeInterface interface is a {@link FilterItemInterface} to elect `part instances`
 * from their **string** attributes metadata.
 *
 * This {@link FilterItemInterface} selects `part instances` based on their string attributes or their parent string attributes (as defined in  <a href="../index.html#documents">Documents</a>).
 * Each `part instance` is considered to have the union between their own attributes and the attributes of its genealogy (parent, grand-parent, etc ...) :
 * the `joined attribute set`.
 *
 * The list of attributes and their types is available through the {@link AttributesDictionaryInterface.getDictionary} function.
 *
 * The FilterAttributeInterface defines 2 sets of string values : two `search attribute sets`:
 * * exact `search attribute set`.
 * * full text `search attribute set`.
 *
 * This {@link FilterItemInterface} selects `part instances` having a string attribute inside their `joined attribute set` whose value :
 * * {@link setExactValues} : is exactly inside the exact `search attribute set` (an item in the exact `search attribute set` is exactly the value of the attribute).
 * * or {@link setContainsValues} : contains at least one of the values (full text search) of the full text `search attribute set`.
 *
 * If the given attribute has only a limited number of possible values across the DMU (less than 1024), and its values are not too large,
 * then such a string attribute is `enumerable` (see {@link AttributeInfoInterface.isEnumerable}).
 * If an attribute is `enumerable`, then the special value `N/A` (not applicable) is available and selected as default
 * ({@link hasNaValueAvailable}, {@link setNaValueActivated}, {@link setNaValueChecked}). The special value `N/A` selects all other `part instances`
 * than the union of all possible values of such a string attribute. Due to this special handling and implementation details, using the `N/A` value
 * is not compatible with using the {@link setContainsValues} function. Moreover, the N/A value is not available if the FilterAttributeInterface is used
 * inside a {@link FilterCompoundInterface}.
 * Any FilterAttributeInterface on an `enumerable` attribute has {@link hasNaValueAvailable} `true`. The {@link hasNaValueAvailable} is updated only on a
 * {@link setAttributeName} call.<br>
 * The FilterAttributeInterface has also a boolean property telling if the `N/A` value should be handled ({@link setNaValueActivated}, {@link hasNaValueActivated})
 * in order to use or discard the `N/A` value if the given attribute is `enumerable`.
 * The {@link hasNaValueActivated} boolean property is not changed when {@link setAttributeName} is called. The {@link hasNaValueActivated} is true as default.<br>
 * If you want to use the {@link FilterAttributeInterface.setContainsValues} function, you will have to make a <code>{@link setNaValueActivated | setNaValueActivated(false);}</code> call.<br>
 * Finally, the FilterAttributeInterface has an internal property telling to use the `N/A` value if it is available ({@link setNaValueChecked}, {@link isNaValueChecked}),
 * this internal property is only changed on a {@link setNaValueChecked} call, but {@link isNaValueChecked} will return true only if
 * {@link hasNaValueActivated}, {@link hasNaValueAvailable} and this internal property are true. This internal property is true as default.
 *
 * In the case of a {@link setContainsValues} call, the `search attribute set` is a full text search.
 * It is case insensitive and allows the use of wildcards symbols :
 * * `?` replaces one and only one character,
 * * `*` replaces any number of characters (by default, the engine prefixes and postfixes the search with `*`)
 * * `$` prevents the default addition of the prefix or postfix `*` wildcard (if `$` is positioned at the beginning of the text, no `*` prefix added;
 * if `$` is positioned at the end, no `*` postfix added). This allows to make queries like : gets all the `part instances` whose type begins
 * by "electrical" (`$electrical`).
 * Please see  <a href="../index.html#3djuump-infinite-literal-and-search-query-language">3djuump infinite literal and search query language</a>
 * for more information about using special characters.
 *
 * The FilterAttributeInterface interface is created through the {@link DataSessionInterface.createFilterAttribute}.
 *
 * The FilterAttributeInterface has the type {@link FilterType.FT_ATTRIBUTE} ({@link FilterItemInterface.getFilterType}).
 *
 * The FilterAttributeInterface has a depth contribution of 2.
 *
 * ```typescript
 * [[include:FiltersExamples_electrical_state.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSolverInterface}
 * @see {@link FilterItemInterfaceSignal}
 * @category Metadata/Filters
 */
interface FilterAttributeInterface extends FilterItemInterface {
    /**
     * Sets the name of the attribute to filter with.
     *
     * Calling this function with the same attribute name
     * that was set previously does nothing and returns true. If the attribute name is
     * changed and valid, this function will clear any values previously set by
     * {@link setExactValues}, {@link setContainsValues}.<br>
     * Calling this function with an unknown attribute, or an attribute that does
     * not have a string value is illegal. In that case, the FilterAttributeInterface
     * is left unchanged and false is returned.
     * Please refer to the {@link AttributesDictionaryInterface.getDictionary} to get the
     * list of available attributes and their types.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterAttributeInterface data changed.
     *
     * A FilterAttributeInterface has an empty (invalid) attribute name by default.
     *
     * @param pAttributeName <div class="const">in</div> The new name of the attribute to use.
     * @return `true` if the new attribute name was set and legal.
     */
    setAttributeName(pAttributeName: string): boolean;
    /**
     * Gets the name of the attribute to filter with.
     *
     * @return The name of the attribute to use.
     */
    getAttributeName(): string;
    /**
     * Tells if the `N/A` value is available.
     *
     * The `N/A` value is available if {@link setAttributeName} is called with an
     * attribute that has {@link AttributeInfoInterface.isEnumerable} true.
     *
     * This property is updated only on a {@link setAttributeName} call.
     *
     * @return `true` is the filter has the `NA` value available.
     */
    hasNaValueAvailable(): boolean;
    /**
     * Enables/disables the `N/A` value handling if it is available.
     *
     * This boolean property tells if the `N/A` value should be handled ({@link hasNaValueActivated})
     * in order to use or discard the `N/A` value if the given attribute is `enumerable`.
     * The {@link hasNaValueActivated} boolean property is not changed when {@link setAttributeName} is called.
     *
     * The {@link hasNaValueActivated} is true as default.
     *
     * Remember that If you want to use the {@link FilterAttributeInterface.setContainsValues} function, you will have to make a
     * `{@link setNaValueActivated | setNaValueActivated(false);}` call.
     *
     * **NB** : this does not include necessarily `part instances` with a `N/A` value, {@link setNaValueChecked | setNaValueChecked(true)} should
     * be called in order to include `part instances` with the `N/A` value.
     *
     * @param pEnabled <div class="const">in</div> If true, enables the special <code>N/A</code> value handling.
     */
    setNaValueActivated(pEnabled: boolean): void;
    /**
     * Tells if the `N/A` value handling should be used if available.
     *
     * This boolean property tells if the `N/A` value should be handled
     * in order to use or discard the `N/A` value if the given attribute is `enumerable`.
     * The {@link hasNaValueActivated} boolean property is not changed when {@link setAttributeName} is called.
     *
     * The {@link hasNaValueActivated} is true as default.
     *
     * This call returns the internal property : it is legal that {@link hasNaValueAvailable} returns false
     * and hasNaValueActivated returns true.
     *
     * **NB** : this does not include necessarily `part instances` with a `N/A` value, {@link setNaValueChecked | setNaValueChecked(true)} should
     * be called in order to include `part instances` with the `N/A` value.
     *
     * @return `true` if the <code>N/A</code> value handling is activated.
     */
    hasNaValueActivated(): boolean;
    /**
     * Includes/removes the `N/A` value in the exact `search attribute set`.
     *
     * Sets if `part instances` with the `N/A` value should be included in the FilterAttributeInterface.
     *
     * The FilterAttributeInterface has an internal property that tells to include `part instances` with the `N/A` value
     * if if is available, this call updates this internal property,
     * but {@link isNaValueChecked} will return true if and only if {@link hasNaValueActivated}, {@link hasNaValueAvailable} and
     * this internal property are true.<br>
     * This internal property is not reset on a {@link setAttributeName} call.
     *
     * It is legal to make setNaValueChecked(true) but {@link isNaValueChecked} returns false.
     *
     * This internal property is true as default.
     *
     * Returns true if the call will change the content of the FilterAttributeInterface, e.g. calling setNaValueChecked when
     * {@link hasNaValueActivated} or {@link hasNaValueActivated} is false will return false.
     *
     * @param pChecked <div class="const">in</div> Tells if the <code>N/A</code> value should be included/removed in the exact
     * <code>search attribute set</code>.
     * @return `true` if the FilterAttributeInterface is updated.
     */
    setNaValueChecked(pChecked: boolean): boolean;
    /**
     * Tells if the `N/A` value is included in the exact `search attribute set`.
     *
     * The FilterAttributeInterface has an internal property that tells to include `part instances` with the `N/A` value
     * if if is available ({@link setNaValueChecked}),
     * but isNaValueChecked will return true if and only if {@link hasNaValueActivated}, {@link hasNaValueAvailable} and
     * this internal property are true.<br>
     * This internal property is not reset on a {@link setAttributeName} call.
     *
     * It is legal to make {@link setNaValueChecked | setNaValueChecked(true)} but isNaValueChecked returns false.
     *
     * This internal property is true as default.
     *
     * @return `true` if the <code>N/A</code> value is included in the exact <code>search attribute set</code>.
     */
    isNaValueChecked(): boolean;
    /**
     * Sets the exact `search attribute set`.
     *
     * Elects all the `part instances` having the attribute whose name set with {@link setAttributeName}
     * in their `joined attribute set` that have a value exactly included in the given string array.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterAttributeInterface data changed.
     *
     * Do not forget to call {@link DataSessionInterface.update} to trigger a recomputation of
     * the changed filters when you have done all your changes.
     *
     * There is no check to make sure the new `search attribute set` is different from the former
     * `search attribute set`. In this case, the FilterAttributeInterface is modified and
     * recomputed even if the `search attribute set` is not modified.
     *
     * @param pValues <div class="const">in</div> The exact acceptable values.
     */
    setExactValues(pValues: Array<string>): void;
    /**
     * Adds a value in the exact `search attribute set`.
     *
     * Elects all the `part instances` having the attribute whose name set with {@link setAttributeName}
     * in their `joined attribute set` that have a value in the exact `search attribute set`.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterAttributeInterface data changed.
     *
     * Do not forget to call {@link DataSessionInterface.update} to trigger a recomputation of
     * the changed filters when you have done all your changes.
     *
     * @param pValue <div class="const">in</div> The exact value to add in the exact <code>search attribute set</code>.
     */
    addExactValue(pValue: string): void;
    /**
     * Removes a value from the exact `search attribute set`.
     *
     * Elects all the `part instances` having the attribute whose name set with {@link setAttributeName}
     * in their `joined attribute set` that have a value exactly included in the exact `search attribute set`.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterAttributeInterface data changed.
     *
     * Do not forget to call {@link DataSessionInterface.update} to trigger a recomputation of
     * the changed filters when you have done all your changes.
     *
     * @param pValue <div class="const">in</div> The exact value to remove from the <code>search attribute set</code>.
     * @return `true` if such a value was found in the exact <code>search attribute set</code> and therefore the FilterAttributeInterface
     * was modified.
     */
    removeExactValue(pValue: string): boolean;
    /**
     * Tests if a value is in the exact `search attribute set`.
     *
     * @param pValue <div class="const">in</div> The value to query.
     * @return `true` if pValue is in the exact <code>search attribute set</code>.
     */
    isInExactValues(pValue: string): boolean;
    /**
     * Gets a copy of the exact `search attribute set`.
     *
     * @param pExactValuesOut <div class="const">out</div> The array that will be the copy of the exact <code>search attribute set</code>.
     */
    getExactValues(pExactValuesOut: Array<string>): void;
    /**
     * Sets the full text `search attribute set`.
     *
     * Elects all the `part instances` having the attribute whose name set with {@link setAttributeName}
     * in their `joined attribute set` that have a value that contains at least one of the values in the given string array.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterAttributeInterface data changed.
     *
     * Do not forget to call {@link DataSessionInterface.update} to trigger a recomputation of
     * the changed filters when you have done all your changes.
     *
     * Making this call is not legal if {@link hasNaValueActivated} and {@link hasNaValueAvailable} return true.
     * In this case, the call is disregarded (the FilterAttributeInterface is left unchanged) and returns false.
     *
     * If the call is legal, there is no check to make sure the new `search attribute set` is different from the former
     * `search attribute set`. In this case, the FilterAttributeInterface is modified and
     * recomputed even if the `search attribute set` is not modified.
     *
     * @param pValues <div class="const">in</div> The full text acceptable values.
     * @return `true` if the FilterAttributeInterface is updated.
     */
    setContainsValues(pValues: Array<string>): boolean;
    /**
     * Adds a value in the full text `search attribute set`.
     *
     * Elects all the `part instances` having the attribute whose name set with {@link setAttributeName}
     * in their `joined attribute set` that have a value that contains at least one of the values in the given string array.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterAttributeInterface data changed.
     *
     * Do not forget to call {@link DataSessionInterface.update} to trigger a recomputation of
     * the changed filters when you have done all your changes.
     *
     * Making this call is not legal if {@link hasNaValueActivated} and {@link hasNaValueAvailable} return true.
     * In this case, the call is disregarded (the FilterAttributeInterface is left unchanged) and returns false.
     *
     * @param pValue <div class="const">in</div> The value to add in the full text <code>search attribute set</code>.
     * @return `true` if the FilterAttributeInterface is updated.
     */
    addContainValue(pValue: string): boolean;
    /**
     * Removes a value from the full text `search attribute set`.
     *
     * Elects all the `part instances` having the attribute whose name set with {@link setAttributeName}
     * in their `joined attribute set` that have a value that contains at least one of the values in the given string array.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterAttributeInterface data changed.
     *
     * Do not forget to call {@link DataSessionInterface.update} to trigger a recomputation of
     * the changed filters when you have done all your changes.
     *
     * Making this call is not legal if {@link hasNaValueActivated} and {@link hasNaValueAvailable} return true.
     * In this case, the call is disregarded (the FilterAttributeInterface is left unchanged) and returns false.
     *
     * @param pValue <div class="const">in</div> The value to remove from the full text <code>search attribute set</code>.
     * @return `true` if the call is legal and such a value was found in the full text <code>search attribute set</code> and therefore the FilterAttributeInterface
     * was modified.
     */
    removeContainValue(pValue: string): boolean;
    /**
     * Tests if a value is in the full text `search attribute set`.
     *
     * @param pValue <div class="const">in</div> The value to query.
     * @return `true` if pValue is in the full text <code>search attribute set</code>.
     */
    isInContainValues(pValue: string): boolean;
    /**
     * Gets a copy of the full text `search attribute set`.
     *
     * @param pFullTextValuesOut <div class="const">out</div> The array that will be the copy of the full text <code>search attribute set</code>.
     */
    getContainsValues(pFullTextValuesOut: Array<string>): void;
}
export { FilterAttributeInterface };
/**
 * The FilterRangeItemInterface interface defines a numeric range between a min and max value.
 *
 * Each bound (minimum and maximum) can be included or excluded.
 *
 * Date ranges can also be created, dates must be expressed as **number of milliseconds since January 1, 1970, 00:00:00 UTC**.
 *
 * The FilterRangeItemInterface is used by the {@link FilterRangeInterface} and {@link FilterDiagonalInterface}
 * to define acceptable numeric ranges.<br>
 * The FilterRangeItemInterface belongs to the {@link FilterRangeInterface}, {@link FilterDiagonalInterface}
 * that created it. This relationship cannot be altered.
 *
 * FilterRangeItemInterfaces are created through {@link FilterRangeInterface.createFilterRangeItem} and
 * {@link FilterDiagonalInterface.createFilterRangeItem} functions.
 *
 * When modified, the {@link FilterRangeInterface}, {@link FilterDiagonalInterface} this interface belongs to
 * will trigger {@link FilterItemInterfaceSignal.FilterDataChanged} signals.
 *
 * Upon creation, the FilterRangeItemInterface is created with invalid values, you will have to set some values.
 *
 * Be `MIN` the minimum value of the range, `MAX` the maximum value of the ranges, all combinations of ranges can be created :
 * * `[MIN,MAX]`.
 * * `[MIN,MAX[`.
 * * `]MIN,MAX]`.
 * * `]MIN,MAX[`.
 *
 * ```typescript
 * [[include:FilterDiagonalInterface_Creation.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterRangeInterface}
 * @see {@link FilterDiagonalInterface}
 * @see {@link FilterItemInterfaceSignal}
 * @category Metadata/Filters
 */
interface FilterRangeItemInterface {
    /**
     * Sets the lower bound value.
     *
     * Does nothing if pLowerBound is not a number or if pLowerBound
     * is equal to {@link getLowerBound}.
     *
     * Depending on the numeric range of the attribute (see {@link AttributeInfoInterface.getAttributeNumericType}, {@link AttributeNumericType}),
     * the value will be clamped to the acceptable bounds, e.g. setting `-129` for {@link AttributeNumericType.ATTR_NUM_BYTE} will effectively set
     * `-128`.
     * The lower bound is invalid by default.
     *
     * The {@link FilterRangeInterface} or {@link FilterDiagonalInterface} the FilterRangeItemInterface
     * belongs to will trigger a {@link FilterItemInterfaceSignal.FilterDataChanged}.
     *
     * @param pLowerBound <div class="const">in</div> The new lower bound of the range.
     * @return `true` if the value is valid and the item is modified.
     */
    setLowerBound(pLowerBound: number): boolean;
    /**
     * Gets the lower bound value.
     *
     * The lower bound is invalid by default.
     *
     * @return The lower bound of the range.
     */
    getLowerBound(): number;
    /**
     * Sets the upper bound value.
     *
     * Does nothing if pUpperBound is not a number or if pUpperBound
     * is equal to {@link getUpperBound}.
     *
     * Depending on the numeric range of the attribute (see {@link AttributeInfoInterface.getAttributeNumericType}, {@link AttributeNumericType}),
     * the value will be clamped to the acceptable bounds, e.g. setting `128` for {@link AttributeNumericType.ATTR_NUM_BYTE} will effectively set
     * `127`.
     * The upper bound is invalid by default.
     *
     * The {@link FilterRangeInterface} or {@link FilterDiagonalInterface} the FilterRangeItemInterface
     * belongs to will trigger a {@link FilterItemInterfaceSignal.FilterDataChanged}.
     *
     * @param pUpperBound <div class="const">in</div> The new upper bound of the range.
     * @return `true` if the value is valid and the item is modified.
     */
    setUpperBound(pUpperBound: number): boolean;
    /**
     * Gets the upper bound value.
     *
     * The upper bound is invalid by default.
     *
     * @return The upper bound of the range.
     */
    getUpperBound(): number;
    /**
     * Sets if the lower bound value should be included or excluded (loose or strict, >= or >).
     *
     * Does nothing if pIncluded
     * is equal to {@link getIncludedLower}.
     *
     * The lower bound is included by default.
     *
     * The {@link FilterRangeInterface} or {@link FilterDiagonalInterface} the FilterRangeItemInterface
     * belongs to will trigger a {@link FilterItemInterfaceSignal.FilterDataChanged}.
     *
     * @param pIncluded <div class="const">in</div> If true the lower bound value will be included in the range.
     * @return `true` if the item is modified.
     */
    setIncludedLower(pIncluded: boolean): boolean;
    /**
     * Tells if the lower bound value should be included or excluded (loose or strict, >= or >).
     *
     * The lower bound is included by default.
     *
     * @return `true` if the lower bound value is included in the range.
     */
    getIncludedLower(): boolean;
    /**
     * Sets if the upper bound value should be included or excluded (loose or strict, <= or <).
     *
     * Does nothing if pIncluded
     * is equal to {@link getIncludedUpper}.
     *
     * The upper bound is included by default.
     *
     * The {@link FilterRangeInterface} or {@link FilterDiagonalInterface} the FilterRangeItemInterface
     * belongs to will trigger a {@link FilterItemInterfaceSignal.FilterDataChanged}.
     *
     * @param pIncluded <div class="const">in</div> If true the upper bound value will be included in the range.
     * @return `true` if the item is modified.
     */
    setIncludedUpper(pIncluded: boolean): boolean;
    /**
     * Tells if the upper bound value should be included or excluded (loose or strict, >= or >).
     *
     * The upper bound is included by default.
     *
     * @return `true` if the upper bound value is included in the range.
     */
    getIncludedUpper(): boolean;
}
export { FilterRangeItemInterface };
/**
 * The FilterHasFieldInterface interface is a {@link FilterItemInterface} to elect `part instances`
 * that have an attribute metadata with the given name.
 *
 * This {@link FilterItemInterface} selects `part instances` based on the existence of an attribute in their own attributes or their parent attributes (as defined in  <a href="../index.html#documents">Documents</a>).
 * Each `part instance` is considered to have the union between their own attributes and the attributes of its genealogy (parent, grand-parent, etc ...) :
 * the `joined attribute set`.
 *
 * The list of attributes and their types is available through the {@link AttributesDictionaryInterface.getDictionary} function.
 *
 * The FilterHasFieldInterface defines the name of the attribute to check for existence ({@link setAttributeName}).
 *
 * The FilterHasFieldInterface interface is created through the {@link DataSessionInterface.createFilterHasField}.
 *
 * The FilterHasFieldInterface has the type {@link FilterType.FT_HASFIELD} ({@link FilterItemInterface.getFilterType}).
 *
 * The FilterHasFieldInterface has a depth contribution of 1.
 *
 * ```typescript
 * [[include:FilterHasFieldInterface_example.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSolverInterface}
 * @see {@link FilterItemInterfaceSignal}
 * @category Metadata/Filters
 */
interface FilterHasFieldInterface extends FilterItemInterface {
    /**
     * Sets the name of the attribute to filter with.
     *
     * Calling this function with the same attribute name
     * that was set previously does nothing and returns true. <br>
     * Calling this function with an unknown attribute, or an attribute that does
     * not have a string value is illegal. In that case, the FilterHasFieldInterface
     * is left unchanged and false is returned.
     * Please refer to the {@link AttributesDictionaryInterface.getDictionary} to get the
     * list of available attributes and their types.
     *
     *
     * A FilterHasFieldInterface has an empty (invalid) attribute name by default.
     *
     * @param pAttributeName <div class="const">in</div> The new name of the attribute to use.
     * @return `true` if the new attribute name was set and legal.
     */
    setAttributeName(pAttributeName: string): boolean;
    /**
     * Gets the name of the attribute to filter with.
     *
     * @return The name of the attribute to use.
     */
    getAttributeName(): string;
}
export { FilterHasFieldInterface };
/**
 * The FilterBooleanInterface interface is a {@link FilterItemInterface} to elect `part instances`
 * that have a boolean attribute metadata with the given value (`true`, `false`).
 *
 * This {@link FilterItemInterface} selects `part instances` based on the value of a boolean attribute in their own attributes or their parent attributes
 * (as defined in  <a href="../index.html#documents">Documents</a>).
 * Each `part instance` is considered to have the union between their own attributes and the attributes of its genealogy (parent, grand-parent, etc ...) :
 * the `joined attribute set`.
 *
 * The list of attributes and their types is available through the {@link AttributesDictionaryInterface.getDictionary} function.
 *
 * The FilterBooleanInterface defines the value of the attribute to check (`true`/`false`, {@link setBooleanValue}).
 *
 * The FilterBooleanInterface interface is created through the {@link DataSessionInterface.createBooleanFilter}.
 *
 * The FilterBooleanInterface has the type {@link FilterType.FT_BOOLEAN} ({@link FilterItemInterface.getFilterType}).
 *
 * The FilterBooleanInterface has a depth contribution of 1.
 *
 * ```typescript
 * [[include:FilterBooleanInterface_example.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSolverInterface}
 * @see {@link FilterItemInterfaceSignal}
 * @category Metadata/Filters
 */
interface FilterBooleanInterface extends FilterItemInterface {
    /**
     * Sets the name of the attribute to filter with.
     *
     * Calling this function with the same attribute name
     * that was set previously does nothing and returns true. If the attribute name is
     * changed and valid, this function will reset {@link getBooleanValue} to `false`.<br>
     * Calling this function with an unknown attribute, or an attribute that does
     * not have a boolean value is illegal. In that case, the FilterBooleanInterface
     * is left unchanged and false is returned.
     * Please refer to the {@link AttributesDictionaryInterface.getDictionary} to get the
     * list of available attributes and their types.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterBooleanInterface data changed.
     *
     * A FilterBooleanInterface has the value `false` by default ({@link getBooleanValue}).
     *
     * @param pAttributeName <div class="const">in</div> The new name of the attribute to use.
     * @return `true` if the new attribute name was set and legal.
     */
    setAttributeName(pAttributeName: string): boolean;
    /**
     * Gets the name of the attribute to filter with.
     *
     * @return The name of the attribute to use.
     */
    getAttributeName(): string;
    /**
     * Sets the boolean value of the FilterBooleanInterface.
     *
     * If the boolean value if the same as before, nothing is done and true is
     * returned.<br>
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterBooleanInterface data changed.<br>
     * A FilterBooleanInterface has the boolean value `false` by default.
     *
     * @param pBooleanValue <div class="const">in</div> The boolean value to set.
     * @return `true` if pBooleanValue is valid.
     */
    setBooleanValue(pBooleanValue: boolean): boolean;
    /**
     * Gets the boolean value of the FilterBooleanInterface.
     *
     * A FilterBooleanInterface has the boolean value `false` by default.
     *
     * @return The boolean value of the FilterBooleanInterface.
     */
    getBooleanValue(): boolean;
}
export { FilterBooleanInterface };
/**
 * The FilterLiteralInterface interface is a {@link FilterItemInterface} to elect `part instances` that have at least one
 * document whose set of metadata matches a string query.
 *
 * The FilterLiteralInterface defines a string query (literal) written in the
 * <a href="../index.html#3djuump-infinite-literal-and-search-query-language">3djuump infinite literal and search query language</a>.
 *
 * e.g. : `:PartNumber == "part0" and :toBeRedesigned==true` finds documents that have the metadata `PartNumber` equal to `"part0"`
 * and the metadata `toBeRedesigned` equal to `true`. If any document matches, then the `part instances` linked to these documents
 * are elected. The FilterLiteralInterface **DOES NOT** work on the `joined attribute set`.
 *
 * The FilterLiteralInterface has an empty query string as default, which is illegal.
 *
 * The FilterLiteralInterface interface is created through the {@link DataSessionInterface.createLiteralFilter}.
 *
 * The FilterLiteralInterface has the type {@link FilterType.FT_LITERAL} ({@link FilterItemInterface.getFilterType}).
 *
 * The FilterLiteralInterface has a depth contribution of 1.
 *
 * ```typescript
 * [[include:FilterLiteralInterface_example.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSolverInterface}
 * @see {@link FilterItemInterfaceSignal}
 * @category Metadata/Filters
 */
interface FilterLiteralInterface extends FilterItemInterface {
    /**
     * Sets the query string in the <a href="../index.html#3djuump-infinite-literal-and-search-query-language">
     * 3djuump infinite literal and search query language</a>.
     *
     * If the string value if the same as before, nothing is done.<br>
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged}
     * to know when the FilterLiteralInterface data changed.
     *
     * The FilterLiteralInterface has an empty query string as default, which is illegal.
     *
     * @param pLiteralQuery <div class="const">in</div> The query string to set.
     */
    setLiteralQuery(pLiteralQuery: string): void;
    /**
     * Gets the query string that was previously set by {@link setLiteralQuery}.
     *
     * @return The last query string.
     */
    getLiteralQuery(): string;
}
export { FilterLiteralInterface };
/**
 * The FilterSetInterfaceSignal lists all the signals that may be sent by a {@link FilterSetInterface}.
 *
 * These signals are emitted by the {@link FilterSetInterface} when it is modified.
 *
 * There is no way to know when the {@link FilterSetInterface} is ready : the {@link FilterSetInterface}
 * is contained inside a {@link FilterSolverInterface} that notifies when it is ready.
 * As such, there is no way to get the `part instances` or `geometric instances`
 * that are included in such a {@link FilterSetInterface}.
 *
 * As a {@link FilterItemInterface}, the {@link FilterSetInterface} also fires {@link FilterItemInterfaceSignal | FilterItemInterfaceSignals}.
 *
 * The {@link FilterSetInterface} may fire the following signals :
 * * {@link FilterAdded} : a {@link FilterItemInterface} has been successfully added to the {@link FilterSetInterface} ({@link FilterSetInterface.insertFilter}).
 * * {@link FilterRemoved} : a {@link FilterItemInterface} has been successfully removed from the {@link FilterSetInterface} ({@link FilterSetInterface.removeFilter})..
 * * {@link FilterMoved} : a {@link FilterItemInterface} has changed its position inside the {@link FilterSetInterface} ({@link FilterSetInterface.moveFilter}) has been changed.
 * * {@link FilterBulkChanged} : multiple changes have been done at once inside the {@link FilterSetInterface}.
 *
 * ```typescript
 * [[include:FilterSetInterface_listener.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link FilterSetInterface}
 * @see {@link FilterItemInterface}
 * @see {@link FilterItemInterfaceSignal}
 * @category Events
 */
declare enum FilterSetInterfaceSignal {
    /**
     * Signal sent by the {@link FilterSetInterface} when a {@link FilterItemInterface} has been added.
     *
     * The attachment is the {@link FilterItemInterface} that was added.
     */
    FilterAdded = "filterAdded",
    /**
     * Signal sent by the {@link FilterSetInterface} when a {@link FilterItemInterface} has been removed.
     *
     * The attachment is the id of {@link FilterItemInterface} that was removed.
     */
    FilterRemoved = "filterRemoved",
    /**
     * Signal sent by the {@link FilterSetInterface} when a {@link FilterItemInterface} has been moved.
     *
     * The attachment is a {@link FilterMoveInfoInterface} :
     *
     * ```json
     * {
     * 		"filter": "the id of the {@link FilterItemInterface} that changed",
     * 		"to": pOrder // the new position of the {@link FilterItemInterface} in the list of children of the container
     * }
     * ```
     * <br>
     * Please refer to the {@link FilterMoveInfoInterface}.
     *
     * @see {@link FilterMoveInfoInterface}
     */
    FilterMoved = "filterMoved",
    /**
     * Signal sent by the {@link FilterSetInterface} when a lot of changes occur inside.
     *
     * No attachment for events of this type.
     */
    FilterBulkChanged = "filterBulkChanged"
}
export { FilterSetInterfaceSignal };
/**
 * The FilterSetInterface interface is a {@link FilterItemInterface} that groups {@link FilterItemInterface} together.
 *
 * The FilterSetInterface allows to create an operator precedence with {@link FilterOperator}.
 * It is a sort of "parenthesis" filter.
 *
 * There is a maximum `depth` of imbricated FilterSetInterfaces (a FilterSetInterface inside a FilterSetInterface
 * inside a FilterSetInterface ...), the cumulated depth ({@link getDepthContribution}) cannot be more than
 * @maxDepthContribution
 * Any empty FilterSetInterface is valid and is represented by `0` `part instance` and `0` `geometric instance`.
 *
 * The FilterSetInterface interface is created through the {@link DataSessionInterface.createFilterSet}.
 *
 * The FilterSetInterface has the type {@link FilterType.FT_SET} ({@link FilterItemInterface.getFilterType}).
 *
 * The FilterSetInterface has a depth contribution of `1 + Maximum depth contribution of each included FilterItemInterface`.
 *
 * ```typescript
 * [[include:FilterSetInterface_precedence_2.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#available-filters">Available Filters</a> for a list of other {@link FilterItemInterface | FilterItemInterfaces}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSetInterfaceSignal}
 * @see {@link FilterItemInterfaceSignal}
 * @category Metadata/Filters
 */
interface FilterSetInterface extends FilterItemInterface {
    /**
     * Removes all the {@link FilterItemInterface} contained in this FilterSetInterface.
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterChanged} to know
     * when a {@link FilterItemInterface} is changed.
     */
    removeAllFilters(): void;
    /**
     * Inserts a {@link FilterItemInterface} at the given position.
     *
     * Any invalid position will result in the {@link FilterItemInterface} being pushed at the end of
     * the list of {@link FilterItemInterface | FilterItemInterfaces}.
     * Use {@link addEventListener} on the event {@link FilterSetInterfaceSignal.FilterAdded} to know
     * when a {@link FilterItemInterface} is added to the FilterSetInterface.
     *
     * Returns true if the {@link FilterItemInterface} is added, false if the {@link FilterItemInterface}
     * has been cleaned up, pOrder is not a number, or if the {@link FilterItemInterface} is already inside
     * a container.
     *
     * @param pOrder <div class="const">in</div> The position to insert the {@link FilterItemInterface} to (between <code>0</code> and <code>{@link getFilterCount | getFilterCount()}</code> included).
     * @param pFilter <div class="const">in</div> The {@link FilterItemInterface} to add.
     * @return `true` if the {@link FilterItemInterface} is added.
     */
    insertFilter(pOrder: number, pFilter: FilterItemInterface): boolean;
    /**
     * Gets a {@link FilterItemInterface} previously included by {@link insertFilter} given its identifier.
     *
     * Each {@link FilterItemInterface}, upon creation, is given a unique identifier.
     *
     * @param pFilterId <div class="const">in</div> The {@link FilterItemInterface} identifier to query for.
     * @return The {@link FilterItemInterface} with the given id, or undefined if not found.
     */
    getFilterById(pFilterId: string): FilterItemInterface | undefined;
    /**
     * Gets a {@link FilterItemInterface} previously included by {@link insertFilter} given its order in the container list.
     *
     * @param pOrder <div class="const">in</div> The order in the FilterSetInterface of the {@link FilterItemInterface} to get.
     * @return The {@link FilterItemInterface} at the given order, or undefined if pOrder is out of range <code>[0,{@link getFilterCount | getFilterCount()}-1]</code>.
     */
    getFilterByOrder(pOrder: number): FilterItemInterface | undefined;
    /**
     * Gets the number of {@link FilterItemInterface} in the FilterSetInterface.
     *
     * @return The {@link FilterItemInterface | FilterItemInterfaces} count.
     */
    getFilterCount(): number;
    /**
     * Removes a {@link FilterItemInterface} previously included by {@link insertFilter} given its identifier
     * from the container list.
     *
     * Use {@link addEventListener} on the event {@link FilterSetInterfaceSignal.FilterRemoved} to know
     * when a {@link FilterItemInterface} is removed.
     *
     * @param pFilterId <div class="const">in</div> The identifier of the {@link FilterItemInterface} to remove.
     * @return `true` if the {@link FilterItemInterface} was found and removed.
     */
    removeFilter(pFilterId: string): boolean;
    /**
     * Gets the order of the {@link FilterItemInterface} represented by the given identifier.
     *
     * @param pFilterId <div class="const">in</div> The identifier of the {@link FilterItemInterface} to query.
     * @return The order of the {@link FilterItemInterface} in the solver, or -1 if not found.
     */
    getFilterOrder(pFilterId: string): number;
    /**
     * Changes the order of the given {@link FilterItemInterface} in the container list by its filter id.
     *
     * Orders number must be in the range of existing orders i.e. <code>[0,{@link getFilterCount | getFilterCount()}-1]</code>.
     *
     * @param pFilterId <div class="const">in</div> The identifier of the {@link FilterItemInterface} to move.
     * @param pOrder <div class="const">in</div> The new order of the {@link FilterItemInterface}.
     * @return `true` if the {@link FilterItemInterface} was found and changed.
     */
    moveFilter(pFilterId: string, pOrder: number): boolean;
    /**
     * Gets a deep copy of the internal data of the FilterSetInterface.
     *
     * **WARNING** : This function is copied there from the {@link FilterItemInterface} since included {@link FilterItemInterface | FilterItemInterfaces}
     * from {@link insertFilter} are not dumped there.
     *
     * These information will be provided in future version of the API.
     *
     * @return The internal FilterSetInterface data without included {@link FilterItemInterface | FilterItemInterfaces} data.
     */
    getFilterData(): any;
    /**
     * Sets the content of the FilterSetInterface from a former call to {@link getFilterData}.
     *
     * **WARNING** : {@link getFilterData} is lacking enabled, inverted and FilterItemInterface operator information at the moment.
     *
     * **WARNING** : This function is copied there from the {@link FilterItemInterface} since included {@link FilterItemInterface | FilterItemInterfaces}
     * information are not restored here. This function clears {@link FilterItemInterface} (as a call to {@link removeAllFilters}).
     *
     * Use {@link addEventListener} on the event {@link FilterItemInterfaceSignal.FilterDataChanged} to know when the FilterSetInterface
     * internal data changed.
     *
     * @param pFilterData <div class="const">in</div> The internal FilterItemInterface data to set.
     * @return `true` if the data is set.
     */
    setFilterData(pFilterData: any): boolean;
}
export { FilterSetInterface };
/**
 * The FilterCompoundInterface interface is a very specialized {@link FilterSetInterface}.
 *
 * A {@link FilterCompoundInterface} is a combination of metadata filters (({@link FilterAttributeInterface}, {@link FilterRangeInterface},
 * {@link FilterHasFieldInterface}, {@link FilterBooleanInterface})) that works on
 * individual metadata documents and not on the `joined attribute set` (see <a href="../index.html#available-filters">Available Filters</a>).
 *
 * Only {@link FilterAttributeInterface}, {@link FilterRangeInterface},
 * {@link FilterHasFieldInterface} and {@link FilterBooleanInterface} may be included in a FilterCompoundInterface by {@link insertFilter}.
 *
 * The {@link FilterItemInterface | FilterItemInterfaces} are combined with their operators, but are all matched against the
 * **SAME** metadata document.
 *
 * Any empty FilterCompoundInterface is valid and is represented by all the `part instances` and all the `geometric instances`.
 *
 * The FilterCompoundInterface interface is created through the {@link DataSessionInterface.createCompoundFilter}.
 *
 * The FilterCompoundInterface has the type {@link FilterType.FT_COMPOUND} ({@link FilterItemInterface.getFilterType}).
 *
 * The FilterCompoundInterface has a depth contribution of `1 + Maximum depth contribution of each included FilterItemInterface`.
 *
 * ```typescript
 * [[include:FiltersExamples_electrical_replacement_compound.ts]]
 * ```
 * <br>
 * .
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata/Filters</div><br>
 * @see {@link FilterItemInterface}
 * @see {@link FilterSetInterfaceSignal}
 * @see {@link FilterSetInterface}
 * @category Metadata/Filters
 */
interface FilterCompoundInterface extends FilterSetInterface {
    /**
     * Inserts a {@link FilterItemInterface} at the given position.
     *
     * Any invalid position will result in the {@link FilterItemInterface} being pushed at the end of
     * the list of {@link FilterItemInterface | FilterItemInterfaces}.
     * Use {@link addEventListener} on the event {@link FilterSetInterfaceSignal.FilterAdded} to know
     * when a {@link FilterItemInterface} is added to the FilterCompoundInterface.
     *
     * Returns true if the {@link FilterItemInterface} is added, false if the {@link FilterItemInterface}
     * has been cleaned up, pOrder is not a number, if the {@link FilterItemInterface} is already inside
     * a container, or if the {@link FilterItemInterface} is neither a {@link FilterAttributeInterface},
     * a {@link FilterRangeInterface}, a {@link FilterHasFieldInterface} or a {@link FilterBooleanInterface}.
     *
     * @param pOrder <div class="const">in</div> The position to insert the {@link FilterItemInterface} to (between <code>0</code> and <code>{@link getFilterCount | getFilterCount()}</code> included).
     * @param pFilter <div class="const">in</div> The {@link FilterItemInterface} to add.
     * @return `true` if the {@link FilterItemInterface} is added.
     */
    insertFilter(pOrder: number, pFilter: FilterItemInterface): boolean;
}
export { FilterCompoundInterface };
/**
 * The FontLoaderInterfaceSignal lists all the signals that may be sent by the {@link FontLoaderInterface}.
 *
 * These signals are emitted by the {@link FontLoaderInterface} has finished loading.
 *
 * The {@link FontLoaderInterface} may fire the following signals :
 * * {@link FontsLoaded} : the {@link FontLoaderInterface} has finished loading all the fonts that were queried to load.
 *
 * ```typescript
 * [[include:FontLoaderInterface_loadFont.ts]]
 * ```
 * <br>
 * Please refer to the {@link FontLoaderInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link FontLoaderInterface}
 * @category Events
 */
declare enum FontLoaderInterfaceSignal {
    /**
     * Signal sent by the {@link FontLoaderInterface} when the {@link FontLoaderInterface} has finished loading the fonts.
     * This does not tell if the loading was successful.
     *
     * No attachment for events of this type.
     */
    FontsLoaded = "fontsLoaded"
}
export { FontLoaderInterfaceSignal };
/**
 * The result of a font loading request or the loading status of a font.
 *
 * This result is obtained from a call to {@link FontLoaderInterface.loadFont} and {@link FontInterface | FontInterfaces} inside a {@link FontLoaderInterface.getRegisteredFonts} request.
 *
 * ```typescript
 * [[include:FontLoaderInterface_loadFont.ts]]
 * ```
 * <br>
 * Or asynchronously :
 *
 * ```typescript
 * [[include:FontLoaderInterface_asyncLoadFont.ts]]
 * ```
 * Please refer to {@link FontLoaderInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link FontLoaderInterface}
 * @category 3D Rendering
 */
declare enum FontLoadingStatus {
    /**
     * The font is not loaded and no pending loading request is ongoing.
     */
    FLS_NotLoaded = 0,
    /**
     * The font is already loaded.
     */
    FLS_Loaded = 1,
    /**
     * The font is loading.
     */
    FLS_Loading = 2,
    /**
     * The font loading procedure has just started.
     */
    FLS_LoadingStarted = 3,
    /**
     * The given font was not found.
     */
    FLS_FontMissing = 4,
    /**
     * The previous call to {@link FontLoaderInterface.loadFont} failed.
     *
     * There is no way to update the buffer or url of a given font, just
     * unregister and register the font again with correct parameters.
     *
     * @see {@link FontLoaderInterface.unregisterFont}
     * @see {@link FontLoaderInterface.registerFont}
     */
    FLS_FontError = 5
}
export { FontLoadingStatus };
/**
 * The memory representation of a font in the {@link FontLoaderInterface}.
 *
 * FontInterface are retrieved through {@link FontLoaderInterface.getRegisteredFonts}.
 *
 * ```typescript
 * [[include:FontLoaderInterface_loadFont.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link FontLoaderInterface}
 * @category 3D Rendering
 */
interface FontInterface {
    /**
     * The family name of the given font (lowercase).
     */
    familyName: string;
    /**
     * Tells if the font is a bold one.
     */
    bold: boolean;
    /**
     * Tells if the font is an italic one.
     */
    italic: boolean;
    /**
     * The loading status of the font.
     */
    loadStatus: FontLoadingStatus;
    /**
     * The error message (if any) of the font.
     */
    error?: string;
}
export { FontInterface };
/**
 * The {@link FontLoaderInterface} is used when displaying annotations.
 *
 * The underlying optimized rendering algorithms of annotations must have
 * access to glyph data and metrics, but these information are
 * not accessible through the native javascript api. For these reasons,
 * the api must be able to parse font files and retrieve such data.
 *
 * There may be some specific fonts needed to display your annotations.
 * Just register them to the system with {@link registerFont}.
 * You may register fonts from their url, or from their data with an ArrayBuffer.
 * Font loading takes place when annotations are parsed and included in the {@link AnnotationRendererInterface}.
 * Even if {@link loadFont} is provided, the font loading mechanism will automatically take place when required.
 *
 * The api provides a default font (included in its script, nothing must be done to get this default font),
 * regular "euclid flex b".
 *
 * If a font is not found in the FontLoaderInterface, then the default font is used. And if the default font could
 * not be loaded, then regular "euclid flex b" font is used.
 * You may override the default font when registering fonts. There can be only one default font.
 * The default font is changed through calling {@link registerFont} or {@link setDefaultFont} accordingly.
 * If such a font is not valid, then the api will use the "euclid flex b" font.
 *
 * Family names in the FontLoaderInterface are case insensitive, all family names will be used lowercase.
 *
 * For memory reasons, there is only one FontLoaderInterface, and thus FontLoaderInterface(s) cannot be created, but
 * is accessed through {@link InfiniteApiControllerInterface.getFontLoader}.
 *
 * You may choose to load specific fonts beforehand, in that case when all the fonts are loaded, a {@link FontLoaderInterfaceSignal}
 * signal is sent when the loading is over.
 *
 * ```typescript
 * [[include:FontLoaderInterface_loadFont.ts]]
 * ```
 * <br>
 * Or asynchronously :
 *
 * ```typescript
 * [[include:FontLoaderInterface_asyncLoadFont.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * Remember that font loading is only used with {@link AnnotationRendererInterface | Annotations}.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link InfiniteFactory}
 * @see {@link AnnotationRendererInterface}
 * @category 3D Rendering
 */
interface FontLoaderInterface extends EventDispatcherInterface {
    /**
     * Registers a font to the font loading mechanism.
     *
     * As fonts are loaded only when needed, the family name,
     * bold and italic properties must be set beforehand in order
     * to tell the system to load such a font when an annotation
     * needs it.
     *
     * If a font with the same properties was already registered, the call
     * to registerFont may only change the default font if set. It will return true even
     * if the font url or data is not changed.
     *
     * If pUrlOrContent is a string, next calls to {@link loadFont} will be asynchronous, if
     * pUrlOrContent is an ArrayBuffer, then next calls to {@link loadFont} will be synchronous
     * (and the {@link FontLoaderInterfaceSignal.FontsLoaded} signal is sent right away).
     *
     * @param pFamilyName <div class="const">in</div> The family name of the font to register.
     * @param pBold <div class="const">in</div> Tells if the font is a bold one.
     * @param pItalic <div class="const">in</div> Tells if the font is an italic one.
     * @param pUrlOrContent <div class="const">in</div> The url or the content of the font file.
     * @param pDefaultFont <div class="const">in</div> Tells that this font will be the default one (defaults to false).
     * @return `true` if the input parameters are correct. If such a font is already registered, true is also returned.
     */
    registerFont(pFamilyName: string, pBold: boolean, pItalic: boolean, pUrlOrContent: string | ArrayBuffer, pDefaultFont?: boolean): boolean;
    /**
     * Unregisters a font from the font loading mechanism.
     *
     * You cannot unregister the default font nor the "euclid flex b" font.
     *
     * Returns true if such a font was found and unregistered.
     *
     * @param pFamilyName <div class="const">in</div> The family name of the font to unregister.
     * @param pBold <div class="const">in</div> Tells if the font is a bold one.
     * @param pItalic <div class="const">in</div> Tells if the font is an italic one.
     * @return `true` if such a font was found and unregistered.
     */
    unregisterFont(pFamilyName: string, pBold: boolean, pItalic: boolean): boolean;
    /**
     * Sets an already registered font to be the default one.
     *
     * Returns true if such a font was found and set as default.
     *
     * @param pFamilyName <div class="const">in</div> The family name of the font to set as default.
     * @param pBold <div class="const">in</div> Tells if the font is a bold one.
     * @param pItalic <div class="const">in</div> Tells if the font is an italic one.
     * @return `true` if such a font was found and set as default. If such a font was already the default font,
     * then `true` is also returned.
     */
    setDefaultFont(pFamilyName: string, pBold: boolean, pItalic: boolean): boolean;
    /**
     * Asks the font loader to begin loading the given font.
     *
     * Returns a {@link FontLoadingStatus} that tells the result of the load request.
     *
     * @param pFamilyName <div class="const">in</div> The family name of the font to load.
     * @param pBold <div class="const">in</div> Tells if the font is a bold one.
     * @param pItalic <div class="const">in</div> Tells if the font is an italic one.
     * @return The {@link FontLoadingStatus} that tells the result of the load request.
     * @see {@link FontLoadingStatus}
     */
    loadFont(pFamilyName: string, pBold: boolean, pItalic: boolean): FontLoadingStatus;
    /**
     * Waits till no font loading is in progress.
     *
     * @return A Promise. There is no return value.
     */
    asyncWaitForFontsLoaded(): Promise<void>;
    /**
     * Tells if the given font is loaded.
     *
     * Returns true if the given font is loaded.
     *
     * @param pFamilyName <div class="const">in</div> The family name of the font to query.
     * @param pBold <div class="const">in</div> Tells if the font is a bold one.
     * @param pItalic <div class="const">in</div> Tells if the font is an italic one.
     * @return `true` if the font  is loaded.
     * @see {@link FontLoadingStatus}
     */
    isFontLoaded(pFamilyName: string, pBold: boolean, pItalic: boolean): boolean;
    /**
     * Gets the number of registered fonts.
     *
     * Fonts are registered by a call to {@link registerFont}.
     *
     * @return The number of registered fonts.
     */
    getNbRegisteredFonts(): number;
    /**
     * Gets all the registered fonts.
     *
     * Fonts are registered by a call to {@link registerFont}.
     *
     * @param pFonts <div class="const">out</div> The list of all registered fonts.
     */
    getRegisteredFonts(pFonts: Array<FontInterface>): void;
    /**
     * Gets the number of loading fonts.
     *
     * Fonts are loaded by a call to {@link loadFont}.
     *
     * @return The number of loading fonts.
     */
    getNbLoadingFonts(): number;
    /**
     * Gets the concatenation of all fonts error.
     *
     * Fonts are loaded by a call to {@link loadFont}.
     *
     * @return The number of loading fonts.
     */
    getFontErrors(): string;
    /**
     * Gets the number of fonts in error.
     *
     * Fonts are loaded by a call to {@link loadFont}.
     *
     * @return The number of fonts in error.
     */
    getNbFontsInError(): number;
}
export { FontLoaderInterface };
/**
 * There is a typo error in this enumeration.
 *
 * GeometricInstanceConverterSignal should have been {@link GeometricInstanceConverterInterfaceSignal}.
 *
 * @deprecated Please use {@link GeometricInstanceConverterInterfaceSignal} instead.
 * @internal
 */
declare enum GeometricInstanceConverterSignal {
    /**
     * No description.
     */
    GeometricInstanceConverterReady = "ready",
    /**
     * No description.
     */
    GeometricInstanceConverterCancelled = "cancelled"
}
export { GeometricInstanceConverterSignal };
/**
 * The {@link GeometricInstanceConverterInterfaceSignal} lists all the signals that may be sent by the {@link GeometricInstanceConverterInterface}.
 *
 * These signals are emitted by the {@link GeometricInstanceConverterInterface} when some data is available.
 *
 * * {@link GeometricInstanceConverterReady} : the {@link GeometricInstanceConverterInterface} is ready, i.e. when it has
 * finished getting the `part instance ids` that are expressed by the given `geometric instance ids`, {@link GeometricInstanceConverterInterface.getPartInstanceIds} can be called.
 * * {@link GeometricInstanceConverterCancelled} : the retrieving of the content by the {@link GeometricInstanceConverterInterface} has been cancelled.
 *
 * ```typescript
 * [[include:GeometricInstanceConverterInterface_example.ts]]
 * ```
 * <br>
 * Or with an idcard request :
 *
 * ```typescript
 * [[include:IdCardGetterInterface_example_picking.ts]]
 * ```
 * <br>
 * Please refer to {@link IdCardGetterInterface} for more information about idcard requests.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link GeometricInstanceConverterInterface}
 * @category Events
 */
declare enum GeometricInstanceConverterInterfaceSignal {
    /**
     * Signal sent by the {@link GeometricInstanceConverterInterface} when {@link GeometricInstanceConverterInterface.convert} is ready, i.e.
     * {@link GeometricInstanceConverterInterface.getPartInstanceIds} can be called.
     *
     * Such a signal does not tell if the call was successful or not but that the result is ready to be fetched.
     *
     * Warning, the GeometricInstanceConverterReady signal is sent later when {@link GeometricInstanceConverterInterface.cancel} is called and the {@link GeometricInstanceConverterInterface} is fetching data.
     *
     * The attachment is a string telling the request id of the call to {@link GeometricInstanceConverterInterface.convert} ({@link GeometricInstanceConverterInterface.getLastRequestId}).
     */
    GeometricInstanceConverterReady = "ready",
    /**
     * Signal sent by the {@link GeometricInstanceConverterInterface} when the {@link GeometricInstanceConverterInterface} has been cancelled.
     *
     * Such an event is fired if the {@link GeometricInstanceConverterInterface} was fetching data and {@link GeometricInstanceConverterInterface.cancel} is called.
     * The {@link GeometricInstanceConverterReady} signal will be triggered shortly after.
     *
     * No attachment for events of this type.
     */
    GeometricInstanceConverterCancelled = "cancelled"
}
export { GeometricInstanceConverterInterfaceSignal };
/**
 * The GeometricInstanceConverterInterface interface is used to get, from a list of `geometric instance ids`, the corresponding `part instance ids`.
 *
 * Such a conversion is dependant on a filtering context. Indeed, the list of current `part instances` is dependant on the chosen configuration (at least)
 * and may be dependant on the list of visible `part instances` for example. For this reason, converting `geometric instance ids` (from a picking request for example)
 * needs a filtering context (see {@link VisibilityContextInterface}).
 *
 * This conversion is particularly useful to make idcard requests, since metadata documents are retrieved from `part instances` and not `geometric instances`.
 *
 * The GeometricInstanceConverterInterface interfaces are created through the {@link DataSessionInterface.createGeometricInstanceConverter} method.
 *
 * The list of signals the GeometricInstanceConverterInterface may trigger is available in the {@link GeometricInstanceConverterInterfaceSignal} enumeration.
 *
 * The conversion mechanism is triggered through the {@link convert} method. The result is not available right away, but the
 * event {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterReady} is triggered when the result of the GeometricInstanceConverterInterfaceSignal is available.
 * The result is available through the {@link getPartInstanceIds} function.
 *
 * Warning : there may be cases when the {@link getPartInstanceIds} is not available such as when the GeometricInstanceConverterInterface is fetching data, i.e. when
 * {@link isRunning} returns true, or when the GeometricInstanceConverterInterface has been cancelled, i.e. when {@link isCancelled} returns true.
 *
 * A GeometricInstanceConverterInterface may be interrupted (cancelled) when the GeometricInstanceConverterInterface is fetching data and {@link cancel}
 * is called. In such cases, the {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterCancelled} signal is fired, and shortly after,
 * {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterReady}
 * signal is fired, but {@link getPartInstanceIds} will return undefined. Just call {@link convert} with another (or the same) `geometric instance ids` to trigger a new
 * retrieval.
 *
 * If you call multiple times {@link convert} before receiving the {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterReady},
 * only one {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterReady} will be sent with the content of the last call to {@link convert}.
 *
 * This interface performs the opposite operation of the {@link PartInstanceConverterInterface} interface.
 *
 * ```typescript
 * [[include:GeometricInstanceConverterInterface_example.ts]]
 * ```
 * <br>
 * Or with async calls :
 *
 * ```typescript
 * [[include:GeometricInstanceConverterInterface_asyncExample.ts]]
 * ```
 * <br>
 * An example with an idcard request :
 *
 * ```typescript
 * [[include:IdCardGetterInterface_example_picking.ts]]
 * ```
 * <br>
 * Please refer to {@link IdCardGetterInterface} for more information about idcard requests.
 *
 * [[include:constStyle.md]]<br><div class="const">Converters</div><br>
 * @see {@link GeometricInstanceConverterInterfaceSignal}
 * @category Converters
 */
interface GeometricInstanceConverterInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Triggers a request to "translate" the given `geometric instance ids` to the corresponding `part instance ids`.
     *
     * The server will try to find all `part instances` that are linked to the given `geometric instance ids` in the given filtering context.
     * An event {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterReady} is fired when the translation is finished,
     * use {@link getLastError | getLastError()} to check if it was correctly performed.
     *
     * Returns true if the "conversion" is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed.
     *
     * If `pVisibilityContext` is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pGeometricInstanceIds <div class="const">in</div> The <code>geometric instance ids</code> to perform the conversion with.
     * @param pVisibilityContext <div class="const">in</div> The filtering context to use when "converting" to a list of <code>part instance ids</code>.
     * @return `true` if the conversion process has started, just wait for {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterReady}.
     */
    convert(pGeometricInstanceIds: Uint32Array | number[], pVisibilityContext: VisibilityContextInterface): boolean;
    /**
     * Asynchronously triggers a request to "translate" the given `geometric instance ids` to the corresponding `part instance ids`.
     *
     * The server will try to find all `part instances` that are linked to the given `geometric instance ids` in the given filtering context.
     *
     * Returns a promise.
     *
     * Please note that in case of multiple promises running at the same time on the same GeometricInstanceConverterInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getPartInstanceIds}
     * after awaiting will return the same value.
     *
     * If pVisibilityContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pGeometricInstanceIds <div class="const">in</div> The <code>geometric instance ids</code> to perform the conversion with.
     * @param pVisibilityContext <div class="const">in</div> The filtering context to use when "converting" to a list of <code>part instance ids</code>.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the requested Uint32Array.
     */
    asyncConvert(pGeometricInstanceIds: Uint32Array | number[], pVisibilityContext: VisibilityContextInterface): Promise<AsyncUInt32ArrayResult>;
    /**
     * Each call to {@link convert} is assigned a request id.
     *
     * This call tels the id of the last call to {@link convert}.
     *
     * @return The id of the last call to {@link convert}.
     */
    getLastRequestId(): string;
    /**
     * Cancels the computation of the conversion process (if any).
     *
     * A {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterCancelled} signal is emitted if the
     * GeometricInstanceConverterInterface is retrieving data.
     *
     * @return `true` if the GeometricInstanceConverterInterface was running, else false.
     * @see {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterCancelled}
     */
    cancel(): boolean;
    /**
     * Tells if the GeometricInstanceConverterInterface is converting data.
     *
     * This is the case after calling {@link convert}.
     *
     * @return `true` if the GeometricInstanceConverterInterface is converting.
     */
    isRunning(): boolean;
    /**
     * Tells if the GeometricInstanceConverterInterface has been cancelled.
     *
     * This is generally the case after calling {@link cancel} when the GeometricInstanceConverterInterface
     * is retrieving data.
     *
     * @return `true` if the GeometricInstanceConverterInterface is cancelled.
     */
    isCancelled(): boolean;
    /**
     * Gets the last error returned by the {@link convert} call of the GeometricInstanceConverterInterface.
     *
     * @return The last error message (if any, or an empty string if no error occurred).
     */
    getLastError(): string;
    /**
     * Gets the result of the last {@link convert} call of the `geometric instance ids` conversion.
     *
     * An Uint32Array representing the `part instance ids` is returned if the GeometricInstanceConverterInterface
     * has finished computing.
     * Use {@link addEventListener} on the event {@link GeometricInstanceConverterInterfaceSignal.GeometricInstanceConverterReady} to
     * know when the GeometricInstanceConverterInterface is ready.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of <code>part instance ids</code>,
     * result of the conversion, or undefined if the converter had an error, or is still computing.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     */
    getPartInstanceIds(): Uint32Array | undefined;
}
export { GeometricInstanceConverterInterface };
/**
 * The PartInstanceInfoStatusFlag enumeration tells some information about
 * the `part instances` inside the DMU.
 *
 * The PartInstanceInfoStatusFlag gives added information to a `part instance`
 * when retrieved by an {@link IdCardGetterInterface} or a {@link ChildrenIdCardGetterInterface}. It is included in the {@link PartInstanceInfoInterface} or {@link ChildrenPartInstanceInfoInterface}.
 *
 * It mainly tells if a `part instance` contains at least one geometry ({@link PI_Displayable}).
 *
 * The PartInstanceInfoStatusFlag is a flag, which means several information are stored in a number as
 * an OR of the enumeration values.
 *
 * ```typescript
 * [[include:IdCardGetterInterface_example.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:IdCardGetterInterface_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link PartInstanceInfoInterface}
 * @see {@link IdCardGetterInterface}
 * @see {@link ChildrenIdCardGetterInterface}
 * @category Data Retrievers
 */
declare enum PartInstanceInfoStatusFlag {
    /**
     * Tells if this `part instance` contains at least one valid geometry and can be displayed in a 3D View.
     */
    PI_Displayable = 1,
    /**
     * Tells if at least one geometry was missing when the `part instance` was built (only for leaves).
     */
    PI_ConversionGeometryMissing = 2,
    /**
     * Tells if at least one leaf of this `part instance` had the {@link PI_ConversionGeometryMissing} flag (only for nodes).
     */
    PI_ConversionGeometryPropagated = 4,
    /**
     * Tells that at least one child (immediate or not) `part instance` (`sub-part`, leaf or node) was missing when this `part instance`
     * was built.
     */
    PI_InstantiationErrorPartMissing = 8,
    /**
     * Tells that an instantiation loop (i.e. this `part instance` is a child, immediate or not of itself) was detected when this `part instance`
     * was built.
     */
    PI_InstantiationErrorLoopDetectedOrDepthLimitReached = 16,
    /**
     * Tells that this `part instance` contains at least one child (immediate or not) that has the
     * {@link PI_InstantiationErrorPartMissing} or {@link PI_InstantiationErrorLoopDetectedOrDepthLimitReached} flag.
     */
    PI_InstantiationErrorPropagated = 32,
    /**
     * Tells that the given `part instance` is not totally included in the DMU Axis Aligned Bounding Box (AABB).
     */
    PI_OutOfLimitAABB = 64,
    /**
     * Tells that at least one triangle was removed when this `part instance` was built (only for leaves).
     */
    PI_ConversionGeometryPartial = 128,
    /**
     * Tells that this `part instance` contains at least one annotation.
     */
    PI_Annotated = 256,
    /**
     * Tells that this `part instance` has a geometry instance id and a class id that was aggregated with
     * another `part instance` with a tolerance (not a strict xform equality leaves).
     *
     * Multiple `part instances` may have the same `geometric instance id` if they have the same
     * geometric model and at the "same" location.
     */
    PI_GeometryInstanceIdAndClassIdMutualizedWithTolerance = 512
}
export { PartInstanceInfoStatusFlag };
/**
 * The ObjectMetadataInterface tells about the metadata documents linked to a `part instance` and its genealogy or children.
 *
 * The ObjectMetadataInterface is included in the {@link PartInstanceInfoInterface} or {@link ChildrenPartInstanceInfoInterface},
 * which is retrieved by an {@link IdCardGetterInterface} or {@link ChildrenIdCardGetterInterface}.
 *
 * The ObjectMetadataInterface shows the metadata documents attached to a `part instance`.
 *
 * Such metadata documents may be :
 * * A `part metadata document` : this document is shared among all the instances of the same `part` (a bolt for example).
 * * A `link metadata document` : when defining a child of a `part instance`, a metadata document may be set to define the link between the two
 * `part instances`. The child `part instance` includes the link metadata document (as such, the root `part instance` of the DMU cannot have a link metadata document and thus
 * its link metadata document is empty).
 * * An `instance metadata document` : the given `part instance` may have specific information attached to it (not shared by all the instances of the same `part`, as a
 * serial number for example).
 *
 * All these documents follow the same pattern :
 * * A string document id, which is a unique string inside the DMU that identifies the document.
 * * The actual metadata object.
 *
 * Please remember that `metadata documents` cannot be queried by their `string id` but by their `integer ids` (see <a href="../index.html#search">Search</a>).
 *
 * The `link metadata document` also features specific information (the "effectivity" property) that are used for the configurations ({@link ConfContextInterface}, <a href="../index.html#main-concepts">Main concepts</a>) of the DMU.
 * A configuration is a set of included or excluded effectivities. Any link that satisfies the requirements in term of effectivities is
 * included in the configuration, meaning the child `part instance` of the link will be available in the given configuration.
 * If not, the given child `part instance` will not be included. Any `link metadata document` may thus have an effectivity property.
 *
 * The ObjectMetadataInterface is organized by document types (`part metadata documents`, `link metadata documents`, `part instance metadata documents`).
 *
 * There is a unique instantiation chain of `part instances`, starting from the root `part instance` to a specific `part instance` (this is the definition of a
 * `part instance`).
 * The ObjectMetadataInterface may include the documents of such a chain if included in a {@link PartInstanceInfoInterface}, if included in a {@link ChildrenPartInstanceInfoInterface},
 * then the documents are the one of the children of the requested `part instance`.
 * The {@link IdCardGetterInterface.retrieveIdCard} and {@link ChildrenIdCardGetterInterface.retrieveChildrenIdCard} will return the same number of {@link PartInstanceInfoInterface} than
 * the number of `part instances` that were queried.
 *
 * A chain, or children are exposed by the ObjectMetadataInterface as arrays. All the arrays have the same size, the size of the chain of the `part instances`, or the number
 * of children of a part instance.
 * In the case of a chain, the first
 * item in the chain is the root `part instance`, the last item is the `part instance` that was queried by {@link IdCardGetterInterface.retrieveIdCard}.
 * In the case of a children list, the order is irrelevant.
 *
 * Warning : the `metadata documents` content may change depending on the given configuration, as such, some fields may appear or not depending on the chosen
 * configuration.
 *
 * The ObjectMetadataInterface also shows if a `part instance` in the chain or children has any attached document (see {@link DocumentContentGetterInterface} and {@link AttachedDocumentInfoInterface})
 * attached to it. A boolean array (size fo the instantiation chain, number of children) exposes this property.
 *
 * Here is an example of an ObjectMetadataInterface :
 *
 * ```json
 * {
 * 	"partmd": [{
 * 			"id": "469ce189fc6c4c70a8a1d32dda2b8005",
 * 			"metadata": {
 * 				"Input Format and Emitter": "CATIA V5R19 SP0 HF0",
 * 				"Name": "Main car",
 * 				"Original Name": "Main car",
 * 				"OriginalUnits": {
 * 					"Original length unit (m)": 0.0010000000000000002,
 * 					"Original mass unit (kg)": 1,
 * 					"Original time unit (s)": 1
 * 				}
 * 			}
 * 		}, {
 * 			"id": "c12cd6c431ef415597d8c13df06bcaf9",
 * 			"metadata": {
 * 				"Input Format and Emitter": "CATIA V5R19 SP0 HF0",
 * 				"Name": "REMONTAGE-VITRAGE",
 * 				"Original Name": "REMONTAGE-VITRAGE",
 * 				"OriginalUnits": {
 * 					"Original length unit (m)": 0.0010000000000000002,
 * 					"Original mass unit (kg)": 1,
 * 					"Original time unit (s)": 1
 * 				},
 * 				"PartNumber": "REMONTAGE-VITRAGE",
 * 				"subsystem": "5200-Vitrages",
 * 				"system": "5000-Exter"
 * 			},
 * 		}, {
 * 			"id": "cd303e312a0641a4b9f84ba496cf2ae5",
 * 			"metadata": {
 * 				"Design Name": "3507-5219-A02-Plexi-zone-AV-CBI-080113",
 * 				"Design Status": "Published",
 * 				"Input Format and Emitter": "CATIA V5R19 SP0 HF0",
 * 				"Issue": "A",
 * 				"Last update": "06/12/2012",
 * 				"Name": "3507-5219-A02-Plexi_zone_AV-CBI-080113",
 * 				"Original Name": "3507-5219-A02-Plexi_zone_AV-CBI-080113",
 * 				"OriginalUnits": {
 * 					"Original length unit (m)": 0.0010000000000000002,
 * 					"Original mass unit (kg)": 1,
 * 					"Original time unit (s)": 1
 * 				},
 * 				"PartNumber": "3507-5219-A02-Plexi_zone_AV-CBI-080113",
 * 				"subsystem": "5200-Vitrages",
 * 				"system": "5000-Exter"
 * 			}
 * 		}
 * 	],
 * 	"linkmd": [{}, {
 * 			"id": "linkmd_withoutconf_46",
 * 			"metadata": {
 * 				"branch": "test"
 * 			}
 * 		}, {}
 * 	],
 * 	"instancemd": [{}, {}, {}
 * 	],
 * 	"hasattacheddocuments": [false, false, false]
 * }
 * ```
 * <br>
 * The ObjectMetadataInterface may be queried by the following code :
 *
 * ```typescript
 * [[include:IdCardGetterInterface_example.ts]]
 * ```
 * <br>
 * or :
 *
 * ```typescript
 * [[include:ChildrenIdCardGetterInterface_example.ts]]
 * ```
 * <br>
 * With async calls :
 *
 * ```typescript
 * [[include:IdCardGetterInterface_asyncExample.ts]]
 * ```
 * <br>
 * or :
 *
 * ```typescript
 * [[include:ChildrenIdCardGetterInterface_asyncExample.ts]]
 * ```
 * <br>
 * The ObjectMetadataInterface contains fields that are included in the {@link AttributesDictionaryInterface}.
 * Just look for the given attribute in the {@link AttributesDictionaryInterface}.
 *
 * Any field may have the given type, an array of the given type or null (depending on {@link AttributeInfoInterface.getAttributeType}).
 * Any field may have not be included in the {@link AttributesDictionaryInterface}, in this case, such metadata cannot be filtered, since it cannot be indexed.
 * Such data that cannot be indexed can be found in the {@link nonindexedpartmd}, {@link nonindexedlinkmd}
 * or {@link nonindexedinstancemd} fields.
 *
 * Consider now the following example :
 *
 * Dictionary :
 *
 * ```json
 * [[include:dictionary.json]]
 * ```
 * <br>
 * ObjectMetadataInterface :
 *
 * ```javascript
 * [[include:metadata.js]]
 * ```
 * <br>
 * Please refer to {@link PartInstanceInfoInterface} and {@link ChildrenPartInstanceInfoInterface} to get more information about metadata requests.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link PartInstanceInfoInterface}
 * @see {@link ChildrenPartInstanceInfoInterface}
 * @category Data Retrievers
 */
interface ObjectMetadataInterface {
    /**
     * The `part` metadata document is shared among all `part instances` of the same `part`.
     *
     * This is a list of all the `part` metadata documents of the `part instance` and its genealogy, or the `part` metadata documents of the children of the `part instance`.
     *
     * A `part` is the definition of a single element of the DMU that may be instantiated multiple times.
     * This is the genealogy chain or children list of `part` metadata documents. Any missing document for the given chain/list is represented by
     * an empty object.
     *
     * The document may be empty, meaning no information is available for the specific ancestor, or this `part instance` has no child.
     * * "id" is the name (string) of the `part` metadata document (unique across the DMU) for this `part instance`.
     * * "metadata" is the optional metadata of the `part instance` (parent or current `part instance`).
     */
    partmd: ({} | {
        /**
         * The name of the part metadata document.
         */
        id: string;
        /**
         * The content of the metadata.
         */
        metadata?: Object;
    })[];
    /**
     * Each child of an instance is linked to its parent with an optional link metadata document.
     *
     * This is a list of all the link metadata documents of the `part instance` and its genealogy, or the list of link metadata documents of the children of this `part instance`.
     *
     * If defined, the link document is retrieved and included in the definition of the child.
     *
     * The document may be empty, meaning no information is available for the specific ancestor (or current
     * `part instance`), or specific child.
     * * "id" is the name of the link metadata document (unique across the DMU).
     * * "metadata" is the optional metadata included in the link between the `part instance` (parent or current `part instance`).
     * * "effectivity" is the optional effectivity (configuration information) included in the `part instance` (parent or current `part instance`, or child depending on the call).
     */
    linkmd: ({} | {
        /**
         * The name of the link metadata document.
         */
        id: string;
        /**
         * The content of the metadata.
         */
        metadata?: Object;
        /**
         * The optional effectivity.
         */
        effectivity?: Object;
    })[];
    /**
     * Each `part instance` may have specific metadata attached to it with a specific document.
     *
     * This is a list of all the instance metadata documents of the `part instance` and its genealogy,
     * or the list of instance metadata documents of the children of this `part instance`.
     *
     * The document may be empty, meaning no information is available for the specific ancestor (or current
     * `part instance`), or specific child.
     * * "id" is the name of the instance metadata document (unique across the DMU).
     * * "metadata" is the optional instance metadata of this specific `part instance`.
     */
    instancemd: ({} | {
        /**
         * The name of the instance metadata document.
         */
        id: string;
        /**
         * The content of the metadata.
         */
        metadata?: Object;
    })[];
    /**
     * Tells if the `part instance` has annotations attached to it.
     */
    hasannotations: boolean[];
    /**
     * Each `part instance` may have specific documents attached to it.
     *
     * This is a list that tells if any `part instance` in the `part instance` genealogy / children has documents.
     *
     * @see {@link IdCardGetterInterface}
     */
    hasattacheddocuments: boolean[];
    /**
     * Tells the fields of the part metadata document that are either:
     * * missing from the {@link AttributesDictionaryInterface}.
     * * `null`.
     * * not of the expected type.
     *
     * The name of the attribute is the path (concatenated with '.') and optionally a number (if the type is an array), or gt, gte, lt, lte for range values.
     */
    nonindexedpartmd: string[];
    /**
     * Tells the fields of the link metadata document that are either:
     * * missing from the {@link AttributesDictionaryInterface}.
     * * null
     * * not of the expected type.
     *
     * The name of the attribute is the path (concatenated with '.') and optionally a number (if the type is an array), or gt, gte, lt, lte for range values.
     */
    nonindexedlinkmd: string[];
    /**
     * Tells the fields of the instance metadata document that are either:
     * * missing from the {@link AttributesDictionaryInterface}.
     * * null
     * * not of the expected type.
     *
     * The name of the attribute is the path (concatenated with '.') and optionally a number (if the type is an array), or gt, gte, lt, lte for range values.
     */
    nonindexedinstancemd: string[];
}
export { ObjectMetadataInterface };
/**
 * The PartInstanceInfoInterface interface is used to store properties about a `part instance` and its genealogy.
 *
 * The PartInstanceInfoInterface is retrieved when using a {@link IdCardGetterInterface.retrieveIdCard} i.e. when the metadata
 * attached to a `part instance` are retrieved.
 *
 * The interface provides information about
 * * an instantiation chain /root/.../grandparent/parent/instance with the methods
 * {@link getAncestors}, {@link getAncestorsStatusFlags}, {@link getAttachedDocumentInfos}, {@link getWorldXForms}, {@link getIdCardHierarchy}
 * * and the specific instance with the methods {@link getInstanceId}, {@link isInstanceLeaf}, {@link getObbDiagonalSquared}, {@link getAABB}.
 *
 * The following code snippet allows to get the metadata of a `part instance` by the {@link IdCardGetterInterface}.
 * Such a code may be triggered while receiving the {@link IdCardGetterInterfaceSignal.IdCardReady} of the {@link IdCardGetterInterface}.
 *
 * **WARNING** : there is a unique instantiation chain from the root to a `part instance`, as such, if you query multiple `part instances`
 * when using {@link IdCardGetterInterface.retrieveIdCard}, {@link IdCardGetterInterface.getPartInstanceInfos} will return an array of
 * PartInstanceInfoInterface of the same size.
 *
 * ```typescript
 * [[include:IdCardGetterInterface_example.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:IdCardGetterInterface_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link IdCardGetterInterface}
 * @see {@link IdCardGetterInterfaceSignal}
 * @see {@link PartInstanceInfoStatusFlag}
 * @see {@link ObjectMetadataInterface}
 * @category Data Retrievers
 */
interface PartInstanceInfoInterface {
    /**
     * Gets the `part instance ids` from the root to this `part instance` /root/.../grandparent/parent/instance.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The array of <code>part instance ids</code> from the root to this
     * <code>part instance</code>.
     */
    getAncestors(): Array<number>;
    /**
     * Gets the minimum part instance id of the leaves that represent the node/leaf part instance for the full hierarchy.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * The given part instance at depth `i` is therefore represented by the leaves part instances :
     *
     * `getSubLeafMinimums()[i], getSubLeafMinimums()[i] + 1, getSubLeafMinimums()[i] + 2, ..., getSubLeafMaximums()[i]`.
     *
     * @return <div class="const">const</div> The minimum part instance leaf of the full hierarchy.
     */
    getSubLeafMinimums(): Array<number>;
    /**
     * Gets the maximum part instance id of the leaves that represent the node/leaf part instance for the full hierarchy.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * The given part instance at depth `i` is therefore represented by the leaves part instances :
     *
     * `getSubLeafMinimums()[i], getSubLeafMinimums()[i] + 1, getSubLeafMinimums()[i] + 2, ..., getSubLeafMaximums()[i]`.
     *
     * @return <div class="const">const</div> The maximum part instance leaf of the full hierarchy.
     */
    getSubLeafMaximums(): Array<number>;
    /**
     * Gets the `part instance class ids` from the root to this `part instance` /root/.../grandparent/parent/instance.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The array of <code>part instance class ids</code> from the root to this
     * <code>part instance</code>.
     */
    getAncestorsClassIds(): Array<number>;
    /**
     * Gets the status flags from the root to this `part instance` : /root/.../grandparent/parent/instance.
     *
     * The status flags are the OR of {@link PartInstanceInfoStatusFlag}.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The array of status flags from the root to this <code>part instance</code>.
     * @see {@link PartInstanceInfoStatusFlag}
     */
    getAncestorsStatusFlags(): Array<number>;
    /**
     * Gets the list of the attached documents from the root to this `part instance` : /root/.../grandparent/parent/instance.
     *
     * This gets an array of array of AttachedDocumentInfoInterface.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of attached documents properties from the root to this <code>part instance</code>.
     * @see {@link AttachedDocumentInfoInterface}
     */
    getAttachedDocumentInfos(): Array<Array<AttachedDocumentInfoInterface>>;
    /**
     * Gets the list of the annotation views properties from the root to this `part instance` : /root/.../grandparent/parent/instance.
     *
     * This gets an array of array of AnnotationViewInfoInterface.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of annotation views properties from the root to this <code>part instance</code>.
     * @see {@link AnnotationViewInfoInterface}
     */
    getAnnotationViews(): Array<Array<AnnotationViewInfoInterface>>;
    /**
     * Gets the world transformations from the root to this `part instance` : /root/.../grandparent/parent/instance.
     *
     * The Matrix4 follows the column multiplication convention (see {@link Matrix4}).
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of Matrix4 from the root to this <code>part instance</code>.
     * @see {@link Matrix4}
     */
    getWorldXForms(): Array<Matrix4>;
    /**
     * Gets all the metadata from the root to this `part instance` : /root/.../grandparent/parent/instance.
     *
     * DO NOT modify in place this object, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The metadata from the root to this <code>part instance</code>.
     * @see {@link ObjectMetadataInterface}
     */
    getIdCardHierarchy(): ObjectMetadataInterface;
    /**
     * Gets the id of the `part instance` that was queried by {@link IdCardGetterInterface.retrieveIdCard}.
     *
     * @return The <code>part instance id</code> this object refers to.
     * @see {@link IdCardGetterInterface.retrieveIdCard}
     */
    getInstanceId(): number;
    /**
     * Tells if this `part instance` is a leaf, which means it has no child.
     *
     * @return `true` if the <code>part instance</code> this object refers to is a leaf.
     */
    isInstanceLeaf(): boolean;
    /**
     * Gets the square of the diagonal of the Oriented Bounding Box of this `part instance`.
     *
     * @return The square of the diagonal of the Oriented Bounding Box of this <code>part instance</code>.
     */
    getObbDiagonalSquared(): number;
    /**
     * Gets the Axis Aligned Bounding Box of the `part instance` this
     * object refers to.
     *
     * DO NOT modify in place this object, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The Axis Aligned BoundingBox of this <code>part instance</code>.
     * @see {@link AABB}
     */
    getAABB(): AABB;
}
export { PartInstanceInfoInterface };
/**
 * The ChildrenPartInstanceInfoInterface interface is used to store properties about the children of a `part instance`.
 *
 * The ChildrenPartInstanceInfoInterface is retrieved when using a {@link ChildrenIdCardGetterInterface.retrieveChildrenIdCard}.
 *
 * The interface provides information about
 * * a children list with the methods
 * {@link getChildren}, {@link getChildrenStatusFlags}, {@link getAttachedDocumentInfos}, {@link getWorldXForms}, {@link getChildrenIdCard}.
 * * and the specific instance with the methods {@link getInstanceId}, {@link isInstanceLeaf}, {@link getObbDiagonalSquared}, {@link getAABB}.
 *
 * The following code snippet allows to get the metadata of the children `part instances` by the {@link ChildrenIdCardGetterInterface}.
 * Such a code may be triggered while receiving the {@link IdCardGetterInterfaceSignal.IdCardReady} of the {@link ChildrenIdCardGetterInterface}.
 *
 *
 * ```typescript
 * [[include:ChildrenIdCardGetterInterface_example.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:ChildrenIdCardGetterInterface_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link ChildrenIdCardGetterInterface}
 * @see {@link IdCardGetterInterfaceSignal}
 * @see {@link PartInstanceInfoStatusFlag}
 * @see {@link ObjectMetadataInterface}
 * @category Data Retrievers
 */
interface ChildrenPartInstanceInfoInterface {
    /**
     * Gets the `part instance ids` of the children of this `part instance`.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The array of children <code>part instance ids</code>.
     */
    getChildren(): Array<number>;
    /**
     * Tells if the children of this `part instance` are leaves.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The array of children leaf information.
     */
    areChildrenLeaves(): Array<boolean>;
    /**
     * Gets the minimum part instance id of the leaves that represent the node/leaf part instance for the full children list.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * The `i`th children part instance is therefore represented by the leaves part instances :
     *
     * `getSubLeafMinimums()[i], getSubLeafMinimums()[i] + 1, getSubLeafMinimums()[i] + 2, ..., getSubLeafMaximums()[i]`.
     *
     * @return <div class="const">const</div> The minimum part instance leaf of all the children.
     */
    getSubLeafMinimums(): Array<number>;
    /**
     * Gets the maximum part instance id of the leaves that represent the node/leaf part instance for the full children list.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * The `i`th children part instance is therefore represented by the leaves part instances :
     *
     * `getSubLeafMinimums()[i], getSubLeafMinimums()[i] + 1, getSubLeafMinimums()[i] + 2, ..., getSubLeafMaximums()[i]`.
     *
     * @return <div class="const">const</div> The maximum part instance leaf of all the children.
     */
    getSubLeafMaximums(): Array<number>;
    /**
     * Gets the `part instance class ids` of the children of this `part instance`.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The array of children <code>part instance class ids</code>.
     */
    getChildrenClassIds(): Array<number>;
    /**
     * Gets the status flags of the children of this `part instance`.
     *
     * The status flags are the OR of {@link PartInstanceInfoStatusFlag}.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The array of children status flags of this <code>part instance</code>.
     * @see {@link PartInstanceInfoStatusFlag}
     */
    getChildrenStatusFlags(): Array<number>;
    /**
     * Gets the list of the attached documents of the children of this `part instance`.
     *
     * This gets an array of array of AttachedDocumentInfoInterface.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of attached documents properties of the children of this <code>part instance</code>.
     * @see {@link AttachedDocumentInfoInterface}
     */
    getAttachedDocumentInfos(): Array<Array<AttachedDocumentInfoInterface>>;
    /**
     * Gets the list of the annotation views properties of the children of this `part instance`.
     *
     * This gets an array of array of AnnotationViewInfoInterface.
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of annotation views properties of the children of this <code>part instance</code>.
     * @see {@link AnnotationViewInfoInterface}
     */
    getAnnotationViews(): Array<Array<AnnotationViewInfoInterface>>;
    /**
     * Gets the world transformations of the children of this `part instance`.
     *
     * The Matrix4 follows the column multiplication convention (see {@link Matrix4}).
     *
     * DO NOT modify in place this array, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of Matrix4 of the children of this <code>part instance</code>.
     * @see {@link Matrix4}
     */
    getWorldXForms(): Array<Matrix4>;
    /**
     * Gets all the metadata of the children of this `part instance`.
     *
     * DO NOT modify in place this object, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The metadata of the children of this <code>part instance</code>.
     * @see {@link ObjectMetadataInterface}
     */
    getChildrenIdCard(): ObjectMetadataInterface;
    /**
     * Gets the id of the `part instance` that was queried by {@link ChildrenIdCardGetterInterface.retrieveChildrenIdCard}.
     *
     * @return The <code>part instance id</code> this object refers to.
     * @see {@link ChildrenIdCardGetterInterface.retrieveChildrenIdCard}
     */
    getInstanceId(): number;
    /**
     * Tells if this `part instance` is a leaf, which means it has no child (in this case, arrays from {@link getChildrenIdCard}, {@link getWorldXForms} etc are empty).
     *
     * @return `true` if the <code>part instance</code> this object refers to is a leaf.
     */
    isInstanceLeaf(): boolean;
    /**
     * Gets the square of the diagonal of the Oriented Bounding Box of this `part instance`.
     *
     * @return The square of the diagonal of the Oriented Bounding Box of this <code>part instance</code>.
     */
    getObbDiagonalSquared(): number;
    /**
     * Gets the Axis Aligned Bounding Box of the `part instance` this
     * object refers to.
     *
     * DO NOT modify in place this object, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The Axis Aligned BoundingBox of this <code>part instance</code>.
     * @see {@link AABB}
     */
    getAABB(): AABB;
}
export { ChildrenPartInstanceInfoInterface };
/**
 * The ChildrenIdCardGetterInterfaceSignal lists all the signals that may be sent by the {@link ChildrenIdCardGetterInterface}.
 *
 * These signals are emitted by the {@link ChildrenIdCardGetterInterface} when some data is available.
 *
 * * {@link ChildrenIdCardReady} : the {@link ChildrenIdCardGetterInterface} is ready, i.e. when it has finished processing its `part instance ids`,
 * {@link ChildrenIdCardGetterInterface.getPartInstanceInfos} can be called.
 * * {@link ChildrenIdCardCancelled} : the update (running) of the {@link ChildrenIdCardGetterInterface} has been cancelled.
 *
 * ```typescript
 * [[include:ChildrenIdCardGetterInterface_example.ts]]
 * ```
 * <br>
 * Please refer to {@link ChildrenIdCardGetterInterface} for more information about children idcard requests.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link ChildrenIdCardGetterInterface}
 * @category Events
 */
declare enum ChildrenIdCardGetterInterfaceSignal {
    /**
     * Signal sent by the {@link ChildrenIdCardGetterInterface} when the {@link ChildrenIdCardGetterInterface} is ready,
     * i.e. {@link ChildrenIdCardGetterInterface.getPartInstanceInfos} can be called.
     *
     * Warning, the ChildrenIdCardReady signal is sent when {@link ChildrenIdCardGetterInterface.cancel} is called and the {@link ChildrenIdCardGetterInterface}
     * is updating. In such a case,
     * not data will be available in the {@link ChildrenIdCardGetterInterface}.
     *
     * The attachment is a string telling the request id of the call to {@link ChildrenIdCardGetterInterface.retrieveChildrenIdCard} ({@link ChildrenIdCardGetterInterface.getLastRequestId}).
     */
    ChildrenIdCardReady = "ready",
    /**
     * Signal sent by the {@link ChildrenIdCardGetterInterface} when the {@link ChildrenIdCardGetterInterface}
     * is requested to be cancelled.
     *
     * Such an event is fired if the {@link ChildrenIdCardGetterInterface} was running and {@link ChildrenIdCardGetterInterface.cancel}
     * is called.
     * The {@link ChildrenIdCardReady} signal will be triggered shortly after.
     *
     * No attachment for events of this type.
     */
    ChildrenIdCardCancelled = "cancelled"
}
export { ChildrenIdCardGetterInterfaceSignal };
/**
 * The IdCardGetterInterfaceSignal lists all the signals that may be sent by the {@link IdCardGetterInterface}.
 *
 * These signals are emitted by the {@link IdCardGetterInterface} when some data is available.
 *
 * * {@link IdCardReady} : the {@link IdCardGetterInterface} is ready, i.e. when it has finished processing its `part instance ids`,
 * {@link IdCardGetterInterface.getPartInstanceInfos} can be called.
 * * {@link IdCardCancelled} : the update (running) of the {@link IdCardGetterInterface} has been cancelled.
 *
 * ```typescript
 * [[include:IdCardGetterInterface_example.ts]]
 * ```
 * <br>
 * Please refer to {@link IdCardGetterInterface} for more information about idcard requests.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link IdCardGetterInterface}
 * @category Events
 */
declare enum IdCardGetterInterfaceSignal {
    /**
     * Signal sent by the {@link IdCardGetterInterface} when the {@link IdCardGetterInterface} is ready,
     * i.e. {@link IdCardGetterInterface.getPartInstanceInfos} can be called.
     *
     * Warning, the IdCardReady signal is sent when {@link IdCardGetterInterface.cancel} is called and the {@link IdCardGetterInterface}
     * is updating. In such a case,
     * not data will be available in the {@link IdCardGetterInterface}.
     *
     * The attachment is a string telling the request id of the call to {@link IdCardGetterInterface.retrieveIdCard} ({@link IdCardGetterInterface.getLastRequestId}).
     */
    IdCardReady = "ready",
    /**
     * Signal sent by the {@link IdCardGetterInterface} when the {@link IdCardGetterInterface}
     * is requested to be cancelled.
     *
     * Such an event is fired if the {@link IdCardGetterInterface} was running and {@link IdCardGetterInterface.cancel}
     * is called.
     * The {@link IdCardReady} signal will be triggered shortly after.
     *
     * No attachment for events of this type.
     */
    IdCardCancelled = "cancelled"
}
export { IdCardGetterInterfaceSignal };
/**
 * Represents an id card request promise result.
 *
 * [[include:constStyle.md]]<br><div class="const">Converters</div><br>
 * @see {@link IdCardGetterInterface.asyncRetrieveIdCard}
 * @category Converters
 */
interface AsyncPartInstanceInfoResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     *
     * @see {@link PartInstanceInfoInterface}
     * @see {@link IdCardGetterInterface.asyncRetrieveIdCard}
     */
    value?: Array<PartInstanceInfoInterface>;
}
export { AsyncPartInstanceInfoResult };
/**
 * The IdCardGetterInterface interface is used to retrieve information about a `part instance` (and not a `geometric instance id`),
 * such as the metadata of the `part`, link, the list of `part instance` ancestors,
 * attached document, ....
 *
 * The `metadata documents` and general information can be retrieved by knowing the `part instance` to retrieve. If such a document
 * is requested after a picking, then the given `geometric instance id` must be converted to a `part instance id`.
 *
 * The IdCardGetterInterface interfaces are created through the {@link DataSessionInterface.createIdCardGetter} method.
 *
 * ```typescript
 * [[include:IdCardGetterInterface_example_picking.ts]]
 * ```
 * <br>
 * The list of signals the IdCardGetterInterface may trigger is available in the {@link IdCardGetterInterfaceSignal} enumeration.
 *
 * The metadata retrieval procedure is triggered through the {@link retrieveIdCard} methods. The result is not available right away, but the
 * event {@link IdCardGetterInterfaceSignal.IdCardReady} is triggered when the result of the IdCardGetterInterface is available.
 * The result is available through the [getPartInstanceInfos]] function.
 *
 * Warning : there may be cases when the {@link getPartInstanceInfos} is not available such as when the IdCardGetterInterface is updating, i.e. when
 * {@link isRunning} returns true, or when the IdCardGetterInterface has been cancelled, i.e. when {@link isCancelled} returns true.
 *
 * An IdCardGetterInterface may be interrupted (cancelled) when the IdCardGetterInterface is updating and {@link cancel}
 * is called. In such cases, the {@link IdCardGetterInterfaceSignal.IdCardCancelled} signal is fired, and shortly after, {@link IdCardGetterInterfaceSignal.IdCardReady}
 * signal is fired, but {@link getPartInstanceInfos} will return undefined. Just call {@link retrieveIdCard} with another (or the same) `part instance ids` to trigger a new
 * retrieval.
 *
 * If you call multiple times {@link retrieveIdCard} before receiving the {@link IdCardGetterInterfaceSignal.IdCardReady}, only one {@link IdCardGetterInterfaceSignal.IdCardReady}
 * will be sent with the content of the last call to {@link retrieveIdCard}.
 *
 * The developer may process the results of an id card retrieval with the following code :
 *
 * ```typescript
 * [[include:IdCardGetterInterface_example.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:IdCardGetterInterface_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link IdCardGetterInterfaceSignal}
 * @see {@link PartInstanceInfoInterface}
 * @see {@link ObjectMetadataInterface}
 * @category Data Retrievers
 */
interface IdCardGetterInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Gets the information about the specified `part instance ids` and their genealogy.
     *
     * The {@link IdCardGetterInterfaceSignal.IdCardReady} signal is fired when the
     * `part instance ids` metadata result is ready.
     *
     * Returns `true` if the metadata procedure retrieval is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed. For instance, pPartInstanceIds is considered as invalid input if it is empty
     * or at least one element is out of range [1 : 2^31-1].
     *
     * If pFilteringContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pPartInstanceIds <div class="const">in</div> The list of <code>part instance ids</code> to fetch metadata from.
     * @param pFilteringContext <div class="const">in</div> The current filtering context for configured metadata.
     * @return `true` if the retrieval procedure has begun.
     * @see {@link IdCardGetterInterfaceSignal}
     */
    retrieveIdCard(pPartInstanceIds: Uint32Array | number[], pFilteringContext: VisibilityContextInterface): boolean;
    /**
     * Asynchronously gets the information about the specified `part instance ids` and their genealogy.
     *
     * Returns a promise.
     *
     * Please note that in case of multiple promises running at the same time on the same IdCardGetterInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getPartInstanceInfos}
     * after awaiting will return the same value.
     *
     * If pFilteringContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pPartInstanceIds <div class="const">in</div> The list of <code>part instance ids</code> to fetch metadata from.
     * @param pFilteringContext <div class="const">in</div> The current filtering context for configured metadata.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the requested Array<PartInstanceInfoInterface>.
     */
    asyncRetrieveIdCard(pPartInstanceIds: Uint32Array | number[], pFilteringContext: VisibilityContextInterface): Promise<AsyncPartInstanceInfoResult>;
    /**
     * Asynchronously gets the information about the specified `part instance id` and its genealogy.
     *
     * Returns a promise.
     *
     * Please note that in case of multiple promises running at the same time on the same IdCardGetterInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getPartInstanceInfos}
     * after awaiting will return the same value.
     *
     * If pFilteringContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pPartInstanceId <div class="const">in</div> The <code>part instance id</code> to fetch metadata from.
     * @param pFilteringContext <div class="const">in</div> The current filtering context for configured metadata.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the requested Array<PartInstanceInfoInterface>.
     */
    asyncRetrieveIdCard(pPartInstanceId: number, pFilteringContext: VisibilityContextInterface): Promise<AsyncPartInstanceInfoResult>;
    /**
     * Gets the information about the specified `part instance id` and its genealogy.
     *
     * The event IdCardGetterInterfaceSignal.IdCardReady is fired when the
     * `part instance id` metadata result is ready.
     *
     * If pFilteringContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * Returns `true` if the metadata procedure retrieval is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed. For instance, pPartInstanceIds is considered as invalid input if it is empty
     * or at least one element is out of range [1 - 2^31-1].
     *
     * @param pPartInstanceId <div class="const">in</div> The <code>part instance id</code> to fetch metadata from.
     * @param pFilteringContext <div class="const">in</div> The current filtering context for configured metadata.
     * @return `true` if the retrieval procedure has begun.
     * @see {@link IdCardGetterInterfaceSignal}
     */
    retrieveIdCard(pPartInstanceId: number, pFilteringContext: VisibilityContextInterface): boolean;
    /**
     * Cancels the computation of the metadata retrieving process (if any).
     *
     * A {@link IdCardGetterInterfaceSignal.IdCardCancelled} signal is emitted if the
     * IdCardGetterInterface is retrieving data.
     *
     * @return `true` if the DocumentIdConverterInterface was running, else false.
     * @see {@link IdCardGetterInterfaceSignal.IdCardCancelled}
     */
    cancel(): boolean;
    /**
     * Tells if the IdCardGetterInterface is updating.
     *
     * This is the case after calling {@link retrieveIdCard}.
     *
     * @return `true` if the IdCardGetterInterface is updating.
     */
    isRunning(): boolean;
    /**
     * Tells if the IdCardGetterInterface has been cancelled.
     *
     * This is generally the case after calling {@link cancel} when the IdCardGetterInterface
     * is retrieving data.
     *
     * @return `true` if the IdCardGetterInterface is cancelled.
     */
    isCancelled(): boolean;
    /**
     * Gets the last error returned by the update of the IdCardGetterInterface.
     *
     * @return The last error message (if any, or an empty string if no error occurred).
     */
    getLastError(): string;
    /**
     * Each call to {@link retrieveIdCard} is assigned a request id.
     *
     * This call tels the id of the last call to {@link retrieveIdCard}.
     *
     * @return The id of the last call to {@link retrieveIdCard}.
     */
    getLastRequestId(): string;
    /**
     * Gets the result of the metadata request.
     *
     * The metadata content is returned if the IdCardGetterInterface has finished computing.
     * Use {@link addEventListener} on the event {@link IdCardGetterInterfaceSignal.IdCardReady} to
     * know when the IdCardGetterInterface is ready.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> the metadata of the requested <code>part instance ids</code> and their genealogy (the size of the array is the number of
     * <code>part instance ids</code> requested (see {@link retrieveIdCard})), or
     * undefined if the IdCardGetterInterface is computing or if the IdCardGetterInterface is in error or cancelled.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     * @see {@link PartInstanceInfoInterface}
     */
    getPartInstanceInfos(): Array<PartInstanceInfoInterface> | undefined;
}
export { IdCardGetterInterface };
/**
 * Represents an children id card request promise result.
 *
 * [[include:constStyle.md]]<br><div class="const">Converters</div><br>
 * @see {@link ChildrenIdCardGetterInterface.asyncRetrieveChildrenIdCard}
 * @category Converters
 */
interface AsyncChildrenPartInstanceInfoResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     *
     * @see {@link ChildrenPartInstanceInfoInterface}
     * @see {@link ChildrenIdCardGetterInterface.asyncRetrieveChildrenIdCard}
     */
    value?: Array<ChildrenPartInstanceInfoInterface>;
}
export { AsyncChildrenPartInstanceInfoResult };
/**
 * The ChildrenIdCardGetterInterface interface is used to retrieve information about the children of a `part instance` (and not a `geometric instance id`),
 * such as the metadata of the children `parts`, link, the list of children attached document, ....
 *
 * The `metadata documents` and general information about the children can be retrieved by knowing the `part instance id` of the parent to retrieve. Such documents
 * are usually requested after an idcard request.
 *
 * The ChildrenIdCardGetterInterface interfaces are created through the {@link DataSessionInterface.createChildrenIdCardGetter} method.
 *
 * ```typescript
 * [[include:ChildrenIdCardGetterInterface_example_idcard.ts]]
 * ```
 * <br>
 * The list of signals the ChildrenIdCardGetterInterface may trigger is available in the {@link ChildrenIdCardGetterInterfaceSignal} enumeration.
 *
 * The metadata retrieval procedure is triggered through the {@link retrieveChildrenIdCard} methods. The result is not available right away, but the
 * event {@link ChildrenIdCardGetterInterfaceSignal.ChildrenIdCardReady} is triggered when the result of the ChildrenIdCardGetterInterface is available.
 * The result is available through the [getPartInstanceInfos]] function.
 *
 * Warning : there may be cases when the {@link getPartInstanceInfos} is not available such as when the ChildrenIdCardGetterInterface is updating, i.e. when
 * {@link isRunning} returns true, or when the ChildrenIdCardGetterInterface has been cancelled, i.e. when {@link isCancelled} returns true.
 *
 * An ChildrenIdCardGetterInterface may be interrupted (cancelled) when the ChildrenIdCardGetterInterface is updating and {@link cancel}
 * is called. In such cases, the {@link ChildrenIdCardGetterInterfaceSignal.ChildrenIdCardCancelled} signal is fired, and shortly after,
 * {@link ChildrenIdCardGetterInterfaceSignal.ChildrenIdCardReady}
 * signal is fired, but {@link getPartInstanceInfos} will return undefined. Just call {@link retrieveChildrenIdCard} with another (or the same) `part instance ids` to trigger a new
 * retrieval.
 *
 * If you call multiple times {@link retrieveChildrenIdCard} before receiving the {@link ChildrenIdCardGetterInterfaceSignal.ChildrenIdCardReady}, only one
 * {@link ChildrenIdCardGetterInterfaceSignal.ChildrenIdCardReady}
 * will be sent with the content of the last call to {@link retrieveChildrenIdCard}.
 *
 * The developer may process the results of an id card retrieval with the following code :
 *
 * ```typescript
 * [[include:ChildrenIdCardGetterInterface_example.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:ChildrenIdCardGetterInterface_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link ChildrenIdCardGetterInterfaceSignal}
 * @see {@link ChildrenPartInstanceInfoInterface}
 * @see {@link ObjectMetadataInterface}
 * @category Data Retrievers
 */
interface ChildrenIdCardGetterInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Gets the information about the children of the specified `part instance ids`.
     *
     * The {@link ChildrenIdCardGetterInterfaceSignal.ChildrenIdCardReady} signal is fired when the
     * `part instance ids` metadata result is ready.
     *
     * Returns true if the metadata procedure retrieval is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed. For instance, pPartInstanceIds is considered as invalid input if it is empty
     * or at least one element is out of range [1 : 2^31-1].
     *
     * If pFilteringContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pPartInstanceIds <div class="const">in</div> The list of <code>part instance ids</code> to fetch metadata from.
     * @param pFilteringContext <div class="const">in</div> The current filtering context for configured metadata.
     * @return `true` if the retrieval procedure has begun.
     * @see {@link ChildrenIdCardGetterInterfaceSignal}
     */
    retrieveChildrenIdCard(pPartInstanceIds: Uint32Array | number[], pFilteringContext: VisibilityContextInterface): boolean;
    /**
     * Asynchronously gets the information about the children of the specified `part instance ids`.
     *
     * Returns a promise.
     *
     * If pFilteringContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * Please note that in case of multiple promises running at the same time on the same ChildrenIdCardGetterInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getPartInstanceInfos}
     * after awaiting will return the same value.
     *
     * @param pPartInstanceIds <div class="const">in</div> The list of <code>part instance ids</code> to fetch metadata from.
     * @param pFilteringContext <div class="const">in</div> The current filtering context for configured metadata.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise is resolved with an Array<PartInstanceInfoInterface>.
     */
    asyncRetrieveChildrenIdCard(pPartInstanceIds: Uint32Array | number[], pFilteringContext: VisibilityContextInterface): Promise<AsyncChildrenPartInstanceInfoResult>;
    /**
     * Gets the information about the children of the specified `part instance id`.
     *
     * The event {@link ChildrenIdCardGetterInterfaceSignal.ChildrenIdCardReady} is fired when the
     * `part instance id` metadata result is ready.
     *
     * Returns `true` if the metadata procedure retrieval is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed. For instance, pPartInstanceIds is considered as invalid input if it is empty
     * or at least one element is out of range [1 - 2^31-1].
     *
     * If pFilteringContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pPartInstanceId <div class="const">in</div> The <code>part instance id</code> to fetch metadata from.
     * @param pFilteringContext <div class="const">in</div> The current filtering context for configured metadata.
     * @return `true` if the retrieval procedure has begun.
     * @see {@link ChildrenIdCardGetterInterfaceSignal}
     */
    retrieveChildrenIdCard(pPartInstanceId: number, pFilteringContext: VisibilityContextInterface): boolean;
    /**
     * Asynchronously gets the information about the children of the specified `part instance id`.
     *
     * Returns a promise.
     *
     * Please note that in case of multiple promises running at the same time on the same ChildrenIdCardGetterInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getPartInstanceInfos}
     * after awaiting will return the same value.
     *
     * If pFilteringContext is modified during the execution, then the call is cancelled (see {@link cancel}).
     *
     * @param pPartInstanceId <div class="const">in</div> The <code>part instance id</code> to fetch metadata from.
     * @param pFilteringContext <div class="const">in</div> The current filtering context for configured metadata.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise is resolved with an Array<PartInstanceInfoInterface>.
     */
    asyncRetrieveChildrenIdCard(pPartInstanceId: number, pFilteringContext: VisibilityContextInterface): Promise<AsyncChildrenPartInstanceInfoResult>;
    /**
     * Each call to {@link retrieveChildrenIdCard} is assigned a request id.
     *
     * This call tels the id of the last call to {@link retrieveChildrenIdCard}.
     *
     * @return The id of the last call to {@link retrieveChildrenIdCard}.
     */
    getLastRequestId(): string;
    /**
     * Cancels the computation of the metadata retrieving process (if any).
     *
     * A {@link ChildrenIdCardGetterInterfaceSignal.ChildrenIdCardCancelled} signal is emitted if the
     * ChildrenIdCardGetterInterface is retrieving data.
     *
     * @return `true` if the DocumentIdConverterInterface was running, else false.
     * @see {@link IdCardGetterInterfaceSignal.IdCardCancelled}
     */
    cancel(): boolean;
    /**
     * Tells if the ChildrenIdCardGetterInterface is updating.
     *
     * This is the case after calling {@link retrieveChildrenIdCard}.
     *
     * @return `true` if the ChildrenIdCardGetterInterface is updating.
     */
    isRunning(): boolean;
    /**
     * Tells if the ChildrenIdCardGetterInterface has been cancelled.
     *
     * This is generally the case after calling {@link cancel} when the ChildrenIdCardGetterInterface
     * is retrieving data.
     *
     * @return `true` if the ChildrenIdCardGetterInterface is cancelled.
     */
    isCancelled(): boolean;
    /**
     * Gets the last error returned by the update of the ChildrenIdCardGetterInterface.
     *
     * @return The last error message (if any, or an empty string if no error occurred).
     */
    getLastError(): string;
    /**
     * Gets the result of the metadata request.
     *
     * The metadata content is returned if the ChildrenIdCardGetterInterface has finished computing.
     * Use {@link addEventListener} on the event {@link ChildrenIdCardGetterInterfaceSignal.ChildrenIdCardReady} to
     * know when the ChildrenIdCardGetterInterface is ready.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> the metadata of the requested <code>part instance ids</code> and their genealogy (the size of the array is the number of
     * <code>part instance ids</code> requested (see {@link retrieveChildrenIdCard})), or
     * undefined if the ChildrenIdCardGetterInterface is computing or if the ChildrenIdCardGetterInterface is in error or cancelled.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     * @see {@link ChildrenPartInstanceInfoInterface}
     */
    getPartInstanceInfos(): Array<ChildrenPartInstanceInfoInterface> | undefined;
}
export { ChildrenIdCardGetterInterface };
/**
 * The InfiniteApiControllerInterfaceSignal lists all the signals that may be sent by the {@link InfiniteApiControllerInterface}.
 *
 * This signal is emitted by the {@link InfiniteApiControllerInterface} when the following occurs.
 * The 3djuump infinite API features an internal clock that sets the pace of the whole API : objects are updated at
 * fixed intervals (all at once).
 * The InfiniteApiControllerInterfaceSignal.Tick signal is sent at the end of the modification of the objects.
 *
 * Beware, the {@link InfiniteEngineInterfaceSignal.DisplayDone} signal is not sent just after the update of the objects, but when
 * the image content is displayed on the view (`window.requestAnimationFrame`).
 *
 * If you want to synchronize a behavior when the objects are updated, then use {@link InfiniteApiControllerInterfaceSignal.Tick}.
 *
 * If you want to synchronize a behavior when the display is updated, then use {@link InfiniteEngineInterfaceSignal.DisplayDone}.
 *
 * ```typescript
 * [[include:InfiniteApiControllerInterface_signal.ts]]
 * ```
 * <br>
 * Please refer to {@link InfiniteApiControllerInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @see {@link InfiniteApiControllerInterface}
 * @see {@link InfiniteEngineInterfaceSignal}
 * @see {@link InfiniteEngineInterface}
 * @category Main
 */
declare enum InfiniteApiControllerInterfaceSignal {
    /**
     * Signal sent by the {@link InfiniteApiControllerInterface} when the infinite objects have finished being updated.
     * This signal is emitted on a regular basis.
     *
     * No attachment for events of this type.
     */
    Tick = "tick"
}
export { InfiniteApiControllerInterfaceSignal };
/**
 * The LogLevel enumeration lists the available log levels to be use with the {@link InfiniteApiControllerInterface}.
 *
 * Adding a log item is done with a criticality, and if the current log level (as set by {@link InfiniteApiControllerInterface.setLogLevel})
 * is strictly superior to the criticality of the log item, the log item is discarded.
 *
 * You may also install a custom log behavior on a specific criticality with {@link InfiniteApiControllerInterface.setLogLevelBehavior}.
 *
 * ```typescript
 * [[include:InfiniteApiControllerInterface_log.ts]]
 * ```
 * <br>
 * The default log level is {@link LL_UsingDisposedObject}.
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @see {@link InfiniteApiControllerInterface.setLogLevel}
 * @category Main
 */
declare enum LogLevel {
    /**
     * Debug Log level.
     *
     * Default behavior is a console.log.
     */
    LL_Debug = 0,
    /**
     * A message is outputted when a disposed object is used.
     *
     * Default behavior is a console.log.
     */
    LL_UsingDisposedObject = 1,
    /**
     * Information log level.
     *
     * Default behavior is a console.log.
     */
    LL_Information = 2,
    /**
     * Warning log level.
     *
     * Default behavior is a console.warn.
     */
    LL_Warning = 3,
    /**
     * Critical log level.
     *
     * Default behavior is a console.warn.
     */
    LL_Critical = 4,
    /**
     * Required log level.
     *
     * Log items on this type cannot be overridden, nor discarded.
     *
     * Behavior is a console.log.
     */
    LL_Required = 5
}
export { LogLevel };
/**
 * The LogBehavior enumeration lists the available log level behaviors to be used with the {@link InfiniteApiControllerInterface}.
 *
 * When using {@link InfiniteApiControllerInterface.setLogLevelBehavior}, the default log behavior may be overridden for a specific
 * criticality.
 *
 * ```typescript
 * [[include:InfiniteApiControllerInterface_log.ts]]
 * ```
 * <br>
 * You may install an {@link InfiniteLogger} to get the information of the log item.
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @see {@link InfiniteApiControllerInterface.setLogLevelBehavior}
 * @see {@link InfiniteLogger}
 * @category Main
 */
declare enum LogBehavior {
    /**
     * The log item will be discarded (no-op).
     */
    LB_None = 0,
    /**
     * Behavior is a console.log.
     */
    LB_ConsoleLog = 1,
    /**
     * Behavior is a console.warn.
     */
    LB_ConsoleWarn = 2,
    /**
     * Behavior is to send the log item to an {@link InfiniteLogger}.
     */
    LB_LogToObject = 3,
    /**
     * Reset the log level to its default behavior.
     */
    LB_DefaultLog = 4
}
export { LogBehavior };
/**
 * The InfiniteLogger interface is used to customize the way
 * logging is handled.
 *
 * This interface is used with {@link InfiniteApiControllerInterface.setLogLevelBehavior}.
 * You may create your own class that implements the InfiniteLogger in order to
 * customize the behavior of the logging.
 *
 * ```typescript
 * [[include:InfiniteApiControllerInterface_log.ts]]
 * ```
 * <br>
 * Please refer to the {@link InfiniteApiControllerInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @see {@link InfiniteApiControllerInterface.setLogLevelBehavior}
 * @category Main
 */
interface InfiniteLogger {
    /**
     * Adds a log item.
     *
     * @param pContent <div class="const">in</div> The content of the log.
     * @param pDate <div class="const">in</div> The date of the log.
     * @param pLevel <div class="const">in</div> The level of the log.
     */
    addLog(pContent: string, pDate: Date, pLevel: LogLevel): void;
}
export { InfiniteLogger };
/**
 * The BrowserType enumeration lists the available browsers (as defined by <a href="https://www.npmjs.com/package/bowser">bowser</a>).
 *
 * The browser type is obtained from {@link InfiniteApiControllerInterface.getBrowserType}. This function tells the browser the api is running on.
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @category Main
 */
declare enum BrowserType {
    /**
     * Amazon Silk browser.
     */
    amazon_silk = 0,
    /**
     * Android browser.
     */
    android = 1,
    /**
     * Bada browser.
     */
    bada = 2,
    /**
     * Blackberry browser.
     */
    blackberry = 3,
    /**
     * Chrome browser.
     */
    chrome = 4,
    /**
     * Chromium based browser.
     */
    chromium = 5,
    /**
     * Electron browser.
     */
    electron = 6,
    /**
     * Epiphany browser.
     */
    epiphany = 7,
    /**
     * Firefox browser.
     */
    firefox = 8,
    /**
     * Focus browser.
     */
    focus = 9,
    /**
     * Generic browser (not found in the current enumeration).
     */
    generic = 10,
    /**
     * Google bot browser.
     */
    googlebot = 11,
    /**
     * Google search browser.
     */
    google_search = 12,
    /**
     * Internet Explorer browser.
     */
    ie = 13,
    /**
     * K-Meleon browser.
     */
    k_meleon = 14,
    /**
     * Maxthon browser.
     */
    maxthon = 15,
    /**
     * Edge browser.
     */
    edge = 16,
    /**
     * MZ browser.
     */
    mz = 17,
    /**
     * Naver Whale browser.
     */
    naver = 18,
    /**
     * Opera browser.
     */
    opera = 19,
    /**
     * Opera Coast browser.
     */
    opera_coast = 20,
    /**
     * PhantomJS browser.
     */
    phantomjs = 21,
    /**
     * Puffin browser.
     */
    puffin = 22,
    /**
     * Qupzilla/Falcon browser.
     */
    qupzilla = 23,
    /**
     * QQ browser.
     */
    qq = 24,
    /**
     * QQ browser Lite.
     */
    qqlite = 25,
    /**
     * Safari browser.
     */
    safari = 26,
    /**
     * Sailfish browser.
     */
    sailfish = 27,
    /**
     * Samsung browser.
     */
    samsung_internet = 28,
    /**
     * SeaMonkey browser.
     */
    seamonkey = 29,
    /**
     * Sleipnir browser.
     */
    sleipnir = 30,
    /**
     * Java swing based browser.
     */
    swing = 31,
    /**
     * Samsung Smart TV browser.
     */
    tizen = 32,
    /**
     * UC browser.
     */
    uc = 33,
    /**
     * Vivaldi browser.
     */
    vivaldi = 34,
    /**
     * LG TV Webos browser.
     */
    webos = 35,
    /**
     * WeChat for web browser.
     */
    wechat = 36,
    /**
     * Yandex for web browser.
     */
    yandex = 37
}
export { BrowserType };
/**
 * The InfiniteApiControllerInterface interface is the core singleton of the 3djuump infinite javascript API.
 *
 * It provides access to the following services :
 * * Font loading system.
 * * Logging System.
 * * Power saving management and 3djuump infinite internal clock.
 * * Browser detection.
 *
 * The Font loading mechanism is available through {@link InfiniteApiControllerInterface.getFontLoader}.
 *
 * The font loading mechanism may be done with the following :
 * ```typescript
 * [[include:FontLoaderInterface_loadFont.ts]]
 * ```
 * <br>
 * The logging system is based on string messages with a criticality and a timestamp (date). The 3djuump infinite javascript API
 * may log messages when key events occur.
 *
 * You may :
 * * Discard messages that have a strictly lower criticality than the logger current log level ({@link setLogLevel}, {@link getLogLevel}, {@link LogLevel}).
 * * Customize the way logging is done on a specific criticality ({@link setLogLevelBehavior}, {@link LogBehavior}, {@link InfiniteLogger}).
 * * Add log messages ({@link addLog}).
 *
 * ```typescript
 * [[include:InfiniteApiControllerInterface_log.ts]]
 * ```
 * <br>
 * The power saving system may be used to lower the CPU usage of the API when the browser is hidden. The 3djuump infinite javascript API
 * features an internal clock that ticks every 16 milliseconds if the browser has the focus, and every 1 second when the browser has not the focus and power
 * saving is enabled. When the internal clock ticks, the infinite objects are updated and calculated, and the
 * {@link InfiniteApiControllerInterfaceSignal.Tick} signal is sent when all objects have been updated.
 * Please note that the {@link InfiniteEngineInterfaceSignal.DisplayDone} signal is sent by the {@link InfiniteEngineInterface} when the **rendering** is done (this is not
 * at the same moment than {@link InfiniteApiControllerInterfaceSignal.Tick}).
 * Power saving is enabled with {@link setPowerSavingEnabled}.
 *
 * The browser detection is based on <a href="https://www.npmjs.com/package/bowser">bowser</a>.
 * At the start of the API, the running browser is detected and {@link getBrowserName}, {@link getBrowserType}, {@link getBrowserVersion} provide information
 * about the current browser. The browser type is a {@link BrowserType} that helps identify the current browser.
 *
 * **NB** : In case of Microsoft Edge, two versions exist, one is chromium based, the other not, and {@link isEdgeChromium} helps to disambiguate the two.
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @see {@link InfiniteApiControllerInterfaceSignal}
 * @see {@link FontLoaderInterface}
 * @see {@link LogLevel}
 * @see {@link LogBehavior}
 * @see {@link InfiniteLogger}
 * @see {@link BrowserType}
 * @category Main
 */
interface InfiniteApiControllerInterface extends EventDispatcherInterface {
    /**
     * Gets the font loader responsible to provide font data to the rendering of annotations.
     *
     * There is only one font loader.
     *
     * @return The resulting {@link FontLoaderInterface}.
     */
    getFontLoader(): FontLoaderInterface;
    /**
     * Enables/Disables power saving mode.
     *
     * Power saving mode is used to save CPU usage when the browser tab is hidden by
     * introducing some frame throttling. The {@link FilterSolverInterface} calculations
     * will be slower if the tab is hidden.
     *
     * Power saving is disabled (false) by default.
     *
     * @param pIsPowerSavingEnabled <div class="const">in</div> tells if the power saving mode should be enabled.
     */
    setPowerSavingEnabled(pIsPowerSavingEnabled: boolean): void;
    /**
     * Tells if power saving mode is enabled.
     *
     * Power saving mode is used to save CPU usage when the browser tab is hidden by
     * introducing some frame throttling.
     *
     * @return `true` if the power saving mode is enabled.
     */
    isPowerSavingEnabled(): boolean;
    /**
     * Gets the version of the infinite API.
     *
     * @return The version of the infinite API.
     */
    getVersion(): string;
    /**
     * Sets the log level of the logger.
     *
     * Any message with a criticality strictly lower than `pLogLevel` will be silently discarded.
     * You cannot set a criticality lower than {@link LogLevel.LL_Debug} and bigger than {@link LogLevel.LL_Required}.
     *
     * @param pLogLevel <div class="const">in</div> The new current log level.
     * @return `true` if pLogLevel is correct and that the log level has been set.
     */
    setLogLevel(pLogLevel: LogLevel): boolean;
    /**
     * Gets the current log level.
     *
     * This is the value previously set by {@link setLogLevel}.
     * Defaults to {@link LogLevel.LL_UsingDisposedObject}.
     *
     * @return The current log level.
     */
    getLogLevel(): LogLevel;
    /**
     * Customizes the way logging is handled for a specific criticality.
     *
     * `pLogLevel` must be superior or equal to {@link LogLevel.LL_Debug} and strictly inferior to {@link LogLevel.LL_Required}.
     * Thus, the {@link LogLevel.LL_Required} level cannot be customized.
     *
     * You may choose to :
     * * Discard the given log level ({@link LogBehavior.LB_None}).
     * * Send the given log level to the console ({@link LogBehavior.LB_ConsoleLog}, {@link LogBehavior.LB_ConsoleWarn}).
     * * Use a custom logger ({@link InfiniteLogger}) with {@link LogBehavior.LB_LogToObject}.
     * * Reset the log level to its default behavior ({@link LogBehavior.LB_DefaultLog}).
     *
     * When using {@link LogBehavior.LB_LogToObject}, `pLogObject` cannot be undefined, else pLogObject is ignored.
     *
     * @param pLogLevel <div class="const">in</div> The log level to override.
     * @param pBehavior <div class="const">in</div> The log behavior to use.
     * @param pLogObject <div class="const">in</div> The logger to use in case of {@link LogBehavior.LB_LogToObject}.
     * @return `true` if the log level has been set (i.e. pLogLevel and pBehavior are correct and pLogObject is defined in case of {@link LogBehavior.LB_LogToObject}).
     */
    setLogLevelBehavior(pLogLevel: LogLevel, pBehavior: LogBehavior, pLogObject?: InfiniteLogger): boolean;
    /**
     * Adds a log to be logged.
     *
     * The log timestamp is the current time of the call.
     *
     * @param pContent <div class="const">in</div> The log content.
     * @param pLevel <div class="const">in</div> The criticality of the log message.
     */
    addLog(pContent: string, pLevel: LogLevel): void;
    /**
     * Gets the current browser name.
     *
     * @return The current browser name.
     */
    getBrowserName(): string;
    /**
     * Gets the current browser type.
     *
     * @return The current browser type.
     */
    getBrowserType(): BrowserType;
    /**
     * Gets the current browser version.
     *
     * @return The current browser version.
     */
    getBrowserVersion(): string;
    /**
     * Tells when the current browser is Microsoft Edge if the browser is chromium based.
     *
     * @return `true` if the current browser is Microsoft Edge and if it is chromium based.
     */
    isEdgeChromium(): boolean;
}
export { InfiniteApiControllerInterface };
/**
 * This factory is used to create the infinite cache ({@link InfiniteCacheInterface}).
 *
 * The {@link InfiniteCacheInterface} is based on IndexedDB to limit the bandwidth usage,
 * and speed up the loading and displaying of data when using a {@link DataSessionInterface}.
 *
 * The sizes of the cache are stored in the LocalStorage of the browser.
 * The former values that were set will be used when creating a new cache / reloading the page.
 *
 * The connection procedure to IndexedDB is started when the {@link InfiniteCacheInterface} is created.
 *
 * ```typescript
 * [[include:InfiniteCacheInterface_creation.ts]]
 * ```
 * <br>
 * Please refer to {@link InfiniteCacheInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Factories</div><br>
 * @see {@link InfiniteCacheInterface}
 * @see {@link DataSessionInterface}
 * @category Factories
 */
declare class InfiniteCacheFactory {
    /**
     * Instantiates an Infinite Cache ({@link InfiniteCacheInterface}).
     *
     * Except from very specific needs, there is no need to create multiple {@link InfiniteCacheInterface} instances.
     *
     * @return The resulting {@link InfiniteCacheInterface}.
     */
    static CreateInfiniteCache(): InfiniteCacheInterface;
}
export { InfiniteCacheFactory };
/**
 * The {@link InfiniteCacheInterface} is based on IndexedDB to limit the bandwidth usage,
 * and speed up the loading and displaying of data when using a {@link DataSessionInterface}.
 *
 * The sizes of the cache are stored in the LocalStorage of the browser.
 * The former values that were set will be used when creating a new cache / reloading the page.
 *
 * The {@link InfiniteCacheInterface} handles two types of data :
 * * build data : i.e. initialization data, if such data is cached, the DMU loading will be a lot faster
 * * HD geometric data : upon navigation, the 3djuump infinite resource loader of the {@link InfiniteEngineInterface}
 * loads high definition geometric data to display the most accurate version of the DMU given the budget allocated to
 * the display of resources. Such HD data is cached in the {@link InfiniteCacheInterface}.
 *
 * The two types of data are handled separately in the {@link InfiniteCacheInterface}.
 *
 * The connection procedure to IndexedDB is started when the {@link InfiniteCacheInterface} is created.
 * The {@link isConnected} will return true shortly after creation if IndexedDB is available.
 *
 * ```typescript
 * [[include:InfiniteCacheInterface_creation.ts]]
 * ```
 * <br>
 * The {@link InfiniteCacheInterface} only exposes getters,
 * the cache manipulation is available through the {@link PerformanceInterface}, the {@link PerformanceInterface}
 * also lists absolute minimum and default values ({@link PerformanceInterface.GetMaxBuildCacheSizeMin},
 * {@link PerformanceInterface.GetMaxBuildCacheSizeDefault}, ...).
 *
 * ```typescript
 * [[include:InfiniteCacheInterface_manipulation.ts]]
 * ```
 * <br>
 * Please refer to {@link DataSessionInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Sessions</div><br>
 * @see {@link DataSessionInterface}
 * @category Sessions
 */
interface InfiniteCacheInterface {
    /**
     * Indicates if the cache is connected (indexedDb is available and cache has been connected).
     *
     * @return `true` if the cache is connected.
     */
    isConnected(): boolean;
    /**
     * Clears the cache, i.e. removes all data (build data and HD data).
     *
     * This procedure may take some time.
     *
     * @return `false` if cache is not connected.
     */
    clearCache(): boolean;
    /**
     * Gets the maximum number of bytes used to store HD geometries (in bytes).
     *
     * This is the same value that was set by {@link PerformanceInterface.setMaxGeometryCacheSize}.
     *
     * @return The maximum number of bytes used to store HD geometries (in bytes).
     */
    getMaxHDSize(): number;
    /**
     * Gets the maximum number of bytes used to store build data (in bytes).
     *
     * This is the same value that was set by {@link PerformanceInterface.setMaxBuildCacheSize}.
     *
     * @return The maximum number of bytes used to store build data (in bytes).
     */
    getMaxBuildSize(): number;
    /**
     * Gets the current number of bytes used to store HD geometries (in bytes) in the cache.
     *
     * @return The current number of bytes used to store HD geometries (in bytes) in the cache.
     */
    getCurrentHDSize(): number;
    /**
     * Gets the current number of bytes used to store build data (in bytes) in the cache.
     *
     * @return The current number of bytes used to store build data (in bytes) in the cache.
     */
    getCurrentBuildSize(): number;
}
export { InfiniteCacheInterface };
/**
 * The InfiniteEngineLoadingStateAttachment is the attachment of the
 * {@link InfiniteEngineInterfaceSignal.LoadingStateChanged} signal
 * sent by the {@link InfiniteEngineInterface} when the loading state of the {@link InfiniteEngineInterface}
 * has changed.
 *
 * This object stores information about the previous and new state of the resource management system.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_screenshot.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_asyncScreenshot.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @category Events
 */
interface InfiniteEngineLoadingStateAttachment {
    /**
     * The old loading state of the resource management system inside the {@link InfiniteEngineInterface}.
     */
    oldLoadingState: LoadingStates;
    /**
     * The new loading state of the resource management system inside the {@link InfiniteEngineInterface}.
     */
    newLoadingState: LoadingStates;
}
export { InfiniteEngineLoadingStateAttachment };
/**
 * The InfiniteImageInterface is a 2D image.
 *
 * It is the attachment of the {@link InfiniteEngineInterfaceSignal.ScreenshotReady} signal
 * sent by the {@link InfiniteEngineInterface} when the given screenshot request has terminated.
 * There may be multiple screenshot requests running simultaneously.
 *
 * This object stores information about an image (width, height, format, data).
 * The {@link format} has the same values as the type parameter used by HTMLCanvasElement.toDataURL() or HTMLCanvasElement.toBlob()
 * and thus depends on your browser capabilities: usually 'image/png', 'image/jpeg' are at least handled.
 * The API adds the 'image/raw' type, which is the data in RGB or RGBA unsigned bytes.
 * If the {@link format} is 'image/raw' type, then the {@link components} field tells the number of components the image is composed of (3 for RGB, 4 for RGBA).
 * If the {@link format} is not 'image/raw', then {@link components} is set to 0, since data is encoded, depending on the format.
 *
 * Obviously, the {@link width} and {@link height} parameters tells the size of the resulting image.
 *
 * The {@link base64} field tells about the type of {@link data} : if set to true, then data is base64 encoded, and thus data will be of the string type,
 * if set to false, the data is an Uint8Array.
 *
 * The {@link screenshotid} field is the value returned by {@link InfiniteEngineInterface.screenshot} to allow matching the result with the requested screenshot.
 *
 * If an error occurs, an invalid image is returned, with a {@link data} equal to undefined, and {@link width} and {@link height} set to 0.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_screenshot.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_asyncScreenshot.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @category Events
 */
interface InfiniteImageInterface {
    /**
     * The format of the image, at least 'image/png' is supported.
     *
     * The {@link format} has the same values as the type parameter used by HTMLCanvasElement.toDataURL() or HTMLCanvasElement.toBlob()
     * depending on your browser capabilities.
     *
     * The API adds 'image/raw' type, for non encoded data in RGB or RGBA format.
     */
    format: string;
    /**
     * The width of the image.
     *
     * If an error occurs, width is set to 0.
     */
    width: number;
    /**
     * The height of the image.
     *
     * If an error occurs, height is set to 0.
     */
    height: number;
    /**
     * The number of components of the image.
     *
     * If the format is not 'image/raw', then components is set to 0, since data is encoded, and usually the number of components depends on the format,
     * or the way data is read.
     * If the format is 'image/raw', then components is usually 3 or 4, telling RGB or RGBA format is used.
     */
    components: number;
    /**
     * Tells if {@link data} is base64 encoded.
     *
     * If set to true, {@link data} is a string, else an Uint8Array.
     *
     * If an error occurs, {@link data} is undefined.
     */
    base64: boolean;
    /**
     * The actual image data.
     *
     * Data is compressed with the algorithm defined by the {@link format} ('image/png', 'image/jpeg').
     *
     * If format is set to 'image/raw', then data contains the RGB or RGBA unsigned bytes data.
     *
     * If {@link base64} is set to true, then data is expressed as the base64 string of the actual byte data.
     * If {@link base64} is set to false, then data is expressed as the actual byte data in an Uint8Array.
     * The top area is contained in the first bytes of the array.
     *
     * If an error occurs, {@link data} is undefined.
     */
    data: undefined | string | Uint8Array | Uint32Array;
    /**
     * The screenshot request id this image comes from.
     *
     * This is the value returned by {@link InfiniteEngineInterface.screenshot} to allow matching the image with the requested screenshot.
     */
    screenshotid: string;
}
export { InfiniteImageInterface };
/**
 * (WebXR) An InfiniteXRSession is actually an XRSession.
 *
 * The XRSession type is not used directly to avoid having all the times the XR types included even if not relevant.
 * Please cast InfiniteXRSession to XRSession.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @category 3D Rendering
 */
type InfiniteXRSession = any;
export { InfiniteXRSession };
/**
 * (WebXR) An InfiniteXRFrame is actually an XRFrame.
 *
 * The XRFrame type is not used directly to avoid having all the times the XR types included even if not relevant.
 * Please cast InfiniteXRFrame to XRFrame.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @category 3D Rendering
 */
type InfiniteXRFrame = any;
export { InfiniteXRFrame };
/**
 * The InfiniteEngineInterfaceSignal lists all the signals that may be sent by the {@link InfiniteEngineInterface}.
 *
 * These signals are emitted by the {@link InfiniteEngineInterface} when the following occurs :
 *
 * * {@link Picked} : the last call to the asynchronous {@link InfiniteEngineInterface.pickAt}, {@link InfiniteEngineInterface.pickRect}
 * or {@link InfiniteEngineInterface.pickFromRay} functions is ready,
 * i.e. when the {@link InfiniteEngineInterface} has finished processing the pick request. The pick result is available as an attachment. Multiple pick requests
 * can be queued and retrieved, please use {@link InfiniteEngineInterface.getLastPickingRequestId} and {@link PickingAttachment.pickingRequestId}.
 * * {@link ViewChanged} : triggered when {@link InfiniteEngineInterface.setView} is called.
 * * {@link ViewResized} : triggered when the rendering surface has changed, this may be the result of a canvas size change, or a {@link InfiniteEngineInterface.setMaximumEngineResolution}
 * call that changes the internal texture sizes.
 * * {@link LoadingStateChanged} : triggered when the loading state of the resource loading algorithm has changed (see {@link InfiniteEngineInterface.getResourceLoadingState}).
 * * {@link ScreenshotReady} : triggered after a successful call to {@link InfiniteEngineInterface.screenshot}.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_picking.ts]]
 * ```
 * <br>
 * Screenshot sample :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_screenshot.ts]]
 * ```
 * <br>
 * Please refer to {@link InfiniteEngineInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link InfiniteEngineInterface}
 * @see {@link InfiniteEngineInterface.pickAt}
 * @see {@link InfiniteEngineInterface.pickRect}
 * @category Events
 */
declare enum InfiniteEngineInterfaceSignal {
    /**
     * Signal sent by the {@link InfiniteEngineInterface} when a pick request is completed.
     *
     * A pick request aks to know the content (i.e. geometries, lines, boxes, points) at
     * specific positions in the 3D display. The picking request may concern a specific position ({@link InfiniteEngineInterface.pickAt}),
     * a rectangular area ({@link InfiniteEngineInterface.pickRect}) or the element that intersects a ray ({@link InfiniteEngineInterface.pickFromRay}) inside the 3d view.
     *
     * The actual picked point follows the following convention :
     *
     * <img src="media://picking.svg" alt="picking convention">
     * <br>
     * The attachment for events of this type is a {@link PickingAttachment}.
     *
     * @see {@link PickingAttachment}
     */
    Picked = "picked",
    /**
     * Signal sent by the {@link InfiniteEngineInterface} when the engine has changed its view, i.e. the bound html
     * element is changed (another element or undefined) with a call to {@link InfiniteEngineInterface.setView}.
     *
     * No attachment for events of this type.
     */
    ViewChanged = "viewChanged",
    /**
     * Signal sent by the {@link InfiniteEngineInterface} when the rendering size of the display has changed.
     *
     * This may be due to a change in the canvas size or a call to {@link InfiniteEngineInterface.setMaximumEngineResolution}
     * with values inferior to the current rendering size.
     *
     * No attachment for events of this type.
     */
    ViewResized = "viewResized",
    /**
     * Signal sent by the {@link InfiniteEngineInterface} when the loading state of the resource manager has changed.
     *
     * The event tells the old and new loading state.
     *
     * The attachment for events of this type is a {@link InfiniteEngineLoadingStateAttachment}.
     *
     * @see {@link InfiniteEngineInterface.getResourceLoadingState}
     */
    LoadingStateChanged = "loadingStateChanged",
    /**
     * Signal sent by the {@link InfiniteEngineInterface} when a screenshot request has finished processing.
     *
     * The attachment for events of this type is an {@link InfiniteImageInterface}.
     *
     * @see {@link InfiniteEngineInterface.screenshot}
     */
    ScreenshotReady = "screenshotReady",
    /**
     * Called each time the rendering is over (after all is rendered with the callback to window.requestAnimationFrame).
     *
     * No attachment for events of this type.
     */
    DisplayDone = "displayDone",
    /**
     * (WebXR) Signal sent by the {@link InfiniteEngineInterface} when a XR Session is ready to be used.
     *
     * The attachment for events of this type is an {@link InfiniteXRSession} which is a XRSession.
     * XR types are mapped to avoid installing WebXR types.
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_webxr.ts]]
     * ```
     * <br>
     * The WebXR api should be used with specific hardware.
     *
     * @see {@link InfiniteEngineInterface.setView}
     */
    XRSessionReady = "XRSessionReady"
}
export { InfiniteEngineInterfaceSignal };
/**
 * Represents a picking promise result.
 *
 * This value is used with {@link InfiniteEngineInterface.asyncPickAt}, {@link InfiniteEngineInterface.asyncPickRect}, {@link InfiniteEngineInterface.asyncPickFromRay}.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @category 3D Rendering
 */
interface AsyncPickingResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     *
     * @see {@link PickingAttachment}
     */
    value?: PickingAttachment;
}
export { AsyncPickingResult };
/**
 * Represents a resource loading promise result.
 *
 * This value is used with {@link InfiniteEngineInterface.asyncWaitForDataLoaded}.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @category 3D Rendering
 */
interface AsyncDataLoadedResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     *
     * @see {@link InfiniteEngineLoadingStateAttachment}
     */
    value?: InfiniteEngineLoadingStateAttachment;
}
export { AsyncDataLoadedResult };
/**
 * Represents a screenshot promise result.
 *
 * This value is used with {@link InfiniteEngineInterface.asyncScreenshot}.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @category 3D Rendering
 */
interface AsyncScreenshotResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     *
     * @see {@link InfiniteImageInterface}
     */
    value?: InfiniteImageInterface;
}
export { AsyncScreenshotResult };
/**
 * When using {@link InfiniteEngineInterface.setView}, a specific rendering type may be called :
 * * undefined : defaults to webgl2 is available, else webgl1
 * * webgl1 : webgl1 is used even if webgl2 is available
 * * webxr : (WebXR) a webxr session is started.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_webxr.ts]]
 * ```
 * <br>
 * The WebXR api should be used with specific hardware.
 * Please refer to {@link https://developer.mozilla.org/en-US/docs/Web/API/WebXR_Device_API/Fundamentals}.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link InfiniteEngineInterface.setView}
 * @category 3D Rendering
 */
interface InfiniteEngineViewType {
    /**
     * The requested type of engine :
     * * 'forcewebgl1' : webgl1 is used even if webgl2 is available
     * * 'webxr' : a webxr session is started.
     */
    type: 'webxr' | 'forcewebgl1';
    /**
     * Reserved for future use.
     */
    options?: Object;
}
export { InfiniteEngineViewType };
/**
 * The ScreenshotType enumeration tells the available screenshot types when using {@link InfiniteEngineInterface.screenshot}:
 * * ST_Color : usual RGB/RGBA screenshot encoded in the required format.
 * * ST_GeometricInstanceIdOpaque : an unsigned integer image that tells the given opaque geometric instance at the given pixel.
 * * ST_GeometricInstanceIdGhost : an unsigned integer image that tells the given ghosted geometric instance at the given pixel.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_screenshot.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_asyncScreenshot.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link InfiniteEngineInterface.screenshot}
 * @see {@link InfiniteEngineInterface.asyncScreenshot}
 * @category 3D Rendering
 */
declare enum ScreenshotType {
    /**
     * A colorful screenshot type, in the required format.
     */
    ST_Color = 0,
    /**
     * An unsigned integer image that tells the given opaque geometric instance at the given pixel.
     */
    ST_GeometricInstanceIdOpaque = 1,
    /**
     * An unsigned integer image that tells the given ghosted geometric instance at the given pixel.
     */
    ST_GeometricInstanceIdGhost = 2,
    /**
     * An invalid screenshot type.
     */
    ST_Invalid = -1
}
export { ScreenshotType };
/**
 * The InfiniteEngineInterface interface is used to render the content of the DMU.
 *
 * The usage is very basic :
 * - create an InfiniteEngineInterface connected to a {@link MetadataManagerInterface}.
 * - connect it to a DOMElement.
 * - change the way `geometric instances` are rendered.
 *
 * **WARNING** : There is no automatic conversion of the content of {@link FilterSolverInterface} to the 3D engine. It is the role
 * of the web application to show/hide/ghost geometries depending on the application needs.
 *
 * ## Initialization.
 *
 * The InfiniteEngineInterface is initialized from a {@link MetadataManagerInterface} by an {@link InfiniteFactory}.<br>
 * The InfiniteEngineInterface takes advantage of the {@link MetadataManagerInterface} connected to a {@link DataSessionInterface}.
 * The use of the InfiniteEngineInterface is only relevant when a {@link DataSessionInterface} is connected to a 3djuump infinite build.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_creation.ts]]
 * ```
 * <br>
 * Creation of an InfiniteEngineInterface.
 *
 * ## Rendering.
 *
 * The InfiniteEngineInterface display is governed by a resource controller. Indeed, the display is based on some low definition
 * rendering of the DMU, and HD geometric data rendered on top of it. The amount of acceptable data is governed by the
 * {@link PerformanceInterface} that provides a way to set the budgets. Such budgets rely heavily on the hardware of the end-user.
 * It is recommended to allow the end-user to manipulate such budgets.<br>
 * The low definition version of the DMU is composed of a static version (not configurable), and a dynamic one, whom triangle number can be
 * configured.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_resource_controller.ts]]
 * ```
 * <br>
 * The state of the resource controller (ok, out of budget, running) is retrieved from {@link getResourceLoadingState} and the {@link LoadingStates}
 * enumeration.
 *
 * ```typescript
 *  [[include:InfiniteEngineInterface_LoadingStates.ts]]
 * ```
 * <br>
 * An optional {@link InfiniteCacheInterface} may be included in the {@link DataSessionInterface} upon creation to speed up the retrieving
 * of HD geometric data.
 *
 * `Geometric instances` are accessed through their `geometric instance ids`, and each `geometric instance` may be :
 * * `Hidden` : object is not displayed.
 * * `Selected` : objet is rendered in a pulsing orange.
 * * `Ghosted` : the object is rendered semi transparent in a light blue color.
 * * `Ignore cut plane` : object cannot be hidden by a cut plane.
 *
 * Such a state is a bit field, and bits can be changed in a batch for multiple `geometric instances`.<br>
 * Use {@link VisualStates} with {@link updateGeometricState}, {@link getGeometricState} and {@link updateGeometricStateForAll} functions to change
 * the way geometries are rendered. The visibility is coded on a 8 bits unsigned number.
 *
 * In the following examples, `x` represent an irrelevant bit :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_change_states.ts]]
 * ```
 * <br>
 * The color of each `geometric instance` can be modified with the use of the {@link MaterialManagerInterface} accessed through
 * {@link getMaterialManager}. The {@link MaterialManagerInterface} stores the properties of the default and user defined materials.<br>
 * It is recommended to reuse (if possible) user defined materials since only a limited number of materials can be created.
 * Materials are accessed by their id (a positive number). Colors are defined as a {@link Vector3}, `x` being the red component, `y`
 * being the green component, `z` the blue component in the `[0:1]` range (e.g `[1,0,0]` is pure red, `[0,1,0]` is pure green, ...).
 *
 * ```typescript
 * [[include:MaterialManagerInterface_example.ts]]
 * ```
 * <br>
 * Changing materials.
 *
 * ## Picking.
 *
 * The "picking" is the action to get the visible item under a specific position in the 3d view. The picking is dependant on the
 * state of the resource controller, i.e. querying multiple picks at the same location without changing the camera position
 * may result in different results, depending on the loaded data.
 *
 * The picking is asynchronous, meaning a {@link InfiniteEngineInterfaceSignal.Picked} signal is triggered when the result is available,
 * with an attachment of the {@link PickingAttachment} type, which stores for each picked object :
 * * its id (`geometric instance id`/`point id`/`line id`/`box id`).
 * * its closest 3D position (a virtual ray is casted on the scene for each picking position, the resulting point is the closest point of the set of the intersection of these rays and the object).
 * * the number of times this object is intersected (minimum one).
 * * the picking request id (please see {@link getLastPickingRequestId}).
 *
 * Multiple pick requests
 * can be queued, calling multiple times {@link InfiniteEngineInterface.pickAt} without waiting for the result will
 * trigger multiple {@link InfiniteEngineInterfaceSignal.Picked} events, they can be distinguished with their picking request ids ({@link InfiniteEngineInterface.getLastPickingRequestId} and {@link PickingAttachment.pickingRequestId}).<br>
 * `Geometric instances`, lines, boxes, and points are queried at the same time on a single pick request (see {@link PrimitiveManagerLineInterface}, {@link PrimitiveManagerBoxInterface}, {@link PrimitiveManagerPointInterface}).
 * The {@link pickRay} and {@link projectToScreen} functions may be used with the pick functions to make some more 3D calculations.
 * You may pick a single point ({@link InfiniteEngineInterface.pickAt}), a rectangular area ({@link InfiniteEngineInterface.pickRect}), or the 3D that intersects a 3D ray.
 * There are some limitations to the 3D ray pick :
 * * The ray must be oriented toward the direction of the camera (the dot vector between the direction of the camera and the direction of the ray must be positive).
 * * Only 3D elements actually rendered may be picked (no ray pick outside the frustum).
 * * The pick ray algorithm rely on the projection of the surfaces to the near plane, objects that are less than 2 pixels wide
 * and 2 pixels high cannot be picked.
 *
 * The picking system works with client coordinates :
 * * `x` between `0` and `clientWidth-1`.
 * * `y` between `0` and `clientHeight-1`.
 *
 * The actual picked point follows the following convention :
 * <img src="media://picking.svg" alt="picking convention">
 * <br>
 * The {@link PrimitiveManagerRubberBandInterface} can be used to display a rectangular area on top of the 3d rendering element.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_picking.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_asyncPicking.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * ## Screenshots.
 *
 * Screenshots can be performed with the {@link screenshot} call. The screenshot procedure is asynchronous, and the {@link InfiniteEngineInterfaceSignal.ScreenshotReady}
 * event is fired when the screenshot is ready.
 * The {@link screenshot} call returns an id, and this is must match the attachment of the {@link InfiniteEngineInterfaceSignal.ScreenshotReady} event which is
 * an {@link InfiniteImageInterface} ({@link InfiniteImageInterface.screenshotid}).
 * An empty string means the screenshot procedure has failed (a missing {@link setView} call probably).
 * An empty {@link InfiniteImageInterface.width} or {@link InfiniteImageInterface.height} means the image is invalid.
 * Multiple {@link screenshot} calls can be executed concurrently.
 *
 * You may want to call {@link setFixedResolution} to get a screenshot of this size.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_screenshot.ts]]
 * ```
 * <br>
 * or with async calls :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_asyncScreenshot.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * ## Camera manipulation.
 *
 * The camera is manipulated from the {@link CameraManagerInterface}, accessed through {@link getCameraManager}. Some
 * default behaviors are available in order to avoid the developer to code a lot of event handling in order to make the camera
 * move in an interesting way. Available behaviors are orbital ones, i.e. the camera moves around a center of interest, but mouse
 * manipulation is different from modes.
 *
 * You may create animations to move, rotate the camera.
 *
 * ## Cut Plane manipulation.
 *
 * A cut plane is a theoretical 3D plane that "cut" objects. A cut plane ([3D Affine hyperplane](https://en.wikipedia.org/wiki/Hyperplane)) is defined by a position `P`
 * that belong to the plane and a normal `N : (Nx,Ny,Nz)`. Such a plane
 * has the equation : &#123;x,y,z&#125; / Nx.x + Ny.y + Nz.z = dot(N,P). Each vertex on the side of the normal `N` will be discarded, so every vertex V : &#123;Vx,Vy,Vz&#125; /
 * Nx.Vx + Ny.Vy + Nz.Vz > dot(N,P) will be discarded.
 *
 * Cut planes are modified through the {@link CutPlaneManagerInterface} accessible through {@link getCutPlaneManager}.
 *
 * A {@link CutPlaneManipulatorInterface} may be created and bound to a cut plane to display a GUI element that allows the end user to change the cut plane orientation / position through
 * {@link CutPlaneManagerInterface.createCutPlaneManipulator}.
 *
 * Object with a visual state {@link VisualStates.S_IgnoreCutPlane} will not be affected by cut planes.
 *
 * ```typescript
 * [[include:CutPlaneManagerInterface_set_cut_plane.ts]]
 * ```
 * <br>
 * Setting cut planes.
 *
 * ## Annotations.
 *
 * The rendering may be enriched to display 2D data representing measures, drawings (Functional Tolerancing and Annotations,
 * Product Manufacturing Information, Measures, etc). These data are basically a 3D plane containing information in form of texts, curves, and
 * symbols in this plane. Data of this type is referred to an Annotation.
 *
 * ## 3D Primitives.
 *
 * 3D Points, 3D lines, 3D boxes and a single 2D rectangle can be rendered on top of the DMU, with a rendering hint depending on the relative depth between the 3D primitives and the geometries.
 * 3D primitives are accessed through the {@link getPrimitiveManager} function ({@link PrimitiveManagerInterface}). The colors of the 3D primitive are handled by the {@link PrimitiveManagerMaterialInterface},
 * accessed through {@link PrimitiveManagerInterface.getMaterialManager}. Materials are accessed by their ids, and modified by their ids. 3D primitives are
 * rendered colored with their material id, and not directly from a color.
 *
 * ### 3D Points.
 *
 * The rendering can be enriched with 3D points rendered on top of the geometries, with a rendering hint telling if the point is over or under a geometry.
 *
 * <img src="media://point.svg" alt="point display"/>
 * <br>
 * Point Creation sample :
 *
 * ```typescript
 * [[include:PrimitiveManagerPointInterface_create_point.ts]]
 * ```
 * <br>
 * Creating a point.
 *
 * ### 3D Lines.
 *
 * The rendering can be enriched with 3D lines rendered on top of the geometries.
 *
 * <img src="media://line.svg" alt="line display"/>
 * <br>
 * Line Creation sample :
 *
 * ```typescript
 * [[include:PrimitiveManagerLineInterface_create_line.ts]]
 * ```
 * <br>
 * Creating a line.
 *
 * ### 3D Boxes.
 *
 * The rendering can be enriched with 3D boxes rendered on top of the geometries, with a rendering hint telling if the edge/face is over or under a geometry.
 *
 * <img src="media://box.svg" alt="box display"/>
 * <br>
 * Box Creation sample :
 *
 * ```typescript
 * [[include:PrimitiveManagerBoxInterface_create_box.ts]]
 * ```
 * <br>
 * Creating a Box.
 *
 * ### 2D rectangle (rubber band).
 *
 * The rendering can be enriched with a single 2D rectangle rendered on top of the geometries.
 * The color cannot be changed.
 *
 * <img src="media://rectangle.svg" alt="rubber band display"/>
 * <br>
 * Rectangle usage sample :
 *
 * ```typescript
 * [[include:PrimitiveManagerRubberBandInterface_create_rectangle.ts]]
 * ```
 * Using a Rectangle.
 *
 * ## WebXR (experimental).
 *
 * You may use the InfiniteEngineInterface api with WebXR sessions :
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_webxr.ts]]
 * ```
 * <br>
 * Please note that dedicated hardware is required to use WebXR. See {@link https://developer.mozilla.org/en-US/docs/Web/API/WebXR_Device_API/Fundamentals}.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link InfiniteEngineInterfaceSignal}
 * @see {@link InfiniteFactory}
 * @see {@link MaterialManagerInterface}
 * @see {@link PrimitiveManagerInterface}
 * @see {@link PerformanceInterface}
 * @see {@link CutPlaneManagerInterface}
 * @category 3D Rendering
 */
interface InfiniteEngineInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Gets the version of the 3djuump infinite API.
     *
     * @return The version of the 3djuump infinite API.
     * @deprecated Please use {@link InfiniteApiControllerInterface.getVersion}.
     */
    getVersion(): string;
    /**
     * Sets the HTML element to be used to render the DMU.
     *
     * @param pView <div class="const">in</div> The HTML element to render the DMU into.
     * @param pForceEngineType <div class="const">in</div> If set to true, then webgl1 rendering will be used, even if webgl2 is available.
     * If set to an object, the rendering can be forced to webgl1 or webxr (see {@link InfiniteEngineViewType}).
     * When requesting a webxr session, pView must be set to undefined.
     * @return `true` if the 3D view is set.
     */
    setView(pView: HTMLElement | undefined, pForceEngineType?: boolean | InfiniteEngineViewType): boolean;
    /**
     * Gets the annotation renderer.
     *
     * The annotation renderer is there to include and display annotations in the 3D view.
     *
     * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
     *
     * @return The {@link AnnotationRendererInterface} of the given InfiniteEngineInterface.
     * @see {@link AnnotationRendererInterface}
     */
    getAnnotationRenderer(): AnnotationRendererInterface;
    /**
     * Gets the HTML element used to render the 3D.
     *
     * @return The HTML element used to render 3D or undefined if not set.
     */
    getView(): HTMLElement | undefined;
    /**
     * Sets the maximum resolution of the rendering "window".
     *
     * On some High DPI devices, the rendering size of the canvas may be too big
     * to have a correct frame rate. You may choose to limit the maximum resolution to increase performance.
     * The default maximum resolution is set to 1920 * 1080.
     *
     * @param pWidth <div class="const">in</div> The maximum number of horizontal pixels for the rendering window.
     * @param pHeight <div class="const">in</div> The maximum number of vertical pixels for the rendering window.
     */
    setMaximumEngineResolution(pWidth: number, pHeight: number): void;
    /**
     * Sets the overridden rendering resolution of the rendering "window".
     *
     * If the HTMLElement set with the {@link setView} call ('HTMLViewElement') has not the same aspect ratio, then the
     * 3d display is centered inside the given 'HTMLViewElement', showing the background color of the
     * 'HTMLViewElement'.
     *
     * If the resolution is superior to the size of the 'HTMLViewElement', the rendering is scaled down to the available
     * size.
     *
     * The 3D objects are rendered with ceil(pWidth) by ceil(pHeight) resolution, but you may want to have more precision on the
     * displayed texts of annotations. For these reason, texts are rendered on an area of
     * ceil(pWidth * pWindowRatio) by ceil(pHeight * pWindowRatio), allowing to have maximum performance on the 3D displayed objects
     * and texts still readable.
     *
     * Screenshot request will have a resolution of ceil(pWidth * pWindowRatio) by ceil(pHeight * pWindowRatio).
     *
     * @param pWidth <div class="const">in</div> The maximum number of horizontal pixels for the rendering window (valid values are > 0).
     * @param pHeight <div class="const">in</div> The maximum number of vertical pixels for the rendering window (valid values are > 0).
     * @param pWindowRatio <div class="const">in</div> The new window ratio for rendering texts (must be superior or equal to 1).
     * @return `true` if the fixed resolution has been set and the 3 parameters are valid.
     */
    setFixedResolution(pWidth: number, pHeight: number, pWindowRatio: number): boolean;
    /**
     * Restores the rendering to use the HTMLElement set with the {@link setView} as the full display area.
     *
     * This function is the opposite to the {@link setFixedResolution} call.
     */
    unsetFixedResolution(): void;
    /**
     * Tells if {@link setFixedResolution} is in use at the moment, and gets the fixed resolution is {@link setFixedResolution} is in use.
     *
     * If {@link setFixedResolution}(pWidth, pHeight, pWindowRatio) is in use,
     * pResolution.x will be equal to Math.ceil(pWidth),
     * pResolution.y will be equal to Math.ceil(pHeight),
     * and pResolution.z will be equal to pWindowRatio and true is returned.
     *
     * If {@link setFixedResolution} is not in use, false is returned and pResolution is left as is.
     *
     * @param pResolution <div class="const">out</div> The fixed resolution parameters if {@link setFixedResolution} has been called.
     * @return `true` if {@link setFixedResolution} is in use, else false.
     */
    getFixedResolution(pResolution: Vector3): boolean;
    /**
     * Triggers an asynchronous screenshot request.
     *
     * The current image of the display is grabbed (without the navigation cube and manipulators such as cut plane manipulators), or
     * the geometric instance ids, depending on the screenshot request type,
     * and a screenshot request id is returned.
     * An invalid screenshot request is an empty string.
     *
     * If the screenshot request can be started, then a {@link InfiniteEngineInterfaceSignal.ScreenshotReady} signal will be fired
     * shortly after.
     *
     * The resulting screenshot data will be base64 encoded if pInBase64 is set to `true` ({@link InfiniteImageInterface.data} will be a string,
     * else a UInt8Array).
     *
     * When using {@link ScreenshotType.ST_Color},
     * `pFormat` tells the resulting format of the data, 'image/png' is by default.
     * Available formats are the same as the `HTMLCanvasElement.toDataURL()` or `HTMLCanvasElement.toBlob()` calls.
     * The API adds the 'image/raw' format to deal with individual pixels if requested (without image encoding).
     * `pQuality` is a number in the [0,1] range, telling the expected quality if the requested format is a lossy compression
     * scheme, such as 'image/jpeg'.
     * If the requested format is not handled by the browser, 'image/png' type will be used.
     *
     * When using other formats, `pFormat` and `pQuality` are ignored.
     *
     * If pInBase64 is true, you may create a data url with the following code :
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_screenshot.ts]]
     * ```
     * <br>
     * Making a color screenshot.
     *
     * @param pScreenshotType <div class="const">in</div> The request screenshot type (image, geometric instance ids, etc).
     * @param pInBase64 <div class="const">in</div> If set to true, {@link InfiniteImageInterface.data} will be a base64 string.
     * @param pFormat <div class="const">in</div> The expected format of the resulting screenshot data ('image/png' by default).
     * @param pQuality <div class="const">in</div> The expected quality of the resulting screenshot data, in the [0,1] range in case of lossy compression formats.
     * @return A unique screenshot id request, or an empty string in case of error.
     * @see {@link InfiniteImageInterface}
     * @see {@link InfiniteEngineInterfaceSignal.ScreenshotReady}
     */
    screenshot(pScreenshotType: ScreenshotType, pInBase64: boolean, pFormat?: string, pQuality?: number): string;
    /**
     * Requests an asynchronous screenshot.
     *
     * The current image of the display is grabbed (without the navigation cube and manipulators such as cut plane manipulators), or
     * the geometric instance ids, depending on the screenshot request type,
     * and a screenshot promise is returned.
     *
     * The resulting screenshot data will be base64 encoded if pInBase64 is set to true ({@link InfiniteImageInterface.data} will be a string,
     * else a UInt8Array).
     *
     * When using {@link ScreenshotType.ST_Color},
     * `pFormat` tells the resulting format of the data, 'image/png' is by default.
     * Available formats are the same as the `HTMLCanvasElement.toDataURL()` or `HTMLCanvasElement.toBlob()` calls.
     * The API adds the 'image/raw' format to deal with individual pixels if requested (without image encoding).
     * `pQuality` is a number in the [0,1] range, telling the expected quality if the requested format is a lossy compression
     * scheme, such as 'image/jpeg'.
     * If the requested format is not handled by the browser, 'image/png' type will be used.
     *
     * When using other formats, `pFormat` and `pQuality` are ignored.
     *
     * If pInBase64 is true, you may create a data url with the following code :
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_asyncScreenshot.ts]]
     * ```
     * <br>
     * The result is an {@link AsyncScreenshotResult}.
     *
     * @param pScreenshotType <div class="const">in</div> The request screenshot type (image, geometric instance ids, etc).
     * @param pInBase64 <div class="const">in</div> If set to true, {@link InfiniteImageInterface.data} will be a base64 string.
     * @param pFormat <div class="const">in</div> The expected format of the resulting screenshot data ('image/png' by default).
     * @param pQuality <div class="const">in</div> The expected quality of the resulting screenshot data, in the [0,1] range in case of lossy compression formats.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise is resolved with the screenshot content.
     */
    asyncScreenshot(pScreenshotType: ScreenshotType, pInBase64: boolean, pFormat?: string, pQuality?: number): Promise<AsyncScreenshotResult>;
    /**
     * Enables / disables anti aliasing.
     *
     * Anti aliasing is disabled by default.
     *
     * @param pEnable <div class="const">in</div> If set, enables the anti aliasing.
     */
    enableAntiAliasing(pEnable: boolean): void;
    /**
     * Tells if anti aliasing is enabled.
     *
     * @return `true` if anti aliasing is enabled.
     */
    isAntiAliasingEnabled(): boolean;
    /**
     * Enables / disables edge detection mechanism.
     *
     * Edge detection is disabled by default.
     *
     * @param pEnable <div class="const">in</div> If set, enables the edge detection mechanism.
     */
    enableEdgeDetect(pEnable: boolean): void;
    /**
     * Tells if edge detection mechanism is enabled.
     *
     * @return `true` if edge detection is enabled.
     */
    isEdgeDetectEnabled(): boolean;
    /**
     * Tells if backface culling is enabled, but this value is meaningless.
     *
     * Back-face culling is a rendering optimization that allows the 3d rendering
     * to display the triangles only one-sided, this setup is now enforced by the DMU maintainer globally, or for
     * individual geometries.
     *
     * @return `true` or `false` but this value is meaningless.
     * @deprecated `isBackfaceCullingEnabled` will be removed in future release.
     */
    isBackfaceCullingEnabled(): boolean;
    /**
     * Does nothing.
     *
     * Back-face culling is a rendering optimization that allows the 3d rendering
     * to display the triangles only one-sided, this setup is now enforced by the DMU maintainer globally, or for
     * individual geometries.
     *
     * @param _pBackfaceCullingEnabled <div class="const">in</div> Discarded.
     * @deprecated `setBackfaceCullingEnabled` will be removed in future release.
     */
    setBackfaceCullingEnabled(_pBackfaceCullingEnabled: boolean): void;
    /**
     * Sets the background to be used in the rendering.
     *
     * The background of the rendering scene can be either a grid or a plain color.
     * The grid is the default background.
     * If pUseBackgroundColor is true the background shows the given color pBackgroundColor (R,G,B in the range [0,1]).
     *
     * Returns true if the background has been set. Setting pUseBackgroundColor to true without a valid color is invalid.
     *
     * @param pUseBackgroundColor <div class="const">in</div> If true shows the given pBackgroundColor else shows the ground and pBackgroundColor is ignored.
     * @param pBackgroundColor <div class="const">in</div> Plain color used if the pShowBackground is set to false.
     * @return `true` if the background has been successfully set.
     */
    setBackground(pUseBackgroundColor: boolean, pBackgroundColor?: Vector3): boolean;
    /**
     * Gets the current background.
     *
     * If the background is a grid then getBackgroundColor returns false and
     * pBackgroundColor is left intact.
     *
     * If the background is a color then getBackgroundColor returns true and
     * pBackgroundColor is set with the current background color.
     *
     * @param pBackgroundColor <div class="const">out</div> The current background color (if the background is a color).
     * @return `true` if the background is a color, false if it is a grid.
     */
    getBackgroundColor(pBackgroundColor: Vector3): boolean;
    /**
     * Tells if the renderer is not computing any frame.
     *
     * If nothing should be done, then the
     * previous frame is displayed without any intensive computation.
     *
     * @return `true` if the renderer is recycling the previous frame.
     */
    isRendererIdle(): boolean;
    /**
     * Gets the loading state of the engine.
     *
     * Tells if the engine has finished loading resources and if so, if the
     * budget was sufficient to load all required data.
     *
     * @return The {@link LoadingStates} of the engine.
     */
    getResourceLoadingState(): LoadingStates;
    /**
     * Requests an asynchronous wait for data loaded.
     *
     * The promise is signaled when the data loading process is over, i.e. when {@link getResourceLoadingState} returns
     * {@link LoadingStates.S_AllLoaded} or {@link LoadingStates.S_OutOfBudget}.
     *
     * An example with a screenshot procedure :
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_asyncScreenshot.ts]]
     * ```
     * <br>
     * Please make sure the destination browser supports promises before using async calls.
     *
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise is resolved with the resource loading state.
     */
    asyncWaitForDataLoaded(): Promise<AsyncDataLoadedResult>;
    /**
     * Enables / disables the copyrighted watermark.
     *
     * The copyrighted watermark is the information provided on the bottom right of the rendering
     * that tells about the authors of the javascript API :) .
     *
     * <img src="media://watermark.svg" alt="copyrighted watermark display"/>
     * <br>
     * The copyrighted watermark is enabled by default.
     *
     * @param pEnabled <div class="const">in</div> If set, enables the copyrighted watermark on the rendering.
     */
    enableCopyrightedWatermark(pEnabled: boolean): void;
    /**
     * Tells if the copyrighted watermark is enabled.
     *
     * The copyrighted watermark is the information provided on the bottom right of the rendering
     * that tells about the authors of the javascript API :) .
     *
     * <img src="media://watermark.svg" alt="copyrighted watermark display"/>
     * <br>
     * This is set by {@link enableCopyrightedWatermark}.
     *
     * @return `true` if the copyrighted watermark is enabled.
     */
    isCopyrightedWatermarkEnabled(): boolean;
    /**
     * (WebXR) Gets the current XRFrame.
     *
     * The XRFrame is valid only during the time of the rendering loop, you may access the given frame only when connecting
     * to {@link CameraManagerInterface} events, or {@link InfiniteEngineInterfaceSignal.DisplayDone}.
     *
     * If the session is not an XRSession, or if calld outside of the rendering loop, returns undefined.
     *
     * @return The current XRFrame if called inside the rendering loop in an XR session, else undefined.
     */
    getXRFrame(): InfiniteXRFrame | undefined;
    /**
     * Requests an asynchronous pick in the 3D at coordinates [pX,pY] (`(0,0)` is on the upper left of screen).
     *
     * Fires the event {@link InfiniteEngineInterfaceSignal.Picked} when the result is ready. Attachment is a {@link PickingAttachment}.
     * Please use {@link getLastPickingRequestId} to know the pick request id in case of multiple pick requests.
     *
     * @param pX <div class="const">in</div> Number between 0 and view3d.width-1, origin is the left side of the view (as retrieved from the MouseEvent).
     * @param pY <div class="const">in</div> Number between 0 and view3d.height-1, origin is the upper side of the view (as retrieved from the MouseEvent).
     * @return `true` if [pX pY] is inside the 3D area.
     */
    pickAt(pX: number, pY: number): boolean;
    /**
     * Requests an asynchronous pick in the 3D at coordinates [pX,pY] (`(0,0)` is on the upper left of screen).
     *
     * The result is an {@link AsyncPickingResult}.
     *
     * @param pX <div class="const">in</div> Number between 0 and view3d.width-1, origin is the left side of the view (as retrieved from the MouseEvent).
     * @param pY <div class="const">in</div> Number between 0 and view3d.height-1, origin is the upper side of the view (as retrieved from the MouseEvent).
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise is resolved with the picking attachment.
     */
    asyncPickAt(pX: number, pY: number): Promise<AsyncPickingResult>;
    /**
     * Requests an asynchronous pick in the 3D from a screen space rectangle.
     *
     * (0,0) is on the upper left of screen.
     * Fires the event {@link InfiniteEngineInterfaceSignal.Picked} when the result is ready. Attachment is a PickingAttachment.
     * Please use {@link getLastPickingRequestId} to know the pick request id in case of multiple pick requests.
     *
     * @param pRectangle <div class="const">in</div> the screen space rectangle to pick from. Origin of the screen is the top-left of the view (as retrieved from the MouseEvent).
     * @return `true` if the rectangle is valid and not empty.
     */
    pickRect(pRectangle: Rectangle): boolean;
    /**
     * Requests an asynchronous pick in the 3D from a screen space rectangle.
     *
     * (0,0) is on the upper left of screen.
     * The result is an {@link AsyncPickingResult}.
     *
     * @param pRectangle <div class="const">in</div> The screen space rectangle to pick from. Origin of the screen is the top-left of the view (as retrieved from the MouseEvent).
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise is resolved with the picking attachment.
     */
    asyncPickRect(pRectangle: Rectangle): Promise<AsyncPickingResult>;
    /**
     * Requests an asynchronous pick in the 3D from a 3D ray.
     *
     * The 3D is not required to come from the position of the camera, but the ray must be oriented toward the direction of the camera
     * (the dot vector between the direction of the camera and the direction of the ray must be positive).
     * The ray is a **half line** that comes from the origin and follows the direction to the infinity.
     *
     * Fires the event {@link InfiniteEngineInterfaceSignal.Picked} when the result is ready. Attachment is a PickingAttachment.
     * Please use {@link getLastPickingRequestId} to know the pick request id in case of multiple pick requests.
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_pickFromRay.ts]]
     * ```
     * <br>
     * There are some limitations to the 3D ray pick :
     * * Only 3D elements actually rendered may be picked (no ray pick outside the frustum).
     * * The pick ray algorithm rely on the projection of the surfaces to the near plane, objects that are less than 2 pixels wide
     * and 2 pixels high cannot be picked.
     *
     * @param pOrigin <div class="const">in</div> The origin of the ray.
     * @param pDirection <div class="const">in</div> The direction of the ray.
     * @return `true` if the picking request is accepted and a pick result will be received soon.
     */
    pickFromRay(pOrigin: Vector3, pDirection: Vector3): boolean;
    /**
     * Requests an asynchronous pick in the 3D from a 3D ray.
     *
     * The 3D is not required to come from the position of the camera, but the ray must be oriented toward the direction of the camera
     * (the dot vector between the direction of the camera and the direction of the ray must be positive).
     * The ray is a **half line** that comes from the origin and follows the direction to the infinity.
     *
     * Fires the event {@link InfiniteEngineInterfaceSignal.Picked} when the result is ready. Attachment is a PickingAttachment.
     * Please use {@link getLastPickingRequestId} to know the pick request id in case of multiple pick requests.
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_asyncPickFromRay.ts]]
     * ```
     * <br>
     * There are some limitations to the 3D ray pick :
     * * Only 3D elements actually rendered may be picked (no ray pick outside the frustum).
     * * The pick ray algorithm rely on the projection of the surfaces to the near plane, objects that are less than 2 pixels wide
     * and 2 pixels high cannot be picked.
     *
     * @param pOrigin <div class="const">in</div> The origin of the ray.
     * @param pDirection <div class="const">in</div> The direction of the ray.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise is resolved with the picking attachment.
     */
    asyncPickFromRay(pOrigin: Vector3, pDirection: Vector3): Promise<AsyncPickingResult>;
    /**
     * Gets the 3D representation of a ray from the camera to the normalized screen ray position.
     *
     * (0,0) is the upper left of screen. (1,1) is the bottom right of screen. The 3D representation is expressed as a position (the camera position) and a direction.
     * Making a pickRay(0,0,lDir0,lCamPos), pickRay(0,1,lDir1,lCamPos), pickRay(1,0,lDir2,lCamPos), pickRay(1,0,lDir3,lCamPos) effectively
     * stores the limiting rays of the view frustum.
     *
     * @param pX <div class="const">in</div> Number between 0 and 1, 0 is the left side of the view.
     * @param pY <div class="const">in</div> Number between 0 and 1, 0 is the upper side of the view.
     * @param pPickRayOut <div class="const">out</div> 3D Vector direction of the picked ray.
     * @param pOrigin <div class="const">out</div> 3D Vector origin of the picked ray (i.e. position of the camera).
     * @return `true` if [pX pY] was inside <code>[0:1]</code>.
     */
    pickRay(pX: number, pY: number, pPickRayOut: Vector3, pOrigin: Vector3): boolean;
    /**
     * Gets the last request id of the running {@link pickAt} or {@link pickRect} call.
     *
     * Just call getLastPickingRequestId after the call to {@link pickAt} or {@link pickRect} to know the request id.
     *
     * If no picking is running, then returns an empty string.
     *
     * @return The id of the last picking request.
     */
    getLastPickingRequestId(): string;
    /**
     * Projects the given 3D point on the screen.
     *
     * The resulting screen coordinates in pScreenCoordinatesOut are normalized between `0` and `1`, `(0,0)` is the upper left of screen,
     * `(1,1)` is the bottom right of screen.
     *
     * @param pPoint <div class="const">in</div> The 3D point to project.
     * @param pScreenCoordinatesOut <div class="const">out</div> The resulting 2D normalized projected point in the range [0:1].
     * @return `true` if the view3D was set and if the point can be projected onto the screen.
     */
    projectToScreen(pPoint: Vector3, pScreenCoordinatesOut: Vector2): boolean;
    /**
     * Given a 3D position, computes the uniform scale to set to an hypothetical unit box to make its projected extents appear to be pExpectedPixelSize pixels wide on the screen.
     *
     * This function may be useful to set a scale to a 3D object to make its size appear to be pExpectedPixelSize pixels big.
     *
     * @param pGizmoPosition <div class="const">in</div> The position of the hypothetical unit box.
     * @param pExpectedPixelSize <div class="const">in</div> The expected projected box extent in pixels.
     * @param pClientHeight <div class="const">in</div> the client height in pixels.
     * @return The uniform scale to set to the hypothetical unit box.
     * @deprecated This function should have never been added to the API.
     * @internal
     */
    computeGizmoScale(pGizmoPosition: Vector3, pExpectedPixelSize: number, pClientHeight: number): number;
    /**
     * Gets the {@link CameraManagerInterface}, useful to control the camera in the scene.
     *
     * @return The {@link CameraManagerInterface} used to manipulate the camera.
     */
    getCameraManager(): CameraManagerInterface;
    /**
     * Gets the {@link MaterialManagerInterface}.
     *
     * The {@link MaterialManagerInterface} is used to change/restore the color of instances.
     *
     * @return The {@link MaterialManagerInterface} used to change colors of instances.
     */
    getMaterialManager(): MaterialManagerInterface;
    /**
     * Gets the {@link CutPlaneManagerInterface}.
     *
     * The {@link CutPlaneManagerInterface} is used to create/manipulate cut planes.
     *
     * @return The {@link CutPlaneManagerInterface} used to manipulate cut planes.
     */
    getCutPlaneManager(): CutPlaneManagerInterface;
    /**
     * Gets the {@link PrimitiveManagerInterface}.
     *
     * The {@link PrimitiveManagerInterface} allows access to the creation of 3D points, 3D lines, 3D boxes and 2D rectangles.
     *
     * @return The {@link PrimitiveManagerInterface} used to manipulate 3D points, 3D lines, 3D boxes and 2D rectangles.
     */
    getPrimitiveManager(): PrimitiveManagerInterface;
    /**
     * (Experimental) Gets the WebGLRenderingContext used to render things.
     *
     * Use this object at your own risks !!!.
     *
     * @return The WebGLRenderingContext used to render things on the 3D view.
     */
    getWebGLRenderingContext(): WebGLRenderingContext | undefined;
    /**
     * Sets the visual state (Visible/hide/Ghost/Ignore cut plane) of `geometric instance ids`.
     *
     * The visual state is a bit field that allows to change a lot of `geometric instance ids` at once.
     *
     * The bit mask is used to tell which bits will be changed by the function call (the use may want to preserve
     * the ghost state of the `geometric instances` for example).
     *
     * The bit value tells the value to set (bits not set by the mask will not be altered).
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_change_states.ts]]
     * ```
     * <br>
     * Please refer to the {@link VisualStates} for a list of possible values.
     *
     * @param pGeometricIds <div class="const">in</div> The <code>geometric instance ids</code> that will be changed by the function call.
     * @param pBitMask <div class="const">in</div> The bit mask ("ORed" value of the {@link VisualStates}) of the states that will be altered.
     * @param pBitValue <div class="const">in</div> The bit value ("ORed" value of the {@link VisualStates}) of the states that will be set.
     * @see {@link VisualStates}
     */
    updateGeometricState(pGeometricIds: Uint32Array | number[], pBitMask: number, pBitValue: number): void;
    /**
     * Gets the visual state (Visible/hide/Ghost/Ignore cut plane) of the given `geometric instance id`.
     *
     * The visual state is a bit field that allows to change a lot of `geometric instance ids` at once.
     *
     * @param pGeometricId <div class="const">in</div> The <code>geometric instance id</code> of the element to get.
     * @return The visual state of the given element.
     * @see {@link VisualStates}
     */
    getGeometricState(pGeometricId: number): number;
    /**
     * Sets the visual state (Visible/hide/Ghost/Ignore cut plane) of all the `geometric instances`.
     *
     * The visual state is a bit field that allows to change a lot of `geometric instance ids` at once.
     *
     * The bit mask is used to tell which bits will be changed by the function call (the use may want to preserve
     * the ghost state of the `geometric instances` for example).
     *
     * The bit value tells the value to set (bits not set by the mask will not be altered).
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_change_states.ts]]
     * ```
     * <br>
     * Please refer to the {@link VisualStates} for a list of possible values.
     *
     * @param pBitMask <div class="const">in</div> The bit mask ("ORed" value of the {@link VisualStates}) of the states that will be altered.
     * @param pBitValue <div class="const">in</div> The bit value ("ORed" value of the {@link VisualStates}) of the states that will be set.
     * @see {@link VisualStates}
     */
    updateGeometricStateForAll(pBitMask: number, pBitValue: number): void;
    /**
     * Computes the Axis Aligned Bounding Box of geometries which have the specified visual state.
     *
     * Each geometry visual state is intersected with the pBitMask. If the resulting value is pBitValue
     * then the geometry bounding box will be used in the computation.
     *
     * Example : computing the axis aligned bounding box of visible geometries :
     *
     * ```typescript
     * [[include:InfiniteEngineInterface_bounding_box.ts]]
     * ```
     * <br>
     * Please refer to the {@link VisualStates} for a list of possible values.
     *
     * @param pBitMask <div class="const">in</div> the bit mask ("ORed" value of the {@link VisualStates}) of the states to check.
     * @param pBitValue <div class="const">in</div> the bit value ("ORed" value of the {@link VisualStates}) of the states that will be checked.
     * @param pAABB <div class="const">out</div> the computed AABB for the given bit composition of {@link VisualStates}.
     * @return `true` if at least one geometry is involved in the computation, else pAABB is left unchanged and false is returned.
     */
    getGeometryAABB(pBitMask: number, pBitValue: number, pAABB: AABB): boolean;
}
export { InfiniteEngineInterface };
/**
 * The {@link InfiniteFactory} factory is used to create the infinite engine, responsible for displaying the DMU.
 *
 * The {@link InfiniteEngineInterface} is optional, you may only use metadata requests to query the 3djuump infinite database.
 * The {@link MetadataManagerInterface} is linked to a {@link DirectorySessionInterface} and a {@link DataSessionInterface}.
 * When a DMU loading is in process, the {@link InfiniteEngineInterface} is updated accordingly from the {@link MetadataManagerInterface}.
 *
 * The bound {@link MetadataManagerInterface} cannot be changed, and two session loadings in parallel are not handled.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_creation.ts]]
 * ```
 * <br>
 * Please refer to the {@link InfiniteEngineInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Factories</div><br>
 * @see {@link InfiniteEngineInterface}
 * @category Factories
 */
declare class InfiniteFactory {
    /**
     * Instantiates an {@link InfiniteEngineInterface} linked to a {@link MetadataManagerInterface}.
     *
     * @param pMetadataManager <div class="const">in</div> The {@link MetadataManagerInterface} to be bound to.
     * @return The resulting {@link InfiniteEngineInterface}.
     */
    static CreateInfiniteEngine(pMetadataManager: MetadataManagerInterface): InfiniteEngineInterface;
    /**
     * Gets the font loader responsible to provide font data to the rendering of annotations.
     *
     * There is only one font loader.
     *
     * @return The resulting {@link FontLoaderInterface}.
     * @deprecated Please use {@link InfiniteApiControllerInterface.getFontLoader} instead.
     * @internal
     */
    static GetFontLoader(): FontLoaderInterface;
    /**
     * Gets the main {@link InfiniteApiControllerInterface}.
     *
     * The main {@link InfiniteApiControllerInterface} offers logging and
     * singleton accesses.
     *
     * @return The main {@link InfiniteApiControllerInterface}.
     */
    static GetInfiniteApiController(): InfiniteApiControllerInterface;
}
export { InfiniteFactory };
/**
 * The InfiniteObjectDispatcherInterfaceSignal lists all the signals that may be sent by any {@link InfiniteObjectDispatcherInterface}.
 *
 * These signals are emitted by any derived interface of {@link InfiniteObjectDispatcherInterface} when the following occurs :
 *
 * * {@link ObjectDisposed} : {@link InfiniteObjectInterface.dispose} has been called on the given object.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_dispose.ts]]
 * ```
 * <br>
 * Please refer to {@link InfiniteObjectDispatcherInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link InfiniteObjectDispatcherInterface}
 * @category Events
 */
declare enum InfiniteObjectDispatcherInterfaceSignal {
    /**
     * Signal sent by the {@link InfiniteObjectDispatcherInterface} at the end of a {@link InfiniteObjectInterface.dispose} call.
     *
     * No attachment for events of this type.
     */
    ObjectDisposed = "disposed"
}
export { InfiniteObjectDispatcherInterfaceSignal };
/**
 * The InfiniteObjectDispatcherInterface interface is the aggregation of an {@link InfiniteObjectInterface} and an
 * {@link EventDispatcherInterface}.
 *
 * The InfiniteObjectDispatcherInterface just provides a new event, {@link InfiniteObjectDispatcherInterfaceSignal.ObjectDisposed}
 * telling the given object has just been disposed, and that no more references to this object should be kept.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_dispose.ts]]
 * ```
 * <br>
 * All future functions called on this object will be a no-op, but a log message with a {@link LogLevel.LL_UsingDisposedObject}
 * criticality will be sent (see {@link InfiniteApiControllerInterface}).
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @see {@link InfiniteObjectDispatcherInterfaceSignal}
 * @see {@link InfiniteApiControllerInterface}
 * @category Main
 */
interface InfiniteObjectDispatcherInterface extends InfiniteObjectInterface, EventDispatcherInterface {
}
export { InfiniteObjectDispatcherInterface };
/**
 * The InfiniteObjectType enumeration tells the type of the {@link InfiniteObjectInterface} of the current object.
 *
 * This value is retrieved from {@link InfiniteObjectInterface.getInfiniteObjectType}.
 * The value of the enum is `"IOT_" + interface_name`.
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @category Main
 */
declare enum InfiniteObjectType {
    /**
     * The object is a {@link MetadataManagerInterface}.
     */
    IOT_MetadataManagerInterface = -1,
    /**
     * The object is a {@link DirectorySessionInterface}.
     */
    IOT_DirectorySessionInterface = 0,
    /**
     * The object is a {@link DataSessionInterface}.
     */
    IOT_DataSessionInterface = 1,
    /**
     * The object is a {@link InfiniteEngineInterface}.
     */
    IOT_InfiniteEngineInterface = 2,
    /**
     * The object is a {@link IdCardGetterInterface}.
     */
    IOT_IdCardGetterInterface = 100,
    /**
     * The object is a {@link ChildrenIdCardGetterInterface}.
     */
    IOT_ChildrenIdCardGetterInterface = 101,
    /**
     * The object is a {@link AnnotationGetterInterface}.
     */
    IOT_AnnotationGetterInterface = 102,
    /**
     * The object is a {@link DocumentIdConverterInterface}.
     */
    IOT_DocumentIdConverterInterface = 103,
    /**
     * The object is a {@link GeometricInstanceConverterInterface}.
     */
    IOT_GeometricInstanceConverterInterface = 104,
    /**
     * The object is a {@link PartInstanceConverterInterface}.
     */
    IOT_PartInstanceConverterInterface = 105,
    /**
     * The object is a {@link SearchInterface}.
     */
    IOT_SearchInterface = 106,
    /**
     * The object is a {@link DocumentContentGetterInterface}.
     */
    IOT_DocumentContentGetterInterface = 107,
    /**
     * The object is a {@link CutPlaneManipulatorInterface}.
     */
    IOT_CutPlaneManipulatorInterface = 200,
    /**
     * The object is a {@link ConfContextInterface}.
     */
    IOT_ConfContextInterface = 300,
    /**
     * The object is a {@link VisibilityContextInterface}.
     */
    IOT_VisibilityContextInterface = 301,
    /**
     * The object is a {@link FilterSolverInterface}.
     */
    IOT_FilterSolverInterface = 302,
    /**
     * The object is a {@link FilterSetInterface}.
     */
    IOT_FilterSetInterface = 400,
    /**
     * The object is a {@link FilterCompoundInterface}.
     */
    IOT_FilterCompoundInterface = 401,
    /**
     * The object is a {@link FilterHasFieldInterface}.
     */
    IOT_FilterHasFieldInterface = 500,
    /**
     * The object is a {@link FilterAttributeInterface}.
     */
    IOT_FilterAttributeInterface = 501,
    /**
     * The object is a {@link FilterRangeInterface}.
     */
    IOT_FilterRangeInterface = 502,
    /**
     * The object is a {@link FilterDiagonalInterface}.
     */
    IOT_FilterDiagonalInterface = 503,
    /**
     * The object is a {@link FilterAllPartsInterface}.
     */
    IOT_FilterAllPartsInterface = 504,
    /**
     * The object is a {@link FilterBooleanInterface}.
     */
    IOT_FilterBooleanInterface = 505,
    /**
     * The object is a {@link FilterLiteralInterface}.
     */
    IOT_FilterLiteralInterface = 506,
    /**
     * The object is a {@link FilterAABBInterface}.
     */
    IOT_FilterAABBInterface = 507,
    /**
     * The object is a {@link FilterPartInstanceListInterface}.
     */
    IOT_FilterPartInstanceListInterface = 508
}
export { InfiniteObjectType };
/**
 * The InfiniteObjectInterface is the base class of most interfaces.
 *
 * It merely defines a way to get rid of an interface, and
 * get information about the type of the given interface.
 *
 * [[include:constStyle.md]]<br><div class="const">Main</div><br>
 * @category Main
 */
interface InfiniteObjectInterface {
    /**
     * Gets rid of this object.
     *
     * After this call, this object can no longer be used.
     *
     * If the object is an {@link InfiniteObjectDispatcherInterface}, then the {@link InfiniteObjectDispatcherInterfaceSignal.ObjectDisposed} signal
     * is emitted.
     *
     * Further uses of the object (with the exception of {@link isDisposed} and {@link getInfiniteObjectType}) will log a message with {@link LogLevel.LL_UsingDisposedObject}.
     *
     * @see {@link LogLevel}
     * @see {@link InfiniteObjectDispatcherInterfaceSignal}
     * @see {@link InfiniteObjectDispatcherInterface}
     */
    dispose(): void;
    /**
     * Tells if this object has been gotten rid off.
     *
     * @return `true` if {@link dispose} has been called on this object.
     */
    isDisposed(): boolean;
    /**
     * Tells the type of the given interface.
     *
     * @return The type of the given interface.
     */
    getInfiniteObjectType(): InfiniteObjectType;
}
export { InfiniteObjectInterface };
/**
 * The MaterialManagerInterface interface allows the coloring of the DMU.
 *
 * This interface is used to create and set materials for `geometric instances` from their `geometric instance ids`.
 * All materials are accessed by their `material id`, there is no specific "material objects" to handle.
 *
 * **WARNING** : there is a fixed limit for the total number of different materials that may be in use.
 *
 * **Original** materials refer to the materials loaded from the DMU and **custom** materials refer
 * to the API created materials. Depending on the number of **original** materials of the DMU,
 * there may be a different number of **custom** materials that may be created. The number of **custom** materials
 * that may be created is given by {@link getNbAvailableCustomMaterials}.
 * Only **custom** materials can be created, modified or removed; **original** materials of the DMU can only be queried but not
 * modified.
 *
 * A single `geometric instance` may have multiple **original** materials, in case of a multi material geometry. For these geometries, it is
 * **NOT** possible to change the coloring of each individual sub-geometry. In case of a **custom** material application, the whole geometry
 * color will be changed.
 *
 * The MaterialManagerInterface is a member of the {@link InfiniteEngineInterface}, accessed through {@link InfiniteEngineInterface.getMaterialManager}.
 *
 * ```typescript
 * [[include:MaterialManagerInterface_example.ts]]
 * ```
 * <br>
 * Please refer to the {@link InfiniteEngineInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link InfiniteEngineInterface}
 * @category 3D Rendering
 */
interface MaterialManagerInterface {
    /**
     * Gets the number of **custom** materials that may be created.
     *
     * This value does not depend on the **custom** materials already created. Once the DMU is loaded,
     * you can expect this value to be constant.
     *
     * @return The number of **custom** materials that may be created.
     */
    getNbAvailableCustomMaterials(): number;
    /**
     * Gets the number of **custom** materials currently in use.
     *
     * @return The number of already created **custom** materials.
     */
    getNbCustomMaterials(): number;
    /**
     * Creates a new **custom** material of the given color.
     *
     * @param pDiffuse <div class="const">in</div> The new diffuse color (3 floats in the range [0:1], x is red, y is green, b is black).
     * @return The id of the new material, or -1 if the material could not be created (i.e. the limit of materials has been reached).
     * @see {@link getNbAvailableCustomMaterials}
     */
    createNewMaterial(pDiffuse: Vector3): number;
    /**
     * Removes the given material from the list of **custom** materials.
     *
     * @param pMaterialId <div class="const">in</div> The id of the custom material to remove.
     * @return `true` if the material has been removed (exists and is a **custom** material).
     */
    removeMaterial(pMaterialId: number): boolean;
    /**
     * Gets a copy of the array containing the list of the **custom** `material ids` currently in use.
     *
     * @return The array of **custom** <code>material ids</code>.
     */
    getMaterialIds(): Array<number>;
    /**
     * Changes the material of the given `geometric instance` to a given **custom** material.
     *
     * Returns `false` if the material does not exist , is not a **custom** one or if the `geometric instance id` is invalid.
     * The **custom** material must have been created before being used (see {@link createNewMaterial}).
     *
     * @param pGeometricInstanceId <div class="const">in</div> The <code>geometric instance id</code> of the <code>geometric instance</code> to change.
     * @param pMaterialId <div class="const">in</div> The material to set (<code>material id</code> as returned by {@link createNewMaterial}).
     * @return `true` if the material has been changed.
     */
    changeMaterialOfInstance(pGeometricInstanceId: number, pMaterialId: number): boolean;
    /**
     * Changes the material of the given `geometric instances` to a given **custom** material.
     *
     * Returns false if the material does not exist, is not a **custom** one or if any `geometric instance id` is invalid.
     * The **custom** material must have been created before being used (see {@link createNewMaterial}).
     *
     * @param pGeometricInstanceIds <div class="const">in</div> The <code>geometric instance ids</code> of the <code>geometric instances</code> to change.
     * @param pMaterialId <div class="const">in</div> The material to set (<code>material id</code> as returned by {@link createNewMaterial}).
     * @return `true` if the material has been changed.
     */
    changeMaterialOfInstances(pGeometricInstanceIds: Uint32Array | number[], pMaterialId: number): boolean;
    /**
     * Changes the material of all `geometric instances` to a given **custom** material.
     *
     * Returns false if the material does not exist, is not a **custom** one.
     * The **custom** material must have been created before being used (see {@link createNewMaterial}).
     *
     * @param pMaterialId <div class="const">in</div> The material to set (<code>material id</code> as returned by {@link createNewMaterial}).
     * @return `true` if the material has been changed.
     */
    changeMaterialOfAllInstances(pMaterialId: number): boolean;
    /**
     * Restores the `geometric instance` to its original color.
     *
     * Restoring a material of a `geometric instance` that still has its **original** material is legal and therefore returns true.
     *
     * @param pGeometricInstanceId <div class="const">in</div> The <code>geometric instance id</code> of the
     * <code>geometric instance</code> to restore.
     * @return `true` if the material has been restored (be it **custom** or not), false if the <code>geometric instance id</code> is invalid.
     */
    restoreOriginalMaterialOfInstance(pGeometricInstanceId: number): boolean;
    /**
     * Restores all the `geometric instances` to their original color.
     */
    restoreOriginalMaterialOfAllInstances(): void;
    /**
     * Restores the `geometric instances` to their original colors.
     *
     * Restoring a material of a `geometric instance` that still has its **original** material is legal.
     * All `geometric instance ids` must be valid in order to success.
     *
     * @param pGeometricInstanceIds <div class="const">in</div> The <code>geometric instance ids</code> of the <code>geometric instances</code> to restore.
     * @return `true` if the material has been restored, false if an invalid id is found in the <code>geometric instance ids</code>.
     */
    restoreOriginalMaterialOfInstances(pGeometricInstanceIds: Uint32Array | number[]): boolean;
    /**
     * Gets the current material applied to the given instance, returns -1 if the `geometric instance id` is invalid.
     *
     * The returned value can refer either to a **custom** `material id` or an **original** `material id`.
     * In case of an **original** material (see {@link isOriginalMaterial}), the id corresponds to the first id of
     * {@link getOriginalMaterialsOfInstance}. Indeed, in case of a multi-material geometry, a single `geometric instance` may have
     * multiple **original** materials.
     *
     * @param pGeometricInstanceId <div class="const">in</div> The instance id number to retrieve the current <code>material id</code> from.
     * @return The current <code>material id</code> of the instance, -1 if the <code>geometric instance id</code> is invalid.
     */
    getMaterialOfInstance(pGeometricInstanceId: number): number;
    /**
     * Retrieves all **original** `material ids` of the given <code>geometric instance</code>, even if a **custom** material
     * has been applied to the <code>geometric instance</code>.
     *
     * This method is recommended when it is certain that the material of the given instance *has not* been overridden, else prefer {@link getMaterialOfInstance}.
     *
     * @param pGeometricInstanceId <div class="const">in</div> The <code>geometric instance id</code> to retrieve the original <code>material ids</code> from.
     * @param pMaterialIdsOut <div class="const">out</div> The resulting original <code>material ids</code> of the given <code>geometric instance id</code>.
     * @return `true` if the list of <code>material ids</code> has been retrieved (thus pGeometricInstanceId is valid).
     */
    getOriginalMaterialsOfInstance(pGeometricInstanceId: number, pMaterialIdsOut: number[]): boolean;
    /**
     * Gets the number of **original** materials of the given instance, even if a **custom** material has been applied to the <code>geometric instance</code>.
     *
     * Any `geometric instance` can have multiple **original** materials.
     *
     * @param pGeometricInstanceId <div class="const">in</div> The <code>geometric instance id</code> to retrieve the original <code>material ids</code> from.
     * @return The current number of materials of the given <code>geometric instance</code> (at least 1), -1 if the given <code>geometric instance id</code> is invalid.
     */
    getOriginalNbMaterialsOfInstance(pGeometricInstanceId: number): number;
    /**
     * Modifies the color of a **custom** material, changing its rendering parameters.
     *
     * It is **NOT** possible to edit **original** materials of the DMU.
     *
     * @param pMaterialId <div class="const">in</div> The custom <code>material id</code> of the material to edit.
     * @param pDiffuse <div class="const">in</div> The new diffuse color (3 floats in the range [0:1], red, green, blue).
     * @return `true` if the material was changed (i.e. the <code>material id</code> was correct and not an **original** <code>material id</code>).
     */
    modifyMaterial(pMaterialId: number, pDiffuse: Vector3): boolean;
    /**
     * Gets the diffuse color of the material.
     *
     * @param pMaterialId <div class="const">in</div> The <code>material id</code> to query.
     * @param pDiffuseOut <div class="const">out</div> The diffuse color of the given material.
     * @return `true` if pDiffuse was modified, and therefore the material exists.
     */
    getMaterialDiffuse(pMaterialId: number, pDiffuseOut: Vector3): boolean;
    /**
     * Tells if the material corresponding to the given `material id` is an already created **custom** material.
     *
     * @param pMaterialId <div class="const">in</div> The <code>material id</code> to check.
     * @return `true` if the **custom** material exists.
     */
    hasCustomMaterial(pMaterialId: number): boolean;
    /**
     * Tells if the material corresponding to the given `material id` is included in the DMU s **original** materials
     * (in opposition to **custom** materials).
     *
     * @param pMaterialId <div class="const">in</div> The <code>material id</code> to check.
     * @return `true` if the <code>material id</code> exists and is an **original** material.
     */
    isOriginalMaterial(pMaterialId: number): boolean;
}
export { MaterialManagerInterface };
/**
 * The MessageInterface is the attachment of the event in the {@link DataSessionInterfaceSignal.MessageReceived} signal.
 *
 * The MessageInterface describes messages sent to the users by the administrator of the 3djuump infinite architecture.
 * It contains the text of the message, if it is a critical message, a unique id, the start date when the message
 * will be or is applicable, and an end date when the message is no longer valid.
 *
 * ```typescript
 * [[include:MessageInterface_handling.ts]]
 * ```
 * <br>
 * Please refer to the {@link DataSessionInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link DataSessionInterfaceSignal}
 * @category Events
 */
interface MessageInterface {
    /**
     * Gets the content of the message.
     *
     * @return The message content.
     */
    getMessage(): string;
    /**
     * Gets the unique id of the message.
     *
     * @return The unique id of the message.
     */
    getId(): number;
    /**
     * Tells if the message is a critical message (i.e. shutdown server for maintenance).
     *
     * @return `true` if it is a critical message.
     */
    isCritical(): boolean;
    /**
     * Gets the start time of the message.
     *
     * @return The time as a Date.
     */
    getStartTime(): Date;
    /**
     * Gets the end time of the message.
     *
     * @return The time as a Date.
     */
    getEndTime(): Date;
}
export { MessageInterface };
/**
 * This factory is used to create the {@link MetadataManagerInterface}.
 *
 * The {@link MetadataManagerInterface} is the central point to connect to a 3djuump infinite build.
 *
 * Connecting to a 3djuump infinite build involves :
 * * A {@link MetadataManagerInterface}.
 * * A {@link DirectorySessionInterface}.
 * * A {@link DataSessionInterface}.
 * * An optional {@link InfiniteEngineInterface}.
 *
 * At any time, the {@link MetadataManagerInterface} can only be connected to a single {@link DirectorySessionInterface}, {@link DataSessionInterface}
 * and {@link InfiniteEngineInterface}, but the same {@link MetadataManagerInterface} may be used to connect to a new {@link DataSessionInterface} after the preceding
 * has been closed.
 *
 * ```typescript
 * [[include:api_popup_authentication.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#authentication-samples">Authentication Samples</a>.
 *
 * [[include:constStyle.md]]<br><div class="const">Factories</div><br>
 * @see {@link MetadataManagerInterface}
 * @category Factories
 */
declare class MetadataManagerFactory {
    /**
     * Instantiates a {@link MetadataManagerInterface}.
     *
     * @return The resulting {@link MetadataManagerInterface}.
     */
    static CreateMetadataManager(): MetadataManagerInterface;
}
export { MetadataManagerFactory };
/**
 * The MetadataManagerInterface is the object that links interfaces of the 3djuump infinite API together.
 *
 * A connection to an infinite build relies on 3/4 interfaces linked together :
 * * a MetadataManagerInterface.
 * * a {@link DirectorySessionInterface} linked to the MetadataManagerInterface.
 * * a {@link DataSessionInterface} linked to the {@link DirectorySessionInterface} and the MetadataManagerInterface.
 * * an optional {@link InfiniteEngineInterface} linked to the MetadataManagerInterface.
 *
 * The MetadataManagerInterface is the first interface to be created, through the {@link MetadataManagerFactory.CreateMetadataManager}.
 *
 * ```typescript
 * [[include:api_samepage_authentication.ts]]
 * ```
 * <br>
 * At any moment, these 4 interfaces can be linked to at most one infinite build, but an infinite build can be closed
 * and another open afterwards. If you need to connect to several infinite builds at once, you will need to create
 * multiple sets of &#123;MetadataManagerInterface / {@link DirectorySessionInterface} / {@link DataSessionInterface} / {@link InfiniteEngineInterface}&#125;.
 *
 * Performance information ({@link PerformanceInterface}) are accessed through the MetadataManagerInterface.
 *
 * The MetadataManagerInterface will be removed in the next version. The relevant api has already been moved to the {@link DataSessionInterface}.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @see {@link DataSessionInterface}
 * @see {@link FilterSolverInterface}
 * @see {@link ConfContextInterface}
 * @see {@link FilterItemInterface}
 * @see {@link VisibilityContextInterface}
 * @category Metadata
 */
interface MetadataManagerInterface extends InfiniteObjectInterface {
    /**
     * Gets the version of the infinite API.
     *
     * @return The version of the infinite API.
     * @deprecated Please use {@link InfiniteApiControllerInterface.getVersion}.
     */
    getVersion(): string;
    /**
     * Gets the object tweaking rendering/cache performance.
     *
     * Cache size, max number of triangles, etc can be
     * manipulated from the {@link PerformanceInterface}.
     *
     * @return The performance interface used to tweak performance.
     */
    getPerformanceInfos(): PerformanceInterface;
    /**
     * Gets the attributes dictionary of the DMU.
     *
     * The {@link AttributesDictionaryInterface} is a read only interface to know all
     * the available attributes and their types.
     *
     * @return <div class="const">const</div> The attribute dictionary of the DMU.
     * @deprecated Please use {@link DataSessionInterface.getAttributesDictionary} instead.
     */
    getAttributesDictionary(): AttributesDictionaryInterface;
    /**
     * Gets the list of configurations.
     *
     * Please see <a href="../index.html#configurations">configurations</a> for more
     * explanations about configurations.
     *
     * Modifying this array in place results in undefined behavior.
     *
     * DO NOT modify this array.
     *
     * @return <div class="const">const</div> The list of configurations.
     * @deprecated Please use {@link DataSessionInterface.getConfigurationList} instead.
     */
    getConfigurationList(): Array<ConfigurationInterface>;
    /**
     * Gets the list of available annotation types.
     *
     * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
     *
     * Modifying this array in place results in undefined behavior.
     *
     * DO NOT modify this array.
     *
     * @return <div class="const">const</div> The list of annotation types.
     * @deprecated Please use {@link DataSessionInterface.getAnnotationTypes} instead.
     */
    getAnnotationTypes(): Array<string>;
    /**
     * Gets a project document by its id.
     *
     * Available project documents are :
     * * <span style="text-decoration: line-through">com.3djuump:scripts (idcard customization script) </span> (deprecated).
     * * com.3djuump:defaultsettings (field of view, orientation, etc).
     * * com.3djuump:indexinfo (elasticsearch index information) : this information is accessible through the {@link AttributesDictionaryInterface}.
     *
     * Please refer to the integration manual to have more information about these
     * documents.
     *
     * It is very unlikely any developer will use these versioned documents.
     *
     * Example (**illustration purpose only**):
     *
     * ```javascript
     * [[include:project_documents.js]]
     * ```
     * <br>
     * These documents are json documents.
     *
     * Returns the json document as a string if the given document is available in the 3djuump infinite project.
     *
     * It is very unlikely any developer will use
     * {@link getProjectDocument | getProjectDocument("com.3djuump:scripts")} as this script should be used for the
     * 3djuump infinite native client.
     *
     * @param pDocumentId <div class="const">in</div> The project document id to fetch.
     * @return The document as a string, or undefined if the document could not be found or the DMU is not loaded.
     * @deprecated Please use {@link DataSessionInterface.getProjectDocument} instead.
     */
    getProjectDocument(pDocumentId: string): string | undefined;
    /**
     * Gets the default customization script as a string.
     *
     * This call is deprecated since it may induce a content security flow (see Content-Security-Policy
     * {@link https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Content-Security-Policy/script-src}).
     *
     * <div style="text-decoration: line-through">
     * In case the [[getProjectDocument | getProjectDocument("com.3djuump:scripts")] is not valid, the
     * API provides a default javascript `"com.3djuump:scripts"."scriptbase64"` as a string value (not base64 encoded).
     *
     * This script is mainly used for idcard display customization (in the case of the 3djuump infinite native client).
     *
     * It is very unlikely any developer will use
     * {@link getProjectDocument | getProjectDocument("com.3djuump:scripts")} as this script should be used for the
     * 3djuump infinite native client.
     *
     * Please refer to the integration manual to have more information about the
     * "com.3djuump:scripts" document.
     *
     * @return The default `"com.3djuump:scripts"."scriptbase64"` javascript script as a string value.
     *
     * @see {@link getProjectDocument}
     * @deprecated
     * @internal
     */
    getDefaultCustomizationScript(): string;
    /**
     * Gets the axis aligned bounding box of the given `geometric instance id`.
     *
     * Call this function after the DMU has been loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess})
     * with a previously created {@link AABB}.
     *
     * Returns true if the given `geometric instance id` is valid.
     *
     * @param pGeometricInstanceId <div class="const">in</div> The <code>geometric instance id</code> to query.
     * @param pAABBOut <div class="const">out</div> The resulting AABB of the <code>geometric instance</code>.
     * @return `true` if pAABBOut is updated.
     * @deprecated Please use {@link DataSessionInterface.getGeometricAABB} instead.
     */
    getGeometricAABB(pGeometricInstanceId: number, pAABBOut: AABB): boolean;
    /**
     * Gets the diagonal **squared** length of the Oriented Bounding Box (OBB) of the given `geometric instance id`.
     *
     * Returns -1 if pGeometricInstanceId is an invalid  `geometric instance id`, or the diagonal squared length of the OBB if it is valid.
     *
     * @param pGeometricInstanceId <div class="const">in</div> The <code>geometric instance id</code> to query.
     * @return The diagonal **squared** length of the OBB of the <code>geometric instance</code> or -1 if invalid.
     * @deprecated Please use {@link DataSessionInterface.getGeometricOBBDiagonalSquaredLength} instead.
     */
    getGeometricOBBDiagonalSquaredLength(pGeometricInstanceId: number): number;
    /**
     * Gets the minimum diagonal **squared** length of all the Oriented Bounding Box (OBB) of all the `geometric instances`.
     *
     * Returns the minimum diagonal squared length of the current DMU, or -1 if a DMU has not been loaded ().
     *
     * @return The minimum diagonal squared length of the current DMU, or -1 if no DMU is loaded.
     * @deprecated Please use {@link DataSessionInterface.getGeometricOBBDiagonalSquaredMin} instead.
     */
    getGeometricOBBDiagonalSquaredMin(): number;
    /**
     * Gets the maximum diagonal **squared** length of all the Oriented Bounding Box (OBB) of all the `geometric instances`.
     *
     * Returns the maximum diagonal squared length of the current DMU, or -1 if a DMU has not been loaded
     * ({@link DataSessionInterfaceSignal.DMULoadingSuccess}).
     *
     * @return The maximum diagonal squared length of the current DMU, or -1 if no DMU is loaded.
     * @deprecated Please use {@link DataSessionInterface.getGeometricOBBDiagonalSquaredMax} instead.
     */
    getGeometricOBBDiagonalSquaredMax(): number;
    /**
     * Gets the {@link Unit} of the DMU currently loaded.
     *
     * Returns {@link Unit.U_Invalid} if no DMU is loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess}).
     *
     * @return The {@link Unit} of the DMU (millimeters, centimeters, meters, etc).
     * @deprecated Please use {@link DataSessionInterface.getDmuBuildUnit} instead.
     */
    getDmuBuildUnit(): Unit;
    /**
     * Gets the Axis Aligned Bounding Box of the currently loaded DMU.
     *
     * Returns `false` if no DMU is loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess}).
     *
     * @param pDmuAABBOut <div class="const">out</div> AABB of the DMU.
     * @return `true` if a DMU is loaded and pDmuAABBOut is updated.
     * @deprecated Please use {@link DataSessionInterface.getDmuAABB} instead.
     */
    getDmuAABB(pDmuAABBOut: AABB): boolean;
    /**
     * Gets the maximum `geometric instance id` of the DMU.
     *
     * Valid `geometric instance ids` range from `1` to <code>{@link getMaximumGeometricId | getMaximumGeometricId()}</code> included.
     * Please refer to <a href="../index.html#main-id-types">Main ID Types</a> for more information.
     *
     * Returns the maximum `geometric instance id` if a DMU is loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess},
     * 0 else.
     *
     * @return The maximum <code>geometric instance id</code>, 0 if no DMU is loaded.
     * @deprecated Please use {@link DataSessionInterface.getMaximumGeometricId} instead.
     */
    getMaximumGeometricId(): number;
    /**
     * Gets the maximum `part instance id` of the DMU.
     *
     * Valid `part instance ids` range from `1` to <code>{@link getMaximumPartInstanceId | getMaximumPartInstanceId()}</code> included.
     * Please refer to <a href="../index.html#main-id-types">Main ID Types</a> for more information.
     *
     * Returns the maximum `part instance id` if a DMU is loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess},
     * 0 else.
     *
     * @return The maximum <code>part instance id</code>, 0 if no DMU is loaded.
     * @deprecated Please use {@link DataSessionInterface.getMaximumPartInstanceId} instead.
     */
    getMaximumPartInstanceId(): number;
    /**
     * Instantiates a filtering context.
     *
     * A filtering context ({@link VisibilityContextInterface}) is composed of a list of
     * {@link FilterSolverInterface | FilterSolverInterface(s)},
     * and a required {@link ConfContextInterface}. Filtering context may be used
     * to restrict the next filtering requests
     * to a set a specific `part instances`.
     *
     * **DO NOT** create {@link VisibilityContextInterface} before the DMU is loaded, such {@link VisibilityContextInterface} will be silently discarded.
     *
     * **Discard** existing {@link VisibilityContextInterface} before loading a new {@link DataSessionInterface}.
     *
     * @return A new filtering context bound to the currently loaded DMU.
     * @see {@link VisibilityContextInterface}
     * @deprecated Please use {@link DataSessionInterface.createVisibilityContext} instead.
     */
    createVisibilityContext(): VisibilityContextInterface;
    /**
     * Gets rid of a {@link VisibilityContextInterface}.
     *
     * You may cleanup existing references in your application to allow the {@link VisibilityContextInterface} to be garbage collected.
     * Such a {@link VisibilityContextInterface} will no longer be used in the 3djuump infinite API.
     *
     * @param pVisibilityContext <div class="const">in</div> The {@link VisibilityContextInterface} to discard.
     * @return `true` if such a {@link VisibilityContextInterface} is valid and has not been removed yet.
     * @see {@link VisibilityContextInterface}
     * @deprecated Please use {@link VisibilityContextInterface.dispose} instead.
     */
    removeVisibilityContext(pVisibilityContext: VisibilityContextInterface): boolean;
    /**
     * Instantiates a {@link ConfContextInterface}.
     *
     * A {@link ConfContextInterface} MUST be used inside a {@link VisibilityContextInterface} to create a valid filtering context.
     * The {@link ConfContextInterface} is the expression of a selected <a href="../index.html#configurations">configuration</a>.
     *
     * The {@link ConfContextInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link ConfContextInterface} before the DMU is loaded, such {@link ConfContextInterface} will be silently discarded.
     *
     * **Discard** existing {@link ConfContextInterface} before loading a new {@link DataSessionInterface} with {@link ConfContextInterface.dispose}.
     *
     * @return A new {@link ConfContextInterface}.
     * @see {@link ConfContextInterface}
     * @deprecated Please use {@link DataSessionInterface.createConfContext} instead.
     */
    createConfContext(): ConfContextInterface;
    /**
     * Gets rid of a {@link ConfContextInterface}.
     *
     * You may cleanup existing references in your application to allow the {@link ConfContextInterface} to be garbage collected.
     * Such a {@link ConfContextInterface} will no longer be used in the 3djuump infinite API.
     *
     * @param pConfContext <div class="const">in</div> The {@link ConfContextInterface} to discard.
     * @return `true` if such a {@link ConfContextInterface} is valid and has not been removed yet.
     * @see {@link ConfContextInterface}
     * @deprecated Please use {@link ConfContextInterface.dispose} instead.
     */
    removeConfContext(pConfContext: ConfContextInterface): boolean;
    /**
     * Instantiates a {@link FilterSolverInterface}.
     *
     * A {@link FilterSolverInterface} is the expression of a combination of multiple
     * {@link FilterItemInterface | FilterItemInterfaces} with a specific operator to create
     * `part instance ids` sets and `geometric instance ids` sets.
     *
     * The {@link FilterSolverInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterSolverInterface} before the DMU is loaded, such {@link FilterSolverInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterSolverInterface} before loading a new {@link DataSessionInterface} with {@link FilterSolverInterface.dispose}.
     *
     * @return A new {@link FilterSolverInterface}.
     * @see {@link FilterSolverInterface}
     * @deprecated Please use {@link DataSessionInterface.createFilterSolver} instead.
     */
    createFilterSolver(): FilterSolverInterface;
    /**
     * Gets rid of a {@link FilterSolverInterface}.
     *
     * You may cleanup existing references in your application to allow the {@link FilterSolverInterface} to be garbage collected.
     * Such a {@link FilterSolverInterface} will no longer be used in the 3djuump infinite API.
     *
     * @param pFilterSolver <div class="const">in</div> The {@link FilterSolverInterface} to discard.
     * @return `true` if such a {@link FilterSolverInterface} is valid and has not been removed yet.
     * @see {@link FilterSolverInterface}
     * @deprecated Please use {@link FilterSolverInterface.dispose} instead.
     */
    removeFilterSolver(pFilterSolver: FilterSolverInterface): boolean;
    /**
     * Instantiates an all `part instances` filter.
     *
     * A {@link FilterAllPartsInterface} is a {@link FilterItemInterface} that elects all `part instances`
     * of the DMU. It is the most optimized way to create a filter that contains all `part instances`.
     *
     * The {@link FilterAllPartsInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterAllPartsInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterAllPartsInterface} before the DMU is loaded, such {@link FilterAllPartsInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterAllPartsInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterAllPartsInterface}.
     * @see {@link FilterAllPartsInterface}
     * @deprecated Please use {@link DataSessionInterface.createFilterAllParts} instead.
     */
    createFilterAllParts(): FilterAllPartsInterface;
    /**
     * Instantiates a metadata text attribute value filter.
     *
     * A {@link FilterAttributeInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that contain in their `joined attribute set` a string attribute whose value matches with the ones contained inside
     * the {@link FilterAttributeInterface}.
     *
     * The {@link FilterAttributeInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface} or {@link FilterCompoundInterface}).
     *
     * The {@link FilterAttributeInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterAttributeInterface} before the DMU is loaded, such {@link FilterAttributeInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterAttributeInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterAttributeInterface}.
     * @see {@link FilterAttributeInterface}
     * @deprecated Please use {@link DataSessionInterface.createFilterAttribute} instead.
     */
    createFilterAttribute(): FilterAttributeInterface;
    /**
     * Instantiates a metadata numeric attribute value filter.
     *
     * A {@link FilterRangeInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that contain in their `joined attribute set` a numeric attribute (number, date, number range, date range) whose value
     * is contained within the ranges contained inside
     * the {@link FilterRangeInterface}.
     *
     * The {@link FilterRangeInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface} or {@link FilterCompoundInterface}).
     *
     * The {@link FilterRangeInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterRangeInterface} before the DMU is loaded, such {@link FilterRangeInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterRangeInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterRangeInterface}.
     * @see {@link FilterRangeInterface}
     * @deprecated Please use {@link DataSessionInterface.createFilterRange} instead.
     */
    createFilterRange(): FilterRangeInterface;
    /**
     * Instantiates a metadata attribute filter.
     *
     * A {@link FilterHasFieldInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that contain in their `joined attribute set` an attribute whose name
     * is set by the {@link FilterHasFieldInterface}.
     *
     * The {@link FilterHasFieldInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface} or {@link FilterCompoundInterface}).
     *
     * The {@link FilterHasFieldInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterHasFieldInterface} before the DMU is loaded, such {@link FilterHasFieldInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterHasFieldInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterHasFieldInterface}.
     * @see {@link FilterHasFieldInterface}
     * @deprecated Please use {@link DataSessionInterface.createFilterHasField} instead.
     */
    createFilterHasField(): FilterHasFieldInterface;
    /**
     * Instantiates a metadata boolean attribute value filter.
     *
     * A {@link FilterBooleanInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that contain in their `joined attribute set` a boolean attribute (true, false) whose value
     * is set by
     * the {@link FilterBooleanInterface}.
     *
     * The {@link FilterBooleanInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface} or {@link FilterCompoundInterface}).
     *
     * The {@link FilterBooleanInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterBooleanInterface} before the DMU is loaded, such {@link FilterBooleanInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterBooleanInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterBooleanInterface}.
     * @see {@link FilterBooleanInterface}
     * @deprecated Please use {@link DataSessionInterface.createBooleanFilter} instead.
     */
    createBooleanFilter(): FilterBooleanInterface;
    /**
     * Instantiates a query string filter.
     *
     * A {@link FilterLiteralInterface} is a {@link FilterItemInterface} that elects `part instances`
     * that have at least one individual metadata document that satisfies the string query set by the {@link FilterLiteralInterface}.
     *
     * Such a query should be written in the <a href="../index.html#3djuump-infinite-literal-and-search-query-language">
     * 3djuump infinite literal and search query language</a>.
     *
     * The {@link FilterLiteralInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterLiteralInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterLiteralInterface} before the DMU is loaded, such {@link FilterLiteralInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterLiteralInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterLiteralInterface}.
     * @see {@link FilterLiteralInterface}
     * @deprecated Please use {@link DataSessionInterface.createLiteralFilter} instead.
     */
    createLiteralFilter(): FilterLiteralInterface;
    /**
     * Instantiates a group metadata filter.
     *
     * A {@link FilterCompoundInterface} is a group filter that elects `documents`
     * whose metadata satisfy the clauses contained inside ({@link FilterAttributeInterface},
     * {@link FilterRangeInterface}, {@link FilterHasFieldInterface}, {@link FilterBooleanInterface}).<br>
     * When the set of documents that match the clauses are found, then `part instances` that
     * are referenced by these documents are elected.
     *
     * The {@link FilterCompoundInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterCompoundInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterCompoundInterface} before the DMU is loaded, such {@link FilterCompoundInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterCompoundInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterCompoundInterface}.
     * @see {@link FilterCompoundInterface}
     * @deprecated Please use {@link DataSessionInterface.createCompoundFilter} instead.
     */
    createCompoundFilter(): FilterCompoundInterface;
    /**
     * Instantiates an oriented bounding box diagonal **SQUARED** length filter.
     *
     * A {@link FilterDiagonalInterface} is a {@link FilterItemInterface} that elects `part instances`
     * whom oriented bounding box diagonal **SQUARED** length is contained within the ranges
     * specified in the {@link FilterDiagonalInterface}.
     *
     * The {@link FilterDiagonalInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterDiagonalInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterDiagonalInterface} before the DMU is loaded, such {@link FilterDiagonalInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterDiagonalInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterDiagonalInterface}.
     * @see {@link FilterDiagonalInterface}
     * @deprecated Please use {@link DataSessionInterface.createDiagonalFilter} instead.
     */
    createDiagonalFilter(): FilterDiagonalInterface;
    /**
     * Instantiates an axis aligned bounding box filter.
     *
     * A {@link FilterAABBInterface} is a {@link FilterItemInterface} that elects `part instances`
     * whom triangles intersect the {@link FilterAABBInterface} {@link AABB}.
     *
     * The {@link FilterAABBInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterAABBInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterAABBInterface} before the DMU is loaded, such {@link FilterAABBInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterAABBInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterAABBInterface}.
     * @see {@link FilterAABBInterface}
     * @deprecated Please use {@link DataSessionInterface.createFilterAABB} instead.
     */
    createFilterAABB(): FilterAABBInterface;
    /**
     * Instantiates a `part instance ids` list filter.
     *
     * A {@link FilterPartInstanceListInterface} is a {@link FilterItemInterface} that elects `part instances`
     * whose `part instance id` is contained in the {@link FilterPartInstanceListInterface} `part instance ids` list.
     *
     * **WARNING** : using a {@link FilterPartInstanceListInterface} is very risky since `part instance ids` are recomputed at
     * each new 3djuump Infinite build process.
     *
     * The {@link FilterPartInstanceListInterface} should be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}).
     *
     * The {@link FilterPartInstanceListInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterPartInstanceListInterface} before the DMU is loaded, such {@link FilterPartInstanceListInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterPartInstanceListInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterPartInstanceListInterface}.
     * @see {@link FilterPartInstanceListInterface}
     * @deprecated Please use {@link DataSessionInterface.createFilterPartInstanceList} instead.
     */
    createFilterPartInstanceList(): FilterPartInstanceListInterface;
    /**
     * Instantiates a "group" filter.
     *
     * A {@link FilterSetInterface} is a {@link FilterItemInterface} that allows grouping
     * [FilterItemInterface | FilterItemInterfaces] to introduce an operator precedence
     * between them. It is similar to a "parenthesis" system (see {@link FilterOperator}).
     *
     * The {@link FilterSetInterface} can be used inside a container ({@link FilterSolverInterface},
     * {@link FilterSetInterface}), but beware there is a hard coded limit on the number of
     * imbricated {@link FilterSetInterface} (a {@link FilterSetInterface} inside a {@link FilterSetInterface}
     * inside a {@link FilterSetInterface} etc ...).<br>
     * Please refer to {@link FilterSetInterface} and {@link FilterItemInterface.getDepthContribution}.
     *
     * The {@link FilterSetInterface} is bound to the currently loaded DMU.
     *
     * **DO NOT** create {@link FilterSetInterface} before the DMU is loaded, such {@link FilterSetInterface} will be silently discarded.
     *
     * **Discard** existing {@link FilterSetInterface} before loading a new {@link DataSessionInterface} with {@link destroyFilter}.
     *
     * @return A new {@link FilterSetInterface}.
     * @see {@link FilterSetInterface}
     * @see {@link FilterOperator}
     * @deprecated Please use {@link DataSessionInterface.createFilterSet} instead.
     */
    createFilterSet(): FilterSetInterface;
    /**
     * Gets rid of a {@link FilterItemInterface}.
     *
     * Such a {@link FilterItemInterface} is removed from its parent FilerSetInterface, all its children
     * {@link FilterItemInterface} are destroyed and clear.
     *
     * A {@link FilterItemInterface} that was destroyed can no longer be used, the application should clear
     * references to it to allow it being garbage collected.
     *
     * Returns true if the {@link FilterItemInterface} has been destroyed, false if it has already been destroyed
     * previously.
     *
     * @param pFilter <div class="const">in</div> The FilterItemInterface to destroy.
     * @return `true` if cleaned up.
     * @see {@link FilterItemInterface}
     * @deprecated Please use {@link FilterItemInterface.dispose} instead.
     */
    destroyFilter(pFilter: FilterItemInterface): boolean;
    /**
     * Instantiates a converter to translate a `document id` to its corresponding `part instance ids` and `geometric instance ids`.
     *
     * A {@link DocumentIdConverterInterface} may be used in a <a href="../index.html#search">search procedure</a>.
     *
     * **DO NOT** create {@link DocumentIdConverterInterface} before the DMU is loaded,
     * such {@link DocumentIdConverterInterface} will be silently discarded.
     *
     * @return A {@link DocumentIdConverterInterface}.
     * @see {@link DocumentIdConverterInterface}
     * @deprecated Please use {@link DataSessionInterface.createDocumentIdConverter} instead.
     */
    createDocumentIdConverter(): DocumentIdConverterInterface;
    /**
     * Instantiates a converter to translate `part instances ids` to their corresponding `geometric instance ids`.
     *
     * This is the opposite of the {@link GeometricInstanceConverterInterface}.
     *
     * **DO NOT** create {@link PartInstanceConverterInterface} before the DMU is loaded,
     * such {@link PartInstanceConverterInterface} will be silently discarded.
     *
     * @return A new {@link PartInstanceConverterInterface}.
     * @see {@link PartInstanceConverterInterface}
     * @deprecated Please use {@link DataSessionInterface.createPartInstanceConverter} instead.
     */
    createPartInstanceConverter(): PartInstanceConverterInterface;
    /**
     * Instantiates a converter to translate `geometric instance ids` to their corresponding `part instance ids`.
     *
     * This is the opposite of the {@link PartInstanceConverterInterface}.
     *
     * **DO NOT** create {@link GeometricInstanceConverterInterface} before the DMU is loaded,
     * such {@link GeometricInstanceConverterInterface} will be silently discarded.
     *
     * @return A new {@link GeometricInstanceConverterInterface}.
     * @see {@link GeometricInstanceConverterInterface}
     * @deprecated Please use {@link DataSessionInterface.createGeometricInstanceConverter} instead.
     */
    createGeometricInstanceConverter(): GeometricInstanceConverterInterface;
    /**
     * Instantiates a <a href="../index.html#search">search procedure</a>.
     *
     * A {@link SearchInterface} is used to trigger search query(ies) in the 3djuump infinite metadata
     * documents, and returns matching metadata documents. The {@link SearchInterface} relies on the
     * <a href="../index.html#3djuump-infinite-literal-and-search-query-language">
     * 3djuump infinite literal and search query language</a>.
     *
     * **DO NOT** create {@link SearchInterface} before the DMU is loaded,
     * such {@link SearchInterface} will be silently discarded.
     *
     * @return A new {@link SearchInterface}.
     * @see {@link SearchInterface}
     * @deprecated Please use {@link DataSessionInterface.createSearch} instead.
     */
    createSearch(): SearchInterface;
    /**
     * Instantiates a data retrieval procedure to retrieve the metadata documents
     * associated to a list of `part instance ids` and their ancestors.
     *
     * Such a list of metadata documents can be merged and customized to
     * display an "idcard" of a `part instance` and its ancestors.
     *
     * **DO NOT** create {@link IdCardGetterInterface} before the DMU is loaded,
     * such {@link IdCardGetterInterface} will be silently discarded.
     *
     * @return A new {@link IdCardGetterInterface}.
     * @see {@link IdCardGetterInterface}
     * @deprecated Please use {@link DataSessionInterface.createIdCardGetter} instead.
     */
    createIdCardGetter(): IdCardGetterInterface;
    /**
     * Instantiates a data retrieval procedure to retrieve the metadata documents
     * of the children of a list of `part instance ids`.
     *
     * Such a list of metadata documents can be merged and customized to
     * display an metadata information of the children of a `part instance`.
     *
     * **DO NOT** create {@link ChildrenIdCardGetterInterface} before the DMU is loaded,
     * such {@link ChildrenIdCardGetterInterface} will be silently discarded.
     *
     * @return A new {@link ChildrenIdCardGetterInterface}.
     * @see {@link ChildrenIdCardGetterInterface}
     * @deprecated Please use {@link DataSessionInterface.createChildrenIdCardGetter} instead.
     */
    createChildrenIdCardGetter(): ChildrenIdCardGetterInterface;
    /**
     * Instantiates a data retrieval procedure to retrieve the content of an attached document.
     *
     * Attached document are documents (images, html pages, links, pdf documents, etc ...)
     * that may be linked to `part instances`. The list of available documents for a `part instance`
     * are retrieved by an {@link IdCardGetterInterface}, then individual documents may be retrieved by
     * a {@link DocumentContentGetterInterface}.
     *
     * **DO NOT** create {@link DocumentContentGetterInterface} before the DMU is loaded,
     * such {@link DocumentContentGetterInterface} will be silently discarded.
     *
     * @return A new {@link DocumentContentGetterInterface}.
     * @see {@link DocumentContentGetterInterface}
     * @deprecated Please use {@link DataSessionInterface.createDocumentContentGetter} instead.
     */
    createDocumentContentGetter(): DocumentContentGetterInterface;
    /**
     * Instantiates an annotation retrieval procedure to retrieve the content of a view of annotations.
     *
     * Please see <a href="../index.html#annotations">Annotations</a> for more explanations about annotations.
     *
     * Annotation views may be linked to `part instances`. The list of annotation views for a `part instance`
     * are retrieved by an {@link IdCardGetterInterface}, then individual views may be retrieved by
     * a {@link AnnotationGetterInterface}.
     *
     * **DO NOT** create {@link AnnotationGetterInterface} before the DMU is loaded,
     * such {@link AnnotationGetterInterface} will be silently discarded.
     *
     * @return A new {@link AnnotationGetterInterface}.
     * @see {@link AnnotationGetterInterface}
     * @deprecated Please use {@link DataSessionInterface.createAnnotationGetter} instead.
     */
    createAnnotationGetter(): AnnotationGetterInterface;
    /**
     * Creates an {@link AttachedDocumentInfoInterface} to be used in a {@link DocumentContentGetterInterface}.
     *
     * If the user knows the `attached document id` and the mime type of an attached document, then a
     * {@link DocumentContentGetterInterface} can be used to retrieve the corresponding attached document.
     *
     * @return A new {@link AttachedDocumentInfoInterface}.
     * @see {@link DocumentContentGetterInterface}
     * @see {@link AttachedDocumentInfoInterface}
     * @deprecated Please use {@link DataSessionInterface.createAttachedDocumentInfo} instead.
     */
    createAttachedDocumentInfo(): AttachedDocumentInfoInterface;
    /**
     * Triggers the calculation of all modified {@link FilterSolverInterface | FilterSolverInterfaces},
     * {@link FilterItemInterface | FilterItemInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces}
     * {@link ConfContextInterface | ConfContextInterfaces}.
     *
     * All the modified items are updated on the server according to their
     * inter-dependencies.
     *
     * The relevant interfaces will asynchronously trigger a "ready" signal when their calculation is over.
     *
     * @deprecated Please use {@link DataSessionInterface.update} instead.
     */
    update(): void;
    /**
     * Cancels the calculation of all running {@link FilterSolverInterface | FilterSolverInterfaces},
     * {@link FilterItemInterface | FilterItemInterfaces}, {@link VisibilityContextInterface | VisibilityContextInterfaces}
     * {@link ConfContextInterface | ConfContextInterfaces}.
     *
     * The relevant interfaces will asynchronously trigger a "cancel" and "ready" signal when the
     * canceling is done.
     *
     * @deprecated Please use {@link DataSessionInterface.cancelUpdate} instead.
     */
    cancelUpdate(): void;
    /**
     * Computes the axis aligned bounding box of the given `geometric instances`.
     *
     * This consists in the union of all the {@link AABB} of the `geometric instances` expressed by their
     * `geometric instance ids`.
     *
     * Call this function once the DMU has been loaded ({@link DataSessionInterfaceSignal.DMULoadingSuccess})
     * with a previously created {@link AABB}.
     *
     * Returns true if at least one `geometric instance id` inside pGeometricInstanceIds is valid. Invalid `geometric instance ids`
     * are silently discarded.
     *
     * @param pGeometricInstanceIds <div class="const">in</div> the list of <code>geometric instance ids</code> to query.
     * @param pAABBOut <div class="const">out</div> the union of all the {@link AABB} o the <code>geometric instance</code>.
     * @return `true` if at least  one `geometric instance id` inside pGeometricInstanceIds is valid and pAABBOut is therefore updated.
     * @deprecated Please use {@link DataSessionInterface.getUnitedAABB} instead.
     */
    getUnitedAABB(pGeometricInstanceIds: Uint32Array | number[], pAABBOut: AABB): boolean;
    /**
     * Gets the conversion ratio between two units.
     *
     * @param pFrom <div class="const">in</div> The reference {@link Unit} where the conversion starts from.
     * @param pTo <div class="const">in</div> The {@link Unit} to convert to.
     * @return A floating number corresponding to the {@link Unit} conversion ratio.
     * @deprecated Please use {@link DataSessionInterface.convertUnitFactor} instead.
     */
    convertUnitFactor(pFrom: Unit, pTo: Unit): number;
    /**
     * Enables/Disables power saving mode.
     *
     * Power saving mode is used to save CPU usage when the browser tab is hidden by
     * introducing some frame throttling. The {@link FilterSolverInterface} calculations
     * will be slower if the tab is hidden.
     *
     * Power saving is disabled (false) by default.
     *
     * @param pIsPowerSavingEnabled <div class="const">in</div> Tells if the power saving mode should be enabled.
     * @deprecated Please use {@link InfiniteApiControllerInterface.setPowerSavingEnabled} instead.
     */
    setPowerSavingEnabled(pIsPowerSavingEnabled: boolean): void;
    /**
     * Tells if power saving mode is enabled.
     *
     * Power saving mode is used to save CPU usage when the browser tab is hidden by
     * introducing some frame throttling.
     *
     * @return `true` if the power saving mode is enabled.
     * @deprecated Please use {@link InfiniteApiControllerInterface.isPowerSavingEnabled} instead.
     */
    isPowerSavingEnabled(): boolean;
    /**
     * Adds a callback triggered each time the internal clock ticks.
     *
     * The internal clock is set at a fixed pace, and the callback is called at the end of
     * the tick.
     *
     * The callback may be called only if the tick triggered a modification in the rendering (in case an
     * {@link InfiniteEngineInterface} is linked to this instance).
     *
     * Returns a string id to use in {@link removeCallback}.
     *
     * @param pCallback <div class="const">in</div> The callback to call at the end of a clock tick.
     * @param pFrameHasNotChanged <div class="const">in</div> If `true`, the callback will be called only if the tick induced
     * a modification in the rendering.
     * @return The id of the callback to be used in {@link removeCallback}.
     * @deprecated Please use {@link InfiniteEngineInterfaceSignal.DisplayDone} and {@link InfiniteApiControllerInterfaceSignal.Tick} events instead.
     */
    addCallback(pCallback: Function, pFrameHasNotChanged: boolean): string;
    /**
     * Removes a callback that was triggered each time the internal clock ticked.
     *
     * The id obtained from {@link addCallback} should be used.
     *
     * @param pCallbackId <div class="const">in</div> The id obtained from {@link addCallback}.
     * @return `true` if the callback was registered and has been removed.
     * @deprecated Please use {@link InfiniteApiControllerInterface.removeEventListener} and {@link InfiniteEngineInterface.removeEventListener} instead.
     */
    removeCallback(pCallbackId: string): boolean;
    /**
     * Indicates if a callback exists in the list of callbacks triggered each time the internal clock ticks.
     *
     * @param pCallbackId <div class="const">in</div> the id obtained from {@link addCallback}.
     * @return `true` if the callback exists in the list of callbacks.
     * @deprecated Please use {@link InfiniteApiControllerInterface.hasEventListener} and {@link InfiniteEngineInterface.hasEventListener} instead.
     */
    hasCallback(pCallbackId: string): boolean;
    /**
     * Does nothing.
     *
     * @deprecated since 3.2.2, this function does nothing.
     * @internal
     */
    clearMetadataManager(): void;
}
export { MetadataManagerInterface };
/**
 * The {@link PartInstanceConverterInterfaceSignal} lists all the signals that may be sent by the {@link PartInstanceConverterInterface}.
 *
 * These signals are emitted by the {@link PartInstanceConverterInterface} when some data is available.
 *
 * * {@link PartInstanceConverterReady} : the {@link PartInstanceConverterInterface} is ready, i.e. when it has
 * finished getting the `geometric instance ids` that are expressed by the given `part instance ids`, {@link PartInstanceConverterInterface.getGeometricInstanceIds} can be called.
 * * {@link PartInstanceConverterCancelled} : the retrieving of the content by the {@link PartInstanceConverterInterface} has been cancelled.
 *
 * ```typescript
 * [[include:PartInstanceConverterInterface_example.ts]]
 * ```
 * <br>
 * Please refer to {@link PartInstanceConverterInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link PartInstanceConverterInterface}
 * @category Events
 */
declare enum PartInstanceConverterInterfaceSignal {
    /**
     * Signal sent by the {@link PartInstanceConverterInterface} when {@link PartInstanceConverterInterface.convert} is ready, i.e.
     * {@link PartInstanceConverterInterface.getGeometricInstanceIds} can be called.
     *
     * Such a signal does not tell if the call was successful or not but that the result is ready to be fetched.
     *
     * Warning, the PartInstanceConverterReady signal is sent later when {@link PartInstanceConverterInterface.cancel} is called and the {@link PartInstanceConverterInterface} is fetching data.
     *
     * The attachment is a string telling the request id of the call to {@link PartInstanceConverterInterface.convert} ({@link PartInstanceConverterInterface.getLastRequestId}).
     */
    PartInstanceConverterReady = "ready",
    /**
     * Signal sent by the {@link PartInstanceConverterInterface} when the {@link PartInstanceConverterInterface} has been cancelled.
     *
     * Such an event is fired if the {@link PartInstanceConverterInterface} was fetching data and {@link PartInstanceConverterInterface.cancel} is called.
     * The {@link PartInstanceConverterReady} signal will be triggered shortly after.
     *
     * No attachment for events of this type.
     */
    PartInstanceConverterCancelled = "cancelled"
}
export { PartInstanceConverterInterfaceSignal };
/**
 * The PartInstanceConverterInterface interface is used to get, from a list of `part instance ids`, the corresponding `geometric instance ids`.
 *
 * Contrary to the {@link GeometricInstanceConverterInterface}, this operation is not dependant on a filtering context.
 *
 * This conversion may be useful to change the display on some `part instances`, from a search query for example , since the {@link InfiniteEngineInterface} works
 * on `geometric instance ids`.
 *
 * The PartInstanceConverterInterface interfaces are created through the {@link DataSessionInterface.createPartInstanceConverter} method.
 *
 * The list of signals the PartInstanceConverterInterface may trigger is available in the {@link PartInstanceConverterInterfaceSignal} enumeration.
 *
 * The conversion mechanism is triggered through the {@link convert} method. The result is not available right away, but the
 * event {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterReady} is triggered when the result of the PartInstanceConverterInterfaceSignal is available.
 * The result is available through the {@link getGeometricInstanceIds} function.
 *
 * Warning : there may be cases when the {@link getGeometricInstanceIds} is not available such as when the PartInstanceConverterInterface is fetching data, i.e. when
 * {@link isRunning} returns true, or when the PartInstanceConverterInterface has been cancelled, i.e. when {@link isCancelled} returns true.
 *
 * A PartInstanceConverterInterface may be interrupted (cancelled) when the PartInstanceConverterInterface is fetching data and {@link cancel}
 * is called. In such cases, the {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterCancelled} signal is fired, and shortly after,
 * {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterReady}
 * signal is fired, but {@link getGeometricInstanceIds} will return undefined. Just call {@link convert} with another (or the same) `part instance ids` to trigger a new
 * retrieval.
 *
 * If you call multiple times {@link convert} before receiving the {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterReady},
 * only one {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterReady} will be sent with the content of the last call to {@link convert}.
 *
 * This interface performs the opposite operation of the {@link GeometricInstanceConverterInterface} interface.
 *
 * ```typescript
 * [[include:PartInstanceConverterInterface_example.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:PartInstanceConverterInterface_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Converters</div><br>
 * @see {@link PartInstanceConverterInterfaceSignal}
 * @category Converters
 */
interface PartInstanceConverterInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Triggers a request to "translate" the given `part instance ids` to the corresponding `geometric instance ids`.
     *
     * The server will try to find all `geometric instances` that are linked to the given `part instance ids`.
     * An event {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterReady} is fired when the translation is finished,
     * use {@link getLastError | getLastError()} to check if it was correctly performed.
     *
     * Returns `true` if the "conversion" is started. If not, just call {@link getLastError} to
     * get the reason why the procedure failed. All values of the input list should be between `0` and <code>2</code><sup><code>31</code></sup><code>-1</code>, if not,
     * returns false.
     *
     * @param pPartInstanceIds <div class="const">in</div> The <code>part instance ids</code> to perform the conversion with.
     * @return `true` if the conversion process has started, just wait for {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterReady}.
     */
    convert(pPartInstanceIds: Uint32Array | number[]): boolean;
    /**
     * Asynchronously triggers a request to "translate" the given `part instance ids` to the corresponding `geometric instance ids`.
     *
     * The server will try to find all `geometric instances` that are linked to the given `part instance ids`.
     *
     * Returns a promise.
     *
     * Please note that in case of multiple promises running at the same time on the same PartInstanceConverterInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getGeometricInstanceIds}
     * after awaiting will return the same value.
     *
     * @param pPartInstanceIds <div class="const">in</div> the <code>part instance ids</code> to perform the conversion with.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the requested Uint32Array.
     */
    asyncConvert(pPartInstanceIds: Uint32Array | number[]): Promise<AsyncUInt32ArrayResult>;
    /**
     * Each call to {@link convert} is assigned a request id.
     *
     * This call tels the id of the last call to {@link convert}.
     *
     * @return The id of the last call to {@link convert}.
     */
    getLastRequestId(): string;
    /**
     * Cancels the computation of the conversion process (if any).
     *
     * A {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterCancelled} signal is emitted if the
     * PartInstanceConverterInterface is retrieving data.
     *
     * @return `true` if the PartInstanceConverterInterface was running, else false.
     * @see {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterCancelled}
     */
    cancel(): boolean;
    /**
     * Tells if the PartInstanceConverterInterface is converting data.
     *
     * This is the case after calling {@link convert}.
     *
     * @return `true` if the PartInstanceConverterInterface is converting.
     */
    isRunning(): boolean;
    /**
     * Tells if the PartInstanceConverterInterface has been cancelled.
     *
     * This is generally the case after calling {@link cancel} when the PartInstanceConverterInterface
     * is retrieving data.
     *
     * @return `true` if the PartInstanceConverterInterface is cancelled.
     */
    isCancelled(): boolean;
    /**
     * Gets the last error returned by the {@link convert} call of the PartInstanceConverterInterface.
     *
     * @return The last error message (if any, or an empty string if no error occurred).
     */
    getLastError(): string;
    /**
     * Gets the result of the last {@link convert} call of the `part instance ids` conversion.
     *
     * An Uint32Array representing the `geometric instance ids` is returned if the PartInstanceConverterInterface
     * has finished computing.
     * Use {@link addEventListener} on the event {@link PartInstanceConverterInterfaceSignal.PartInstanceConverterReady} to
     * know when the PartInstanceConverterInterface is ready.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of <code>geometric instance ids</code>,
     * result of the conversion, or undefined if the converter had an error, or is still computing.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     */
    getGeometricInstanceIds(): Uint32Array | undefined;
}
export { PartInstanceConverterInterface };
/**
 * The PerformanceInterface interface is used to control the resources allocated to the renderer, be it cache, maximum number of
 * displayed triangles and memory consumption.
 *
 * A physical cache (based on <a href="https://developer.mozilla.org/en-US/docs/Web/API/IndexedDB_API">IndexedDB</a>)
 * is used to store build infos (used to initialize the renderer), and a second cache to store HD geometries.
 * Changing values for the performance object stores the new parameters in the localStorage of the browser, thus allowing the user to set
 * specific information for its hardware.
 *
 * The maximum amount of GPU memory consumption (VRAM) for geometries, the working memory (used when simultaneously downloading
 * geometries), the maximum amount of displayed triangles and the quality (in term of number of triangles) of the dynamic low definition
 * of the DMU can also be set.
 *
 * These values should reflect the "quality" of the hardware the API runs on. Please refer to the {@link InfiniteEngineInterface} for more
 * precisions.
 *
 * ```typescript
 * [[include:InfiniteEngineInterface_resource_controller.ts]]
 * ```
 * <br>
 * .
 *
 * [[include:constStyle.md]]<br><div class="const">3D Rendering</div><br>
 * @see {@link InfiniteEngineInterface}
 * @category 3D Rendering
 */
interface PerformanceInterface {
    /**
     * Gets the current number of displayed triangles.
     *
     * @return The number of displayed triangles.
     */
    getCurrentNbTrianglesDisplayed(): number;
    /**
     * Gets the current amount of memory used to render high definition geometries (in **bytes**).
     *
     * @return The current number of bytes used to render HD geometries.
     */
    getCurrentRenderingMemory(): number;
    /**
     * Gets the current amount of memory allocated to process the parallel downloading of
     * high definition geometries (in **bytes**).
     *
     * This parameter is highly optional, you should not need to change anything in most circumstances.
     *
     * @return The current number of working bytes.
     */
    getCurrentWorkingMemory(): number;
    /**
     * Gets the current amount of disk usage used to store build data in the cache (in **bytes**).
     *
     * Caching such data helps starting the application
     * much faster. This value is shared between all the builds that may have been opened by the user.
     *
     * @return The current number of bytes used to cache build information.
     */
    getCurrentBuildCacheSize(): number;
    /**
     * Gets the current amount of disk usage used to store high definition geometries in the cache (in **bytes**).
     *
     * Caching such data helps lowering
     * the bandwidth usage. This value is shared between all the projects that may have been opened by the user.
     *
     * @return The current number of bytes used to cache HD geometries.
     */
    getCurrentGeometryCacheSize(): number;
    /**
     * Sets if the shaders compilation check should be done (debug troubleshooting).
     *
     * This value is set to false by default.
     * The shader compilation check procedure slows the rendering pipe with the first frame and when the window is
     * resized. Turning this value to false increases performance. Such a value is not stored in the localStorage of the browser.
     *
     * @param pCheckCompilation <div class="const">in</div> If set to true, the shader compilation result will be checked.
     */
    setShaderCompilationChecked(pCheckCompilation: boolean): void;
    /**
     * Tells if the shaders compilation check is done (debug troubleshooting).
     *
     * This value is set to false by default.
     * The shader compilation check procedure slows the rendering pipe with the first frame and when the window is
     * resized. Such a value is not stored in the localStorage of the browser.
     *
     * @return `true` if the shader compilation result is checked.
     */
    isShaderCompilationChecked(): boolean;
    /**
     * Sets if the frame buffer objects status should be tested (debug troubleshooting).
     *
     * This value is set to false by default.
     * The frame buffer status check procedure slows the rendering pipe with the first frame and when the window is
     * resized. Such a value is not stored in the localStorage of the browser.
     *
     * @param pCheckStatus <div class="const">in</div> If set to true, the frame buffer objects status will be checked.
     */
    setFrameBufferStatusChecked(pCheckStatus: boolean): void;
    /**
     * Tells if the frame buffer objects status should be tested (debug troubleshooting).
     *
     * This value is set to false by default.
     * The frame buffer status check procedure slows the rendering pipe with the first frame and when the window is
     * resized. Such a value is not stored in the localStorage of the browser.
     *
     * @return `true` if the frame buffer objects status will be checked.
     */
    isFrameBufferStatusChecked(): boolean;
    /**
     * Gets the maximum number of displayed triangles.
     *
     * No more than {@link getMaxNbTrianglesDisplayed} triangles will be displayed.
     *
     * This value defaults to 10 000 000.
     *
     * @return The maximum number of displayed triangles.
     * @see {@link GetMaxTrianglesDefault}
     */
    getMaxNbTrianglesDisplayed(): number;
    /**
     * Sets the maximum number of displayed triangles.
     *
     * This value defaults to 10 000 000.
     *
     * @param pNbMaxTriangles <div class="const">in</div> The new maximum number of displayed triangles.
     * @return `true` if the value greater than {@link GetMaxTrianglesMin}.
     * @see {@link GetMaxTrianglesMin}
     */
    setMaxNbTrianglesDisplayed(pNbMaxTriangles: number): boolean;
    /**
     * Gets the maximum number of displayed triangles for the dynamic low definition model.
     *
     * This value defaults to 1 000 000.
     *
     * @return The maximum number of displayed triangles for the dynamic low definition model.
     * @see {@link GetMaxDynamicLowDefTrianglesDefault}
     */
    getMaxNbDynamicLowDefTrianglesDisplayed(): number;
    /**
     * Sets the maximum number of displayed triangles for the dynamic low definition model.
     *
     * This value defaults to 1 000 000.
     *
     * @param pNbMaxTriangles <div class="const">in</div> The new maximum number of triangles displayed for the dynamic low definition model.
     * @return `true` if the value is greater than {@link GetMaxDynamicLowDefTrianglesMin}.
     * @see {@link GetMaxDynamicLowDefTrianglesMin}
     */
    setMaxNbDynamicLowDefTrianglesDisplayed(pNbMaxTriangles: number): boolean;
    /**
     * Gets the maximum amount of memory used to render high definition geometries (in **bytes**).
     *
     * At most {@link getMaxRenderingMemory} bytes will be used by the graphic card to render HD geometries.
     * Such a budget should be a little less than the VRAM of the graphic card of the end-user.
     *
     * This value defaults to 2 GB.
     *
     * @return The maximum number of dedicated bytes to render HD geometries.
     * @see {@link GetMaxRenderingMemoryDefault}
     */
    getMaxRenderingMemory(): number;
    /**
     * Sets the maximum amount of memory used to render high definition geometries (in **bytes**).
     *
     * At most {@link getMaxRenderingMemory} bytes will be used by the graphic card to render HD geometries.
     * Such a budget should be a little less than the VRAM of the graphic card of the end-user.
     *
     * This value defaults to 2 GB.
     *
     * @param pMaxNbBytes <div class="const">in</div> The maximum number of dedicated bytes to render HD geometries.
     * @return `true` if the value is greater than {@link GetMaxRenderingMemoryMin}.
     * @see {@link GetMaxRenderingMemoryMin}
     */
    setMaxRenderingMemory(pMaxNbBytes: number): boolean;
    /**
     * Gets the maximum amount of memory allocated to process the parallel downloading of
     * high definition geometries (in **bytes**).
     *
     * This parameter is highly optional, you should not need to change anything in most circumstances.
     *
     * This value defaults to 256 MB.
     *
     * @return The maximum number of working bytes.
     * @see {@link GetMaxWorkingMemoryDefault}
     */
    getMaxWorkingMemory(): number;
    /**
     * Sets the maximum amount of memory allocated to process the parallel downloading of
     * high definition geometries (in **bytes**).
     *
     * This parameter is highly optional, you should not need to change anything in most circumstances.
     *
     * This value defaults to 256 MB.
     *
     * @param pMaxNbBytes <div class="const">in</div> The maximum number of dedicated bytes to render HD geometries.
     * @return `true` if the value is greater than {@link GetMaxWorkingMemoryMin}.
     * @see {@link GetMaxWorkingMemoryMin}
     */
    setMaxWorkingMemory(pMaxNbBytes: number): boolean;
    /**
     * Sets the maximum amount of disk usage used to store build data in the cache (in **bytes**).
     *
     * Caching such data helps starting the application
     * much faster. This value is shared between all the builds that may have been opened by the user.
     *
     * This value defaults to 4 GB.
     *
     * @param pMaxNbBytes <div class="const">in</div> The new maximum number of bytes used to cache build information.
     * @return `true` if the value is superior to {@link GetMaxBuildCacheSizeMin}.
     * @see {@link GetMaxBuildCacheSizeMin}
     */
    setMaxBuildCacheSize(pMaxNbBytes: number): boolean;
    /**
     * Gets the maximum amount of disk usage used to store build data in the cache (in **bytes**).
     *
     * Caching such data helps starting the application
     * much faster. This value is shared between all the builds that may have been opened by the user.
     *
     * This value defaults to 4 GB.
     *
     * @return The maximum number of bytes used to cache build information.
     */
    getMaxBuildCacheSize(): number;
    /**
     * Sets the maximum amount of disk usage used to store high definition geometries in the cache (in **bytes**).
     *
     * Caching such data helps lowering
     * the bandwidth usage. This value is shared between all the projects that may have been opened by the user.
     *
     * This value defaults to 4 GB.
     *
     * @param pMaxNbBytes <div class="const">in</div> The new maximum number of bytes used to cache HD geometries.
     * @return `true` if the value is superior to {@link GetMaxGeometryCacheSizeMin}.
     * @see {@link GetMaxGeometryCacheSizeMin}
     *
     */
    setMaxGeometryCacheSize(pMaxNbBytes: number): boolean;
    /**
     * Gets the maximum amount of disk usage used to store high definition geometries in the cache (in **bytes**).
     *
     * Caching such data helps lowering
     * the bandwidth usage. This value is shared between all the projects that may have been opened by the user.
     *
     * This value defaults to 4 GB.
     *
     * @return The maximum number of bytes used to cache HD geometries.
     * @see {@link GetMaxGeometryCacheSizeDefault}
     */
    getMaxGeometryCacheSize(): number;
    /**
     * Gets the minimal amount of disk usage used to store build data in the cache that can be set by
     * {@link setMaxBuildCacheSize} (in **bytes**).
     *
     * This value is a constant, arbitrary set by the library 128 MB.
     *
     * @return The minimum value that can be set by {@link setMaxBuildCacheSize}.
     * @see {@link setMaxBuildCacheSize}
     */
    GetMaxBuildCacheSizeMin(): number;
    /**
     * Gets the default (initialization) maximum amount of disk usage used to store build data in the cache (in **bytes**).
     *
     * This value is a constant, arbitrary set by the library 4 GB.
     *
     * @return The default maximum number of bytes used to store build data.
     */
    GetMaxBuildCacheSizeDefault(): number;
    /**
     * Gets the minimal amount of disk usage used to store high definition geometries in the cache that can be set by
     * {@link setMaxGeometryCacheSize} (in **bytes**).
     *
     * This value is a constant, arbitrary set by the library 128 MB.
     *
     * @return The minimum value that can be set by {@link setMaxGeometryCacheSize}.
     * @see {@link setMaxGeometryCacheSize}
     */
    GetMaxGeometryCacheSizeMin(): number;
    /**
     * Gets the default (initialization) maximum amount of disk usage used to store high definition geometries
     * in the cache (in **bytes**).
     *
     * This value is a constant, arbitrary set by the library 4 GB.
     *
     * @return The default maximum number of bytes used to store HD geometries.
     */
    GetMaxGeometryCacheSizeDefault(): number;
    /**
     * Gets the minimal amount of memory allocated to process the parallel downloading of
     * high definition geometries that can be set by
     * {@link setMaxWorkingMemory} (in **bytes**).
     *
     * This value is a constant, arbitrary set by the library 128 MB.
     *
     * @return The minimum value that can be set by {@link setMaxWorkingMemory}.
     * @see {@link setMaxWorkingMemory}
     */
    GetMaxWorkingMemoryMin(): number;
    /**
     * Gets the default (initialization) maximum amount of memory allocated to process the parallel downloading of
     * high definition geometries (in **bytes**).
     *
     * This value is a constant, arbitrary set by the library 256 MB.
     *
     * @return The default maximum number of working bytes.
     */
    GetMaxWorkingMemoryDefault(): number;
    /**
     * Gets the minimal amount of memory used to render high definition geometries that can be
     * set by {@link setMaxRenderingMemory}(in **bytes**).
     *
     * This value is a constant, arbitrary set by the library 512 MB.
     *
     * @return The minimum value that can be set by {@link setMaxRenderingMemory}.
     * @see {@link setMaxRenderingMemory}
     */
    GetMaxRenderingMemoryMin(): number;
    /**
     * Gets the default (initialization) maximum amount of memory used to render high definition geometries (in **bytes**).
     *
     * This value is a constant, arbitrary set by the library 2 GB.
     *
     * @return The default maximum number of bytes used to render HD geometries.
     */
    GetMaxRenderingMemoryDefault(): number;
    /**
     * Gets the minimum number of displayed triangles that can be set by
     * {@link setMaxNbTrianglesDisplayed}.
     *
     * This value is a constant, arbitrary set by the library 2 000 000.
     *
     * @return The minimum value that can be set by {@link setMaxNbTrianglesDisplayed}.
     * @see {@link setMaxNbTrianglesDisplayed}
     */
    GetMaxTrianglesMin(): number;
    /**
     * Gets the default (initialization) maximum number of displayed triangles.
     *
     * This value is a constant, arbitrary set by the library 10 000 000.
     *
     * @return The default maximum number of displayed triangles.
     */
    GetMaxTrianglesDefault(): number;
    /**
     * Gets the minimum number of displayed triangles for the dynamic low definition model that
     * can be set by {@link setMaxNbDynamicLowDefTrianglesDisplayed}.
     *
     * This value is a constant, arbitrary set by the library 1 000 000.
     *
     * @return The minimum value that can be set by {@link setMaxNbDynamicLowDefTrianglesDisplayed}.
     * @see {@link setMaxNbDynamicLowDefTrianglesDisplayed}
     */
    GetMaxDynamicLowDefTrianglesMin(): number;
    /**
     * Gets the default (initialization) maximum number of displayed triangles
     * for the dynamic low definition model.
     *
     * This value is a constant, arbitrary set by the library 1 000 000.
     *
     * @return The default maximum number of displayed triangles for the dynamic low definition model.
     */
    GetMaxDynamicLowDefTrianglesDefault(): number;
}
export { PerformanceInterface };
/**
 * The PrimitiveManagerBoxInterface is used to display 3D boxes on top of the DMU, with a rendering hint depending on the relative depth
 * between the 3D boxes and the geometries.
 *
 * The PrimitiveManagerBoxInterface is accessed through the {@link PrimitiveManagerInterface.getBoxManager} function (the {@link PrimitiveManagerInterface}
 * is accessed through {@link InfiniteEngineInterface.getPrimitiveManager}).
 *
 * A 3D box is an axis aligned box, consisting of 12 lines, and 6 faces.
 * Rendered 3D boxes are manipulated through their ids (and not a dedicated object), indeed
 * each box is assigned a strictly positive integer id that will represent it. 0 is an invalid box id.
 *
 * The colors of the 3D boxes are handled by the {@link PrimitiveManagerMaterialInterface},
 * accessed through {@link PrimitiveManagerInterface.getMaterialManager}. Materials are accessed by their ids, and modified by their ids. 3D primitives are
 * rendered colored with their `primitive material id`, and not directly from a color.<br>
 * **WARNING** : do not confuse the {@link PrimitiveManagerMaterialInterface} and the {@link MaterialManagerInterface}, these two interfaces are separated,
 * do not use {@link MaterialManagerInterface} ids to draw boxes !!!
 *
 * <img src="media://box.svg" alt="box display"/>
 * <br>
 * Creating a 3D box may be done with the following :
 *
 * ```typescript
 * [[include:PrimitiveManagerBoxInterface_create_box.ts]]
 * ```
 * <br>
 * You may create also lines, points, etc with {@link PrimitiveManagerLineInterface}, {@link PrimitiveManagerPointInterface}, etc.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Primitives</div><br>
 * @see {@link PrimitiveManagerMaterialInterface}
 * @see {@link PrimitiveManagerInterface}
 * @see {@link InfiniteEngineInterface}
 * @category 3D Primitives
 */
interface PrimitiveManagerBoxInterface {
    /**
     * Gets the actual number of rendered boxes.
     *
     * @return The number of drawn boxes.
     */
    getNbBoxes(): number;
    /**
     * Creates a new 3D box at the given position with the given size.
     *
     * A 3D box is an axis aligned box, consisting of 12 lines, and 6 faces.<br>
     * The width of the lines, the color of the lines, the color of the faces are set by the user.<br>
     * The box is assigned a strictly positive id that will represent this box from now on. 0 is an invalid box id. Please note that this id
     * will be reused if the box is removed and another created again.<br>
     * Before creating a box, materials to render this box must be created in the {@link PrimitiveManagerMaterialInterface}.
     *
     * @param pBoxCenter <div class="const">in</div> The 3D position of the center of the box.
     * @param pHalfExtent <div class="const">in</div> The half extent of the box.
     * @param pLineMaterialId <div class="const">in</div> The material that will be used to draw lines (<code>primitive material id</code>).
     * @param pFaceMaterialId <div class="const">in</div> The material that will be used to draw the faces (<code>primitive material id</code>).
     * @param pLineWidth <div class="const">in</div> The width of the lines in pixels.
     * @return The id of the created box.
     */
    createBox(pBoxCenter: Vector3, pHalfExtent: Vector3, pLineMaterialId: number, pFaceMaterialId: number, pLineWidth: number): number;
    /**
     * Removes the box with the given id.
     *
     * @param pBoxId <div class="const">in</div> The id of the box to remove.
     * @return `true` if the box exists and therefore is removed.
     */
    removeBox(pBoxId: number): boolean;
    /**
     * Tells if the given box is visible.
     *
     * Returns `false` if the box does not exists or is {@link setBoxVisible} with `false` has been called.
     *
     * @param pBoxId <div class="const">in</div> The id of the box to query.
     * @return `true` if the box exists and is visible.
     */
    isBoxVisible(pBoxId: number): boolean;
    /**
     * Gets the `primitive material id` used to render the lines of the given box.
     *
     * Returns 0 if a box with the given id does not exist.
     *
     * @param pBoxId <div class="const">in</div> The id of the box to query.
     * @return The <code>primitive material id</code> used to render the lines of the box.
     */
    getBoxLineMaterial(pBoxId: number): number;
    /**
     * Gets the `primitive material id` used to render the faces of the given box.
     *
     * Returns 0 if a box with the given id does not exist.
     *
     * @param pBoxId <div class="const">in</div> The id of the box to query.
     * @return The <code>primitive material id</code> used to render the faces of the box.
     */
    getBoxFaceMaterial(pBoxId: number): number;
    /**
     * Gets the position of the center of the given box.
     *
     * If found, pBoxCenterOut is updated.
     * Returns `true` if the box is found and therefore pBoxCenterOut is updated.
     *
     * @param pBoxId <div class="const">in</div> the id of the box to query.
     * @param pBoxCenterOut <div class="const">out</div> the center of the box.
     * @return `true` if pBoxCenterOut is updated.
     */
    getBoxCenter(pBoxId: number, pBoxCenterOut: Vector3): boolean;
    /**
     * Gets the half extents of the given box.
     *
     * If found, pBoxExtentOut is updated.
     * Returns `true` if the box is found and therefore pBoxExtentOut is updated.
     *
     * @param pBoxId <div class="const">in</div> The id of the box to query.
     * @param pBoxExtentOut <div class="const">out</div> The half extent of the box.
     * @return `true` if pBoxExtentOut was updated.
     */
    getBoxHalfExtent(pBoxId: number, pBoxExtentOut: Vector3): boolean;
    /**
     * Gets the line width in pixels used to render the lines of the given box.
     *
     * Returns 0 if the box was not found.
     *
     * @param pBoxId <div class="const">in</div> The id of the box to query.
     * @return The line width in pixels used to render the lines of the box.
     */
    getBoxLineWidth(pBoxId: number): number;
    /**
     * Updates the rendering of the given box, by changing the colors of the lines and the faces.
     *
     * Returns true if the box was updated.
     *
     * @param pBoxId <div class="const">in</div> The id of the box to change.
     * @param pLineMaterialId <div class="const">in</div> The <code>primitive material id</code> to be used to draw the lines of the box.
     * @param pFaceMaterialId <div class="const">in</div> The <code>primitive material id</code> to be used to draw the faces of the box.
     * @return `true` if the box was found, if the materials exist and therefore the box rendering is updated.
     */
    setBoxMaterial(pBoxId: number, pLineMaterialId: number, pFaceMaterialId: number): boolean;
    /**
     * Updates the shape of the given box, by changing the center, the half extent and the line width.
     *
     * Returns `true` if the box was updated.
     *
     * @param pBoxId <div class="const">in</div> The id of the box to change.
     * @param pCenter <div class="const">in</div> The new center of the box.
     * @param pHalfExtent <div class="const">in</div> The new half extent of the box.
     * @param pLineWidth <div class="const">in</div> The new line width in pixels of the lines of the box.
     * @return `true` if the box was found and therefore the box rendering is updated.
     */
    setBoxGeometry(pBoxId: number, pCenter: Vector3, pHalfExtent: Vector3, pLineWidth: number): boolean;
    /**
     * Updates the rendering of the given box, by changing its visibility.
     *
     * Returns `true` if the box was updated.
     *
     * @param pBoxId <div class="const">in</div> The id of the box to change.
     * @param pVisible <div class="const">in</div> The new visibility state of the box.
     * @return `true` if the box exists.
     */
    setBoxVisible(pBoxId: number, pVisible: boolean): boolean;
}
export { PrimitiveManagerBoxInterface };
/**
 * The PrimitiveManagerInterface interface is the central point used to render primitive elements, such as points, boxes, lines and a
 * rectangle.
 *
 * This interface holds the primitive elements interfaces and the materials that are used by these primitive elements.
 *
 * 3D Points, 3D lines, 3D boxes and a single 2D rectangle can be rendered on top of the DMU, with a rendering hint depending on the relative depth between the 3D primitives and the geometries.
 * This interface accessed through the {@link InfiniteEngineInterface.getPrimitiveManager} function. The colors of the 3D primitive are handled by the {@link PrimitiveManagerMaterialInterface},
 * accessed through {@link getMaterialManager}. Materials are accessed by their ids, and modified by their ids. 3D primitives are
 * rendered colored with their material id, and not directly from a color.
 *
 * ## 3D Points
 *
 * The rendering can be enriched with 3D points rendered on top of the geometries, with a rendering hint telling if the point is over or under a geometry.
 *
 * <img src="media://point.svg" alt="point display"/>
 * <br>
 * Creating a point may be done with the following :
 *
 * ```typescript
 * [[include:PrimitiveManagerPointInterface_create_point.ts]]
 * ```
 * <br>
 *
 * ## 3D Lines
 *
 * The rendering can be enriched with 3D lines rendered on top of the geometries.
 *
 * <img src="media://line.svg" alt="line display"/><br>
 * <br>
 * Creating a line may be done with the following :
 *
 * ```typescript
 * [[include:PrimitiveManagerLineInterface_create_line.ts]]
 * ```
 * <br>
 *
 * ## 3D Boxes
 *
 * The rendering can be enriched with 3D boxes rendered on top of the geometries, with a rendering hint telling if the edge/face is over or under a geometry.
 *
 * <img src="media://box.svg" alt="box display"/>
 * <br>
 * Creating a box may be done with the following :
 *
 * ```typescript
 * [[include:PrimitiveManagerBoxInterface_create_box.ts]]
 * ```
 * <br>
 *
 * ## 2D rectangle (rubber band)
 *
 * The rendering can be enriched with a single 2D rectangle rendered on top of the geometries.
 * The color cannot be changed.
 *
 * <img src="media://rectangle.svg" alt="rubber band display"/>
 * <br>
 * Rendering a 2D rectangle may be done with the following :
 *
 * ```typescript
 * [[include:PrimitiveManagerRubberBandInterface_create_rectangle.ts]]
 * ```
 * <br>
 *
 * [[include:constStyle.md]]<br><div class="const">3D Primitives</div><br>
 * @see {@link PrimitiveManagerMaterialInterface}
 * @see {@link PrimitiveManagerPointInterface}
 * @see {@link PrimitiveManagerLineInterface}
 * @see {@link PrimitiveManagerBoxInterface}
 * @see {@link PrimitiveManagerRubberBandInterface}
 * @category 3D Primitives
 */
interface PrimitiveManagerInterface {
    /**
     * Gets the {@link PrimitiveManagerMaterialInterface} in charge of the primitive colors.
     *
     * @return The {@link PrimitiveManagerMaterialInterface}.
     * @see {@link PrimitiveManagerMaterialInterface}
     */
    getMaterialManager(): PrimitiveManagerMaterialInterface;
    /**
     * Gets the {@link PrimitiveManagerPointInterface} in charge of drawing 3D points.
     *
     * @return The {@link PrimitiveManagerPointInterface}.
     * @see {@link PrimitiveManagerPointInterface}
     */
    getPointManager(): PrimitiveManagerPointInterface;
    /**
     * Gets the {@link PrimitiveManagerLineInterface} in charge of drawing 3D lines.
     *
     * @return The {@link PrimitiveManagerLineInterface}.
     * @see {@link PrimitiveManagerLineInterface}
     */
    getLineManager(): PrimitiveManagerLineInterface;
    /**
     * Gets the {@link PrimitiveManagerBoxInterface} in charge of drawing 3D boxes.
     *
     * @return The {@link PrimitiveManagerBoxInterface}.
     * @see {@link PrimitiveManagerBoxInterface}
     */
    getBoxManager(): PrimitiveManagerBoxInterface;
    /**
     * Gets the {@link PrimitiveManagerRubberBandInterface} in charge of drawing a single 2D rectangle.
     *
     * @return The {@link PrimitiveManagerRubberBandInterface}.
     * @see {@link PrimitiveManagerRubberBandInterface}
     */
    getRubberBandManager(): PrimitiveManagerRubberBandInterface;
}
export { PrimitiveManagerInterface };
/**
 * The PrimitiveManagerLineInterface is used to display 3D lines on top of the DMU.
 *
 * The PrimitiveManagerLineInterface is accessed through the {@link PrimitiveManagerInterface.getLineManager} function (the {@link PrimitiveManagerInterface}
 * is accessed through {@link InfiniteEngineInterface.getPrimitiveManager}).
 *
 * A 3D line is defined as a segment, thus with two 3D points.
 * Rendered 3D lines are manipulated through their ids (and not a dedicated object), indeed
 * each line is assigned a strictly positive integer id that will represent it. 0 is an invalid line id.
 *
 * The color of the 3D line is handled by the {@link PrimitiveManagerMaterialInterface},
 * accessed through {@link PrimitiveManagerInterface.getMaterialManager}. Materials are accessed by their ids, and modified by their ids. 3D primitives are
 * rendered colored with their `primitive material id`, and not directly from a color.<br>
 * **WARNING** : do not confuse the {@link PrimitiveManagerMaterialInterface} and the {@link MaterialManagerInterface}, these two interfaces are separated,
 * do not use {@link MaterialManagerInterface} ids to draw lines !!!
 *
 * <img src="media://line.svg" alt="line display"/><br>
 * <br>
 * Creating a 3D line may be done with the following :
 *
 * ```typescript
 * [[include:PrimitiveManagerLineInterface_create_line.ts]]
 * ```
 * <br>
 * You may create also boxes, points, etc with {@link PrimitiveManagerBoxInterface}, {@link PrimitiveManagerPointInterface}, etc.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Primitives</div><br>
 * @see {@link PrimitiveManagerMaterialInterface}
 * @see {@link PrimitiveManagerInterface}
 * @see {@link InfiniteEngineInterface}
 * @category 3D Primitives
 */
interface PrimitiveManagerLineInterface {
    /**
     * Gets the actual number of rendered lines.
     *
     * @return The number of lines drawn.
     */
    getNbLines(): number;
    /**
     * Creates a new line given two points positions (extents of the segment), A and B, with a given width in pixels.
     *
     * The width in pixels and the color of the line are set by the user.<br>
     * The line is assigned a strictly positive id that will represent this line from now on. 0 is an invalid line id. Please note that this id
     * will be reused if the line is removed and another created again.<br>
     * Before creating a line, materials to render this line must be created in the {@link PrimitiveManagerMaterialInterface}.
     *
     * @param pPointA <div class="const">in</div> The 3D position of the A point of the line.
     * @param pPointB <div class="const">in</div> The 3D position of the B point of the line.
     * @param pLineMaterialId <div class="const">in</div> The material that will be used to draw the line (<code>primitive material id</code>).
     * @param pLineWidth <div class="const">in</div> The width of the line in pixels.
     * @return The id of the created line.
     */
    createLine(pPointA: Vector3, pPointB: Vector3, pLineMaterialId: number, pLineWidth: number): number;
    /**
     * Removes the line with the given id.
     *
     * @param pLineId <div class="const">in</div> The id of the line to remove.
     * @return `true` if the line exists and therefore is removed.
     */
    removeLine(pLineId: number): boolean;
    /**
     * Gets the A point (origin) of the given line.
     *
     * If found, pPointAOut is updated.
     * Returns `true` if the line is found and therefore pPointAOut is updated.
     *
     * @param pLineId <div class="const">in</div> The id of the line to query.
     * @param pPointAOut <div class="const">out</div> The A point of the line.
     * @return `true` if pPointAOut was updated.
     */
    getLinePointA(pLineId: number, pPointAOut: Vector3): boolean;
    /**
     * Gets the B point (other end) of the given line.
     *
     * If found, pPointBOut is updated.
     * Returns `true` if the line is found and therefore pPointBOut is updated.
     *
     * @param pLineId <div class="const">in</div> The id of the line to query.
     * @param pPointBOut <div class="const">out</div> The B point of the line.
     * @return `true` if pPointBOut was updated.
     */
    getLinePointB(pLineId: number, pPointBOut: Vector3): boolean;
    /**
     * Gets the `primitive material id` used to render the given line.
     *
     * Returns 0 if a line with the given id does not exist.
     *
     * @param pLineId <div class="const">in</div> The id of the line to query.
     * @return The <code>primitive material id</code> used to render the line.
     */
    getLineMaterial(pLineId: number): number;
    /**
     * Gets the line width in pixels used to render the given line.
     *
     * Returns 0 if a line with the given id does not exist.
     *
     * @param pLineId <div class="const">in</div> The id of the line to query.
     * @return The line width in pixels used to render the line.
     */
    getLineWidth(pLineId: number): number;
    /**
     * Tells if the given line is visible.
     *
     * Returns `false` if the line does not exists or is {@link setLineVisible} with false has been called.
     *
     * @param pLineId <div class="const">in</div> The id of the line to query.
     * @return `true` if the line exists and is visible.
     */
    isLineVisible(pLineId: number): boolean;
    /**
     * Updates the rendering of the given line, by changing the color of the line.
     *
     * Returns `true` if the line was updated.
     *
     * @param pLineId <div class="const">in</div> The id of the line to change.
     * @param pLineMaterialId <div class="const">in</div> The <code>primitive material id</code> to be used to draw the line.
     * @return `true` if the line was found, if the materials exist and therefore the line rendering is updated.
     */
    setLineMaterial(pLineId: number, pLineMaterialId: number): boolean;
    /**
     * Updates the shape of the given line, by changing the extents and the line width.
     *
     * Returns `true` if the line was updated.
     *
     * @param pLineId <div class="const">in</div> The id of the line to change.
     * @param pPointA <div class="const">in</div> The new 3D position of the A point (origin).
     * @param pPointB <div class="const">in</div> The new 3D position of the B point (other end).
     * @param pLineWidth <div class="const">in</div> The new line width in pixels.
     * @return `true` if the line was found and therefore the line rendering is updated.
     */
    setLineGeometry(pLineId: number, pPointA: Vector3, pPointB: Vector3, pLineWidth: number): boolean;
    /**
     * Updates the rendering of the given line, by changing its visibility.
     *
     * Returns `true` if the line was updated.
     *
     * @param pLineId <div class="const">in</div> The id of the line to change.
     * @param pVisible <div class="const">in</div> The new visibility state of the line.
     * @return `true` if the line exists.
     */
    setLineVisible(pLineId: number, pVisible: boolean): boolean;
}
export { PrimitiveManagerLineInterface };
/**
 * The PrimitiveManagerMaterialInterface is used to change the colors of 3D primitives : Points, lines and boxes.
 *
 * The PrimitiveManagerMaterialInterface is accessed through the {@link PrimitiveManagerInterface.getMaterialManager} function (the {@link PrimitiveManagerInterface}
 * is accessed through {@link InfiniteEngineInterface.getPrimitiveManager}).
 *
 * The color is expressed as a Vector4, RGBA (Red, Green, Blue, Alpha), Red = pRGBAColor.x, etc...<br>
 * Color values are expressed from 0 to 1. An alpha value of 1 means an opaque color, an alpha value of 0 means a transparent color.
 *
 * Materials are expressed and manipulated as strictly positive ids : a `primitive material id`. A `primitive material id` of 0 is invalid.
 * There is a limit on the number of materials that may be created. The {@link getMaxAvailableMaterials} tells the number of primitive materials that may be created.
 *
 * Materials can be modified and removed. However materials used by 3D primitives are protected from removal until no primitive uses them.
 *
 * **WARNING** : do not confuse the {@link PrimitiveManagerMaterialInterface} and the {@link MaterialManagerInterface}, these two interfaces are separated,
 * do not use {@link MaterialManagerInterface} ids to draw 3D primitives !!!
 *
 * <img src="media://point.svg" alt="point display"/>
 * <br>
 * <img src="media://line.svg" alt="line display"/>
 * <br>
 * <img src="media://box.svg" alt="box display"/>
 * <br>
 * Changing material may be done with the following :
 *
 * ```typescript
 * [[include:PrimitiveManagerMaterialInterface_changeMaterial.ts]]
 * ```
 * <br>
 * You may create boxes, points, etc with {@link PrimitiveManagerBoxInterface}, {@link PrimitiveManagerPointInterface}, etc.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Primitives</div><br>
 * @see {@link PrimitiveManagerInterface}
 * @see {@link InfiniteEngineInterface}
 * @see {@link PrimitiveManagerPointInterface}
 * @see {@link PrimitiveManagerLineInterface}
 * @see {@link PrimitiveManagerBoxInterface}
 * @see {@link PrimitiveManagerRubberBandInterface}
 * @category 3D Primitives
 */
interface PrimitiveManagerMaterialInterface {
    /**
     * Creates a new material to be used by 3D primitive items (points, lines, boxes).
     *
     * The color is expressed as a Vector4, RGBA (Red, Green, Blue, Alpha), Red = pRGBAColor.x, etc...<br>
     * Color values are expressed from 0 to 1. An alpha value of 1 means an opaque color, an alpha value of 0 means a transparent color.
     *
     * @param pRGBAColor <div class="const">in</div> The color to set for the drawing of the primitives items.
     * @return The <code>primitive material id</code> of the newly created material.
     * @see {@link getMaxAvailableMaterials}
     */
    createMaterial(pRGBAColor: Vector4): number;
    /**
     * Removes the given material.
     *
     * The `primitive material id` pMaterialId should be valid and should not be currently used by any other primitive.<br>
     * If pMaterialId is in use, then the function returns `false` and the material is not removed.
     *
     * @param pMaterialId <div class="const">in</div> The id of the material to remove.
     * @return `true` if the material has been removed.
     */
    removeMaterial(pMaterialId: number): boolean;
    /**
     * Gets the color of an existing material.
     *
     * If found, pRGBAColorOut is updated.
     * Returns `true` if the primitive material is found and therefore pRGBAColorOut is updated.
     *
     * @param pMaterialID <div class="const">in</div> The <code>primitive material id</code> of the material to query.
     * @param pRGBAColorOut <div class="const">out</div> The color of the given material.
     * @return `true` if pRGBAColorOut was updated.
     */
    getMaterialColor(pMaterialID: number, pRGBAColorOut: Vector4): boolean;
    /**
     * Tells the maximum number of materials that can be created.
     *
     * This number is not updated when new materials are created.
     * This value is constant : 65535.
     *
     * @return The number of materials that may be created.
     */
    getMaxAvailableMaterials(): number;
    /**
     * Tells the number of materials in use.
     *
     * This number is updated when new materials are created or removed.
     *
     * @return The number of created materials.
     */
    getNbMaterials(): number;
    /**
     * Get a copy of the list of `primitive material ids` in use.
     *
     * @return A copy of the <code>primitive material ids</code> in use.
     */
    getMaterialIds(): Array<number>;
    /**
     * Updates the color of an existing material.
     *
     * Returns `true` if a primitive material with the given `primitive material id` exists.
     *
     * @param pMaterialID <div class="const">in</div> The <code>primitive material id</code> of the material to update.
     * @param pRGBAColor <div class="const">in</div> The new color to set for this material.
     * @return `true` if the material exists and is therefore updated.
     */
    setMaterialColor(pMaterialID: number, pRGBAColor: Vector4): boolean;
}
export { PrimitiveManagerMaterialInterface };
/**
 * The PrimitiveManagerPointInterface is used to display 3D points on top of the DMU, with a rendering hint depending on the relative depth
 * between the 3D points and the geometries.
 *
 * The PrimitiveManagerPointInterface is accessed through the {@link PrimitiveManagerInterface.getPointManager} function (the {@link PrimitiveManagerInterface}
 * is accessed through {@link InfiniteEngineInterface.getPrimitiveManager}).
 *
 * A 3D point is defined as a 3D position, an full size in pixels, an inner size in pixels.
 * Rendered 3D points are manipulated through their ids (and not a dedicated object), indeed
 * each point is assigned a strictly positive integer id that will represent it. 0 is an invalid point id.
 *
 * The color of the 3D point is handled by the {@link PrimitiveManagerMaterialInterface},
 * accessed through {@link PrimitiveManagerInterface.getMaterialManager}. Materials are accessed by their ids, and modified by their ids. 3D primitives are
 * rendered colored with their `primitive material id`, and not directly from a color.<br>
 * **WARNING** : do not confuse the {@link PrimitiveManagerMaterialInterface} and the {@link MaterialManagerInterface}, these two interfaces are separated,
 * do not use {@link MaterialManagerInterface} ids to draw points !!!
 *
 * <img src="media://point.svg" alt="point display"/><br>
 * <br>
 * Creating a 3D point may be done with the following :
 *
 * ```typescript
 * [[include:PrimitiveManagerPointInterface_create_point.ts]]
 * ```
 * <br>
 * You may create also boxes, lines, etc with {@link PrimitiveManagerBoxInterface}, {@link PrimitiveManagerLineInterface}, etc.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Primitives</div><br>
 * @see {@link PrimitiveManagerMaterialInterface}
 * @see {@link PrimitiveManagerInterface}
 * @see {@link InfiniteEngineInterface}
 * @category 3D Primitives
 */
interface PrimitiveManagerPointInterface {
    /**
     * Gets the actual number of rendered points.
     *
     * @return The number of points drawn.
     */
    getNbPoints(): number;
    /**
     * Creates a new point at the given position with full and inner sizes.
     *
     * The sizes in pixels and the colors of the point are set by the user.<br>
     * The point is assigned a positive id that will represent this point from now on. 0 is an invalid point id. Please note that this id
     * will be reused if the point is removed and another created again. <br>
     * Before creating a point, materials to render this point must be created in the primitive manager.
     *
     * A newly created point is visible.
     *
     * @param pPointPosition <div class="const">in</div> The 3D position of the point.
     * @param pPointSize <div class="const">in</div> Size in pixels of the point (must be superior than 0.0).
     * @param pInnerSize <div class="const">in</div> Size in pixels of the inner area (must be superior or equal to 0.0, and inferior to pPointSize).
     * @param pInnerMaterialId <div class="const">in</div> The material used to draw the inner circle.
     * @param pOuterMaterialId <div class="const">in</div> The material used to draw the outer circle.
     * @return The id of the created point.
     */
    createPoint(pPointPosition: Vector3, pPointSize: number, pInnerSize: number, pInnerMaterialId: number, pOuterMaterialId: number): number;
    /**
     * Removes the point with id.
     *
     * @param pPointId <div class="const">in</div> The id of the point to remove.
     * @return `true` if the point exists ans therefore is removed.
     */
    removePoint(pPointId: number): boolean;
    /**
     * Gets the position of the given point.
     *
     * If found, pPositionOut is updated with the 3d position of the point.
     * Returns `true` if the point exists and pPositionOut is updated.
     *
     * @param pPointId <div class="const">in</div> The id of the point to query.
     * @param pPositionOut <div class="const">out</div> The position of the point.
     * @return `true` if pPosition was updated.
     */
    getPointPosition(pPointId: number, pPositionOut: Vector3): boolean;
    /**
     * Gets the `primitive material id` used to render the inner area of the point.
     *
     * Returns 0 if a point with the given id does not exist.
     *
     * @param pPointId <div class="const">in</div> The id of the point to query.
     * @return The <code>primitive material id</code> of the inner circle.
     */
    getPointInnerMaterial(pPointId: number): number;
    /**
     * Gets the `primitive material id` used to render the outer area of the point.
     *
     * Returns 0 if a point with the given id does not exist.
     *
     * @param pPointId <div class="const">in</div> The id of the point to query.
     * @return The <code>primitive material id</code> of the inner circle.
     */
    getPointOuterMaterial(pPointId: number): number;
    /**
     * Gets the size in pixels of the inner circle of the point.
     *
     * Returns 0 if a point with the given id does not exist.
     *
     * @param pPointId <div class="const">in</div> The id of the point to query.
     * @return The size in pixels of the inner circle.
     */
    getPointInnerSize(pPointId: number): number;
    /**
     * Tells if the given point is visible.
     *
     * Returns `false` if the point does not exists or is {@link setPointVisible} with `false` has been called.
     *
     * @param pPointId <div class="const">in</div> The id of the point to query.
     * @return `true` if the point exists and is visible.
     */
    isPointVisible(pPointId: number): boolean;
    /**
     * Gets the full size in pixels of the point.
     *
     * Returns 0 if a point with the given id does not exist.
     *
     * @param pPointId <div class="const">in</div> The id of the point to query.
     * @return The size in pixels of the point.
     */
    getPointSize(pPointId: number): number;
    /**
     * Sets the position of the given point.
     *
     * If found, pPointPositionOut is updated.
     * Returns `true` if the point is found and therefore pPointPositionOut is updated.
     *
     * @param pPointId <div class="const">in</div> The id of the point to query.
     * @param pPointPositionOut <div class="const">out</div> The new position of the point.
     * @return `true` if the position was updated.
     */
    setPointPosition(pPointId: number, pPointPositionOut: Vector3): boolean;
    /**
     * Updates the rendering of the given point, by changing its inner and outer colors.
     *
     * Returns `true` if the point was updated.
     *
     * @param pPointId <div class="const">in</div> The id of the point to change.
     * @param pInnerMaterialId <div class="const">in</div> The <code>primitive material id</code> of the inner circle color.
     * @param pOuterMaterialId <div class="const">in</div> The <code>primitive material id</code> of the outer circle color.
     * @return `true` if the point was found and therefore the point rendering is updated.
     */
    setPointMaterial(pPointId: number, pInnerMaterialId: number, pOuterMaterialId: number): boolean;
    /**
     * Updates the rendering of the given point, by changing the pixel size of the inner and outer circles.
     *
     * Returns `true` if the point was updated.
     *
     * @param pPointId <div class="const">in</div> The id of the point to change.
     * @param pPointSize <div class="const">in</div> The new size in pixels of the point.
     * @param pInnerSize <div class="const">in</div> The new size in pixels of the inner circle.
     * @return `true` if the point exists, pPointSize > 0 and pPointSize >= pInnerSize >= 0. If true, the point is updated.
     */
    setPointShape(pPointId: number, pPointSize: number, pInnerSize: number): boolean;
    /**
     * Updates the rendering of the given point, by changing its visibility.
     *
     * Returns `true` if the point was updated.
     *
     * @param pPointId <div class="const">in</div> The id of the point to change.
     * @param pVisible <div class="const">in</div> The new visibility state of the point.
     * @return `true` if the point exists.
     */
    setPointVisible(pPointId: number, pVisible: boolean): boolean;
}
export { PrimitiveManagerPointInterface };
/**
 * The PrimitiveManagerRubberBandInterface is used to display a unique orange rectangle on top of the DMU.
 *
 * It is mainly used to select a rectangular area.
 *
 * The PrimitiveManagerPointInterface is accessed through the {@link PrimitiveManagerInterface.getRubberBandManager} function (the {@link PrimitiveManagerInterface}
 * is accessed through {@link InfiniteEngineInterface.getPrimitiveManager}).
 *
 * The rectangle is defined with the {@link Rectangle} class (with two 3D points).
 *
 * The color of the rectangle of the PrimitiveManagerRubberBandInterface cannot be changed.
 *
 * <img src="media://rectangle.svg" alt="rectangle display"/><br>
 * <br>
 * Displaying a Rectangle may be done with the following :
 *
 * ```typescript
 * [[include:PrimitiveManagerRubberBandInterface_create_rectangle.ts]]
 * ```
 * <br>
 * You may create also boxes, lines, etc with {@link PrimitiveManagerBoxInterface}, {@link PrimitiveManagerLineInterface}, etc.
 *
 * [[include:constStyle.md]]<br><div class="const">3D Primitives</div><br>
 * @see {@link PrimitiveManagerMaterialInterface}
 * @see {@link PrimitiveManagerInterface}
 * @see {@link InfiniteEngineInterface}
 * @category 3D Primitives
 */
interface PrimitiveManagerRubberBandInterface {
    /**
     * Sets the rubber band geometry.
     *
     * This function does not change the previous rubber band visibility.
     *
     * @param pRectangle <div class="const">in</div> The new rubber band rectangle geometry.
     */
    setRubberBandRectangle(pRectangle: Rectangle): void;
    /**
     * Sets the rubber band visibility.
     *
     * This function does not change the previous rubber band geometry.
     *
     * **WARNING** : the rubber band will only be visible if pVisible is true and the Rectangle
     * set by {@link setRubberBandRectangle} is valid and non empty.
     *
     * @param pVisible <div class="const">in</div> The new visibility of the rubber band.
     */
    setRubberBandVisible(pVisible: boolean): void;
}
export { PrimitiveManagerRubberBandInterface };
/**
 * Represents a search request promise result content.
 *
 * This is the value of an {@link AsyncSearchResult}.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link SearchInterface.asyncSearch}
 * @see {@link AsyncSearchResult}
 * @category Data Retrievers
 */
interface AsyncSearchResultContent {
    /**
     * The search document result.
     *
     * This is the same as {@link SearchInterface.getSearchDocuments}.
     */
    searchDocuments?: Array<SearchDocumentResultInterface>;
    /**
     * The `geometric instance ids` result of the search.
     *
     * This is the same as {@link SearchInterface.getGeometricInstanceIds}.
     */
    geometricInstanceIds?: Uint32Array;
    /**
     * Tells if the search result would have returned more results if the search
     * would have been limited to a superior maximum number of hits.
     *
     * This is the same as {@link SearchInterface.hasOtherResults}.
     */
    hasOtherResults: boolean;
}
export { AsyncSearchResultContent };
/**
 * Represents a search request promise result.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link SearchInterface.asyncSearch}
 * @category Data Retrievers
 */
interface AsyncSearchResult {
    /**
     * The reason for the end of the promise.
     */
    reason: AsyncResultReason;
    /**
     * The returned value of the promise.
     * Undefined if the call failed.
     */
    value?: AsyncSearchResultContent;
}
export { AsyncSearchResult };
/**
 * The {@link SearchInterfaceSignal} lists all the signals that may be sent by the {@link SearchInterface}.
 *
 * These signals are emitted by the {@link SearchInterface} when a search result is ready.
 *
 * * {@link SearchReady} : the {@link SearchInterface} is ready, i.e. when it has
 * finished a search request, {@link SearchInterface.getSearchDocuments}, {@link SearchInterface.getGeometricInstanceIds} and
 * {@link SearchInterface.hasOtherResults} can be called.
 * * {@link SearchCancelled} : the search request by the {@link SearchInterface} has been cancelled.
 *
 * ```typescript
 * [[include:Search_example.ts]]
 * ```
 * <br>
 * Please refer to <a href="../index.html#search">Search</a> for more information about search requests.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link SearchInterface}
 * @category Events
 */
declare enum SearchInterfaceSignal {
    /**
     * Signal sent by the {@link SearchInterface} when {@link SearchInterface.search} is ready, i.e.
     * {@link SearchInterface.getSearchDocuments}, {@link SearchInterface.getGeometricInstanceIds} and
     * {@link SearchInterface.hasOtherResults} can be called.
     *
     * Such a signal does not tell if the call was successful or not but that the result is ready to be fetched.
     *
     * Warning, the SearchReady signal is sent later when {@link SearchInterface.cancelSearch} is called and the {@link SearchInterface} is searching.
     *
     * The attachment is a string telling the request id of the call to {@link SearchInterface.search} ({@link SearchInterface.getLastRequestId}).
     */
    SearchReady = "ready",
    /**
     * Signal sent by the {@link SearchInterface} when the {@link SearchInterface} has been cancelled.
     *
     * Such an event is fired if the {@link SearchInterface} was searching and {@link SearchInterface.cancelSearch} is called.
     * The {@link SearchReady} signal will be triggered shortly after.
     *
     * No attachment for events of this type.
     */
    SearchCancelled = "cancelled"
}
export { SearchInterfaceSignal };
/**
 * The SearchInterface interface is used to search content in the DMU.
 *
 * Please see <a href="../index.html#search">Search</a>.
 *
 * The SearchInterface interfaces are created through the {@link DataSessionInterface.createSearch} method.
 *
 * The list of signals the SearchInterface may trigger is available in the {@link SearchInterfaceSignal} enumeration.
 *
 * The search request is triggered through the {@link search} method. The result is not available right away, but the
 * event {@link SearchInterfaceSignal.SearchReady} is triggered when the result of the SearchInterface is available.
 * The result is available through the {@link getSearchDocuments} {@link getGeometricInstanceIds} and {@link hasOtherResults} functions.<br>
 * As said in <a href="../index.html#search">Search</a>, the search result consists in a list of metadata documents that match the query (an array of {@link SearchDocumentResultInterface} from {@link getSearchDocuments}),
 * it may be interesting to find the `part instances`, and the `geometric instances` that are linked to this search query. The {@link DocumentIdConverterInterface}
 * fulfill this requirement.<br>
 * The search result also provides right away the full list of `geometric instance ids` that are involved in the search result, regardless of
 * the given cap included (the **FULL** list of all `geometric instance ids` that match the query are **ALWAYS** provided) with {@link getGeometricInstanceIds}.
 *
 * Warning : there may be cases when the {@link getSearchDocuments}, {@link getGeometricInstanceIds} are not available such as when the SearchInterface is fetching data, i.e. when
 * {@link isRunning} returns true, or when the SearchInterface has been cancelled, i.e. when {@link isCancelled} returns true.
 *
 * A SearchInterface may be interrupted (cancelled) when the SearchInterface is running and {@link cancelSearch}
 * is called. In such cases, the {@link SearchInterfaceSignal.SearchCancelled} signal is fired, and shortly after, {@link SearchInterfaceSignal.SearchReady}
 * signal is fired, but {@link getSearchDocuments} and {@link getGeometricInstanceIds} will return undefined. Just call {@link search} with another (or the same) query to trigger a new
 * search request.
 *
 * If you call multiple times {@link search} before receiving the {@link SearchInterfaceSignal.SearchReady}, only one {@link SearchInterfaceSignal.SearchReady}
 * will be sent with the content of the last call to {@link search}.
 *
 * The syntax of the search is explained in the
 * <a href="../index.html#3djuump-infinite-literal-and-search-query-language">3djuump infinite literal and search query language</a>.
 *
 * The search query must be provided with a maximum number of hits to help reduce server usage and bandwidth. If the search result is capped to this value
 * then {@link hasOtherResults} returns true.
 *
 * The search query is always limited to a filtering context (see <a href="../index.html#filtering-context">Filtering Context</a>) from the {@link VisibilityContextInterface}.
 *
 * The search results may be filtered to include only a subset of the metadata inside the matching document, provided the {@link search} function is called with the list of metadata
 * names to return.
 *
 * ```typescript
 * [[include:Search_example.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:Search_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link SearchInterfaceSignal}
 * @see {@link SearchDocumentResultInterface}
 * @category Data Retrievers
 */
interface SearchInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Triggers a search query.
     *
     * The query is written in the <a href="../index.html#3djuump-infinite-literal-and-search-query-language">3djuump infinite literal and search query language</a>.
     *
     * The user must :
     * * Set a valid {@link VisibilityContextInterface} to limit the search on some documents (it may be the whole DMU with only an unconfigured {@link ConfContextInterface}).
     * * Choose to discard or not documents matching the query but not matching the filtering context (if not, document is present in the result but {@link SearchDocumentResultInterface.isInVisibilityCtx}
     * returns false).
     * * Limit the search to only pMaxDocumentResult results.
     * * Filter the result of the search with pMetadataNamesFilter.
     *
     * The pMetadataNamesFilter parameter defines the fields to retrieve from the metadata documents that match the query.
     * The parameter must be a an array of string containing the fields that must be returned. An empty array means the full
     * document will all the metadata will be returned in the resulting {@link SearchDocumentResultInterface}.
     * For example : if pMetadataNamesFilter is set to ["Name", "srcfile"], the result will return the content of the field
     * `Name` and the field `srcfile` from the metadata document.
     *
     * An event {@link SearchInterfaceSignal.SearchReady} is fired when the search is finished, then {@link getLastError | getLastError()} tells if the search was correctly performed.
     *
     * Returns `true` if the search query is started (and therefore the search query is valid). If not, just call {@link getLastError} to
     * get the reason why the procedure failed.
     *
     * If pVisibilityContext is modified during the execution, then the call is cancelled (see {@link cancelSearch}).
     *
     * @param pQuery <div class="const">in</div> The search query.
     * @param pVisibilityContext <div class="const">in</div> The filtering context.
     * @param pLimitResultsToVisibilityCtx <div class="const">in</div> If `true` limits the search query on the documents included in the filtering context.
     * @param pMaxDocumentResult <div class="const">in</div> The maximum number of resulting documents.
     * @param pMetadataNamesFilter <div class="const">in</div> The list of metadata names to include in the resulting {@link SearchDocumentResultInterface}.
     * @return `true` if the search query (pQuery) is valid, in the case the SearchInterface has begun running.
     */
    search(pQuery: string, pVisibilityContext: VisibilityContextInterface, pLimitResultsToVisibilityCtx: boolean, pMaxDocumentResult: number, pMetadataNamesFilter: Array<string>): boolean;
    /**
     * Asynchronously triggers a search query.
     *
     * The query is written in the <a href="../index.html#3djuump-infinite-literal-and-search-query-language">3djuump infinite literal and search query language</a>.
     *
     * The user must :
     * * Set a valid {@link VisibilityContextInterface} to limit the search on some documents (it may be the whole DMU with only an unconfigured {@link ConfContextInterface}).
     * * Choose to discard or not documents matching the query but not matching the filtering context (if not, document is present in the result but {@link SearchDocumentResultInterface.isInVisibilityCtx}
     * returns false).
     * * Limit the search to only pMaxDocumentResult results.
     * * Filter the result of the search with pMetadataNamesFilter.
     *
     * The pMetadataNamesFilter parameter defines the fields to retrieve from the metadata documents that match the query.
     * The parameter must be a an array of string containing the fields that must be returned. An empty array means the full
     * document will all the metadata will be returned in the resulting {@link SearchDocumentResultInterface}.
     * For example : if pMetadataNamesFilter is set to ["Name","srcfile"], the result will return the content of the field
     * `Name` and the field `srcfile` from the metadata document.
     *
     * Returns a promise.
     *
     * Please note that in case of multiple promises running at the same time on the same SearchInterface,
     * the first promises will be signalled as cancelled, the last as ok, but all calls to {@link getSearchDocuments}
     * after awaiting will return the same value.
     *
     * If pVisibilityContext is modified during the execution, then the call is cancelled (see {@link cancelSearch}).
     *
     * @param pQuery <div class="const">in</div> The search query.
     * @param pVisibilityContext <div class="const">in</div> The filtering context.
     * @param pLimitResultsToVisibilityCtx <div class="const">in</div> If true limits the search query on the documents included in the filtering context.
     * @param pMaxDocumentResult <div class="const">in</div> The maximum number of resulting documents.
     * @param pMetadataNamesFilter <div class="const">in</div> The list of metadata names to include in the resulting {@link SearchDocumentResultInterface}.
     * @return A promise. The promise is resolved with the reason (success, cancelled, disposed, bad input). In case of success, the promise contains the search result.
     */
    asyncSearch(pQuery: string, pVisibilityContext: VisibilityContextInterface, pLimitResultsToVisibilityCtx: boolean, pMaxDocumentResult: number, pMetadataNamesFilter: Array<string>): Promise<AsyncSearchResult>;
    /**
     * Cancels the computation of the search query (if any).
     *
     * A {@link SearchInterfaceSignal.SearchCancelled} signal is emitted if the
     * SearchInterface is retrieving data.
     *
     * @return `true` if the SearchInterface was running, else false.
     * @see {@link SearchInterfaceSignal.SearchCancelled}
     */
    cancelSearch(): boolean;
    /**
     * Tells if a search request is running.
     *
     * This is the case after calling {@link search}.
     *
     * @return `true` if a search request is running.
     */
    isRunning(): boolean;
    /**
     * Tells if a search request has been cancelled.
     *
     * This is generally the case after calling {@link cancelSearch} when the SearchInterface
     * is preforming a search.
     *
     * @return `true` if a search request is cancelled.
     */
    isCancelled(): boolean;
    /**
     * Gets the last error message returned by the search request.
     *
     * @return The last error message.
     */
    getLastError(): string;
    /**
     * Each call to {@link search} is assigned a request id.
     *
     * This call tels the id of the last call to {@link search}.
     *
     * @return The id of the last call to {@link search}.
     */
    getLastRequestId(): string;
    /**
     * Gets the list of all `geometric instance ids` which matched the search query from the last call of {@link search}.
     *
     * An Uint32Array is returned if the SearchInterface has finished computing.
     * Use {@link addEventListener} on the event {@link SearchInterfaceSignal.SearchReady} to
     * know when the SearchInterface is ready. This array contains all the matching `geometric instance ids` regardless
     * of the maximum number of hits set in {@link search}.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of matching <code>geometric instance ids</code>, or undefined if the
     * SearchInterface is searching or if the SearchInterface is in error or cancelled.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     */
    getGeometricInstanceIds(): Uint32Array | undefined;
    /**
     * Gets the list of all matching documents from the last call to the {@link search} function.
     *
     * An array of {@link SearchDocumentResultInterface} is returned if the SearchInterface has finished computing.
     * Use {@link addEventListener} on the event {@link SearchInterfaceSignal.SearchReady} to
     * know when the SearchInterface is ready. The document content is filtered with the pMetadataNamesFilter
     * set in the {@link search} function.
     *
     * The result is limited to the maximum number of hits when {@link search} was called.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The list of matching documents, or undefined if the
     * SearchInterface is searching or if the SearchInterface is in error or cancelled.
     * @see {@link SearchDocumentResultInterface}
     * @see {@link isRunning}
     * @see {@link isCancelled}
     */
    getSearchDocuments(): Array<SearchDocumentResultInterface> | undefined;
    /**
     * Tells if the search result would have returned more results if the search
     * would have been limited to a superior maximum number of hits.
     *
     * Returns `true` if the last search query contained more results than the maximum cap requested by the search and if the SearchInterface has finished computing.
     * Use {@link addEventListener} on the event {@link SearchInterfaceSignal.SearchReady} to
     * know when the SearchInterface is ready.
     *
     * @return `true` if the search result count exceeded the max document result cap and if the SearchInterface is not running.
     * @see {@link isRunning}
     * @see {@link isCancelled}
     */
    hasOtherResults(): boolean;
}
export { SearchInterface };
/**
 * An highlight result tells about the context of the result of a search query.
 *
 * The highlight result is included inside a {@link SearchDocumentResultInterface}.
 * The highlight result is basically a map of :
 * * An attribute name (or an empty string for a full text search).
 * * A list of strings that contain the query string of the search.
 *
 * ```typescript
 * [[include:Search_example.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:Search_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link SearchInterface}
 * @see {@link DocumentIdConverterInterface}
 * @category Data Retrievers
 */
type tHighlightResult = {
    [key: string]: Array<string>;
};
export { tHighlightResult };
/**
 * A SearchDocumentResultInterface interface is the result of a search request ({@link SearchInterface}).
 *
 * It contains :
 * * An integer document Id (do not store this id since it may change on a new build of the same project).
 * * A document name.
 * * A boolean telling if any `part instance` relative to the search request is in the filtering context of the search query.
 * * A metadata document.
 *
 * The metadata document may be used to customize the display of the search result. This metadata document is filtered with
 * the list of metadata names set in the ({@link SearchInterface.search} query.
 *
 * The document Id may be used in a [DocumentIdConverterInterface]] to get the `part instance ids` that are referenced by this metadata
 * document.
 *
 * ```typescript
 * [[include:Search_example.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:Search_asyncExample.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Data Retrievers</div><br>
 * @see {@link SearchInterface}
 * @see {@link DocumentIdConverterInterface}
 * @category Data Retrievers
 */
interface SearchDocumentResultInterface {
    /**
     * Gets the integer id of the document.
     *
     * This id will be used for a request with a {@link DocumentIdConverterInterface}.<br>
     * Warning : this id may change depending on the build you connect to.
     *
     * @return The integer id of the document.
     */
    getDocumentId(): number;
    /**
     * Gets the name of the document.
     *
     * @return The name of the document (informative).
     */
    getDocumentName(): string;
    /**
     * Gets the metadata of the document.
     *
     * Its content depends on the filter parameter set on the search request ({@link SearchInterface.search}).
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The object with the metadata.
     */
    getMetadataDocument(): Object;
    /**
     * Gets the highlight result of the search query.
     *
     * The highlight result consists in a map with 'metadata name' => array of strings that contain the search query.
     * An empty 'metadata name' means a full text search result.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> The highlight contexts of the search result.
     * @see {@link tHighlightResult}
     */
    getHighlights(): tHighlightResult;
    /**
     * Tells if any `part instance` linked to this document
     * is in the filtering context set in the search request ({@link SearchInterface.search}).
     *
     * @return `true` if any <code>part instance</code> linked to this document was in the filtering context set when {@link SearchInterface.search}
     * was called.
     */
    isInVisibilityCtx(): boolean;
}
export { SearchDocumentResultInterface };
/**
 * The VisibilityContextInterfaceSignal lists all the signals that may be sent by the {@link VisibilityContextInterface}.
 *
 * These signals are emitted by the {@link VisibilityContextInterface} when it is modified.
 *
 * There is no way to know when the {@link VisibilityContextInterface} is ready since the {@link VisibilityContextInterface}
 * is only an union of {@link FilterSolverInterface} and a {@link ConfContextInterface} that serves as a limitation for another
 * filtering request (usually a {@link FilterSolverInterface}). As such, there is no way to get the `part instances` or `geometric instances`
 * that are included in such a filtering context.
 *
 * The {@link VisibilityContextInterface} may fire the following signals :
 * * {@link VisibilitySolverAdded} : a {@link FilterSolverInterface} has been added into the {@link VisibilityContextInterface}.
 * * {@link VisibilitySolverRemoved} : a {@link FilterSolverInterface} has been removed from the {@link VisibilityContextInterface}.
 * * {@link VisibilityChanged} : a {@link FilterItemInterface} inside a {@link FilterSolverInterface} of the given {@link VisibilityContextInterface} has changed, or
 * a {@link FilterSolverInterface} has been removed or added.
 * * {@link VisibilityConfChanged} : a {@link ConfContextInterface} has changed or a new {@link ConfContextInterface} (or undefined)
 * has been set for this given {@link VisibilityContextInterface}.
 * * {@link VisibilityParentSolverChanged} : the {@link VisibilityContextInterface} has been set as a filtering context for a {@link FilterSolverInterface}.
 *
 * ```typescript
 * [[include:FilterSolverInterface_colorize.ts]]
 * ```
 * <br>
 * Please refer to the {@link VisibilityContextInterface} for more information.
 *
 * [[include:constStyle.md]]<br><div class="const">Events</div><br>
 * @see {@link VisibilityContextInterface}
 * @see {@link FilterSolverInterface}
 * @category Events
 */
declare enum VisibilityContextInterfaceSignal {
    /**
     * Signal sent by the {@link VisibilityContextInterface} when a {@link FilterSolverInterface} has been added.
     *
     * The attachment is the {@link FilterSolverInterface} that was added.
     */
    VisibilitySolverAdded = "added",
    /**
     * Signal sent by the {@link VisibilityContextInterface} when a {@link FilterSolverInterface} has been removed.
     *
     * The attachment is the {@link FilterSolverInterface} that was removed.
     */
    VisibilitySolverRemoved = "removed",
    /**
     * Signal sent by the {@link VisibilityContextInterface} when a {@link FilterItemInterface} inside a {@link FilterSolverInterface} of the given
     * {@link VisibilityContextInterface} has changed, or a {@link FilterSolverInterface} has been removed or added.
     *
     * In the case a {@link FilterSolverInterface} has been added or removed, the {@link VisibilityChanged} signal is sent after the {@link VisibilitySolverAdded}
     * or {@link VisibilitySolverRemoved} signal.
     *
     * No attachment for events of this type.
     */
    VisibilityChanged = "changed",
    /**
     * Signal sent by the {@link VisibilityContextInterface} when a {@link ConfContextInterface} has changed or a new {@link ConfContextInterface} (or undefined)
     * has been set for this given {@link VisibilityContextInterface}.
     *
     * No attachment for events of this type.
     */
    VisibilityConfChanged = "confChanged",
    /**
     * Signal sent by the {@link VisibilityContextInterface} when it has been set as a filtering context for a {@link FilterSolverInterface} (or unset).
     *
     * The attachment is the {@link FilterSolverInterface} that has been modified, or undefined if the {@link FilterSolverInterface} filtering context has been cleared.
     */
    VisibilityParentSolverChanged = "parentSolverChanged"
}
export { VisibilityContextInterfaceSignal };
/**
 * The VisibilityContextInterface interface is used to restrict further filtering/data retrieving/conversion requests.
 *
 * The VisibilityContextInterface can be seen as a filtering context.
 *
 * The VisibilityContextInterface is basically a union of {@link FilterSolverInterface}, intersected with a required {@link ConfContextInterface}.
 * The resulting `part instances` will be the union of all `part instances` of
 * the {@link FilterSolverInterface} inside this context, intersected with the `part instances` of the configuration expressed by the {@link ConfContextInterface}.
 *
 * Further filtering/data retrieving/conversion requests results will be intersected with these `part instances` (see the functions that takes as arguments
 * a VisibilityContextInterface : e.g. {@link DocumentIdConverterInterface.convert}, {@link FilterSolverInterface.setVisibilityContext}, {@link IdCardGetterInterface.retrieveIdCard}, etc).
 *
 * A VisibilityContextInterface is considered valid when its {@link ConfContextInterface} is not `undefined` (see <a href="../index.html#filtering-context">Filtering Context</a>).
 *
 * This object cannot be used explicitly and is not updated on demand. It neither holds any data.
 * The only way to know the pat instances of a VisibilityContextInterface is to iterate over its {@link FilterSolverInterface}, and make the intersection with the
 * `part instances` of the {@link ConfContextInterface}.
 *
 * This object may be updated upon calls to :
 * * data retrieving/conversion requests : {@link DocumentIdConverterInterface.convert}, {@link IdCardGetterInterface.retrieveIdCard}, etc.
 * * filtering updates : {@link DataSessionInterface.update}.
 *
 * The VisibilityContextInterface interfaces are created through the {@link DataSessionInterface.createVisibilityContext} method.
 *
 * **Discard** (with {@link dispose}) existing VisibilityContextInterface before loading a new {@link DataSessionInterface}.
 *
 * The list of signals the VisibilityContextInterface may trigger is available in the {@link VisibilityContextInterfaceSignal} enumeration.
 *
 * ```typescript
 * [[include:FilterSolverInterface_colorize.ts]]
 * ```
 * <br>
 * or asynchronously :
 *
 * ```typescript
 * [[include:FilterSolverInterface_asyncColorize.ts]]
 * ```
 * <br>
 * Please make sure the destination browser supports promises before using async calls.
 *
 * [[include:constStyle.md]]<br><div class="const">Metadata</div><br>
 * @see {@link VisibilityContextInterfaceSignal}
 * @see {@link FilterSolverInterface}
 * @see {@link ConfContextInterface}
 * @see {@link DataSessionInterface}
 * @category Metadata
 */
interface VisibilityContextInterface extends InfiniteObjectDispatcherInterface {
    /**
     * Sets the required {@link ConfContextInterface} used to compute this filtering context.
     *
     * The `part instances` inside
     * the VisibilityContextInterface will be intersected by this {@link ConfContextInterface}.
     *
     * A VisibilityContextInterface is considered valid if it contains a defined {@link ConfContextInterface}.
     *
     * Triggers the {@link VisibilityContextInterfaceSignal.VisibilityConfChanged} signal.
     *
     * @param pConfContext <div class="const">in</div> The new {@link ConfContextInterface} to use.
     * @return `true` if pConfContext is valid.
     * @see {@link ConfContextInterface}
     */
    setConfContext(pConfContext: ConfContextInterface | undefined): boolean;
    /**
     * Gets the {@link ConfContextInterface} used to compute this filtering context.
     *
     * @return The {@link ConfContextInterface} used to compute this filtering context.
     */
    getConfContext(): ConfContextInterface | undefined;
    /**
     * Sets the identifier of the VisibilityContextInterface.
     * Make sure the id is unique.
     * A unique VisibilityContextInterface identifier is created if the identifier is not overridden.
     *
     * @param pVisibilityCtxId <div class="const">in</div> The new identifier of the VisibilityContextInterface.
     */
    setVisibilityContextId(pVisibilityCtxId: string): void;
    /**
     * Gets the identifier of the VisibilityContextInterface.
     *
     * @return The identifier of the VisibilityContextInterface.
     */
    getVisibilityContextId(): string;
    /**
     * Inserts a {@link FilterSolverInterface} at the given position.
     *
     * Any invalid position will result in the {@link FilterSolverInterface} being pushed at the end of
     * the list of {@link FilterSolverInterface}.
     *
     * For example : <code> insertFilterSolver(-1,lSolver)</code> actually push back a {@link FilterSolverInterface}.
     *
     * Triggers the {@link VisibilityContextInterfaceSignal.VisibilitySolverAdded} and {@link VisibilityContextInterfaceSignal.VisibilityChanged} signals.
     *
     * @param pPosition <div class="const">in</div> The position to insert the filter to (between <code>0</code> and <code>{@link getFilterSolvers | getFilterSolvers()}.length-1</code> included, else pushed back).
     * @param pFilterSolver <div class="const">in</div> The {@link FilterSolverInterface} to add.
     * @return `true` if pFilterSolver is valid.
     */
    insertFilterSolver(pPosition: number, pFilterSolver: FilterSolverInterface): boolean;
    /**
     * Removes a {@link FilterSolverInterface} at the given position.
     *
     * If no filter is at the given position, returns false and does nothing.
     *
     * Triggers the {@link VisibilityContextInterfaceSignal.VisibilitySolverRemoved} and {@link VisibilityContextInterfaceSignal.VisibilityChanged} signals.
     *
     * @param pPosition <div class="const">in</div> The position of the {@link FilterSolverInterface} to remove (between <code>0</code> and <code>{@link getFilterSolvers | getFilterSolvers()}.length-1</code> included).
     * @return `true` if the {@link FilterSolverInterface} was removed.
     */
    removeFilterSolver(pPosition: number): boolean;
    /**
     * Gets the list of {@link FilterSolverInterface} inside this filtering context.
     *
     * DO not insert or remove any {@link FilterSolverInterface} directly in this array, this
     * will make the {@link VisibilityContextInterface} ignore the changes inside and lead to
     * undefined behavior.
     *
     * DO NOT modify this data in place, this results in undefined behavior.
     *
     * @return <div class="const">const</div> the list of {@link FilterSolverInterface} of this filtering context.
     */
    getFilterSolvers(): Array<FilterSolverInterface>;
    /**
     * Gets the order inside the list of {@link FilterSolverInterface} ({@link getFilterSolvers}) of this {@link FilterSolverInterface}.
     *
     * @param pFilterSolver <div class="const">in</div> The {@link FilterSolverInterface} to query position from.
     * @return The order of pFilterSolver, -1 if pFilterSolver is not inside.
     */
    getFilterSolverOrder(pFilterSolver: FilterSolverInterface): number;
    /**
     * Gets the last error returned by the update of the {@link ConfContextInterface} and the list of {@link FilterSolverInterface} inside.
     *
     * Errors may be updated when the VisibilityContextInterface is updated upon :
     * * Data retrieving/conversion requests : {@link DocumentIdConverterInterface.convert}, {@link IdCardGetterInterface.retrieveIdCard}, etc.
     * * Filtering updates : {@link DataSessionInterface.update}.
     *
     * There is no way to explicitly update a VisibilityContextInterface.
     *
     * @return The last error message (if any, or an empty string if no error occurred).
     */
    getLastError(): string;
}
export { VisibilityContextInterface };
/**
 * An axis aligned Bounding Box.
 *
 * This object contains a center and the half extent of the resulting box.
 *
 * [[include:constStyle.md]]<br><div class="const">Maths</div><br>
 * @category Maths
 */
declare class AABB {
    /**
     * The center of the axis aligned bounding box (x,y and z in this order).
     */
    mCenter: Vector3;
    /**
     * The **half** extent on each axis (x,y and z in this order).
     */
    mHalfExtent: Vector3;
    /**
     * Creates an axis aligned bounding box.
     *
     * If omitted, pCenter will be set to the origin (0,0,0).
     * If omitted, pHalfExtent will be set to (0,0,0), this is an invalid box.
     *
     * @param pCenter <div class="const">in</div> The center of the AABB.
     * @param pHalfExtent <div class="const">in</div> The half extent on each axis of the AABB.
     * @return The newly constructed AABB.
     */
    constructor(pCenter?: Vector3, pHalfExtent?: Vector3);
    /**
     * Serializes the given AABB to JSON.
     *
     * @return the JSON representation of the object.
     * @internal
     */
    toJSON(_pKey?: any): Object;
    /**
     * Tells if the two AABB are strictly identical.
     *
     * @param pOtherAABB <div class="const">in</div> The other AABB to compare to.
     * @return `true` if the two AABB are strictly identical.
     */
    equals(pOtherAABB: AABB): boolean;
    /**
     * Initializes the AABB with pMin,pMax.
     *
     * @param pMin <div class="const">in</div> The min point of the AABB.
     * @param pMax <div class="const">in</div> The max point of the AABB.
     * @return `false` if the pMax is lower than pMin else true.
     */
    setMinMax(pMin: Vector3, pMax: Vector3): boolean;
    /**
     * Gets the radius of the sphere centered on the center of the box which contains the box.
     *
     * This is equal to &#x221a;(mHalfExtent.mHalfExtent).
     *
     * @return The box radius.
     */
    getRadius(): number;
    /**
     * Grows the AABB to contain the box defined by Min : (pMinX,pMinY,pMinZ), Max : (pMaxX,pMaxY,pMaxZ).
     *
     * There is no check that pMinX <= pMaxX, pMinY <= pMaxY, pMinZ <= pMaxZ.
     *
     * @param pMinX <div class="const">in</div> Min x coordinate of the box to include.
     * @param pMinY <div class="const">in</div> Min y coordinate of the box to include.
     * @param pMinZ <div class="const">in</div> Min z coordinate of the box to include.
     * @param pMaxX <div class="const">in</div> Max x coordinate of the box to include.
     * @param pMaxY <div class="const">in</div> Max y coordinate of the box to include.
     * @param pMaxZ <div class="const">in</div> Max z coordinate of the box to include.
     * @return `false` is the pMax is lower than pMin else true.
     */
    grow(pMinX: number, pMinY: number, pMinZ: number, pMaxX: number, pMaxY: number, pMaxZ: number): boolean;
    /**
     * Grows the AABB to contain the box defined by pMin, pMax.
     *
     * There is no check that pMin <= pMax.
     *
     * @param pMin <div class="const">in</div> Min point of the box to include.
     * @param pMax <div class="const">in</div> Max point of the box to include.
     * @return `false` if pMax is lower than pMin else true.
     */
    growVector(pMin: Vector3, pMax: Vector3): boolean;
    /**
     * Grows the AABB to contain the box pBox.
     *
     * @param pBox <div class="const">in</div> The box to include.
     * @return `false` is the pBox is invalid else true.
     */
    growAABB(pBox: AABB): boolean;
    /**
     * Grows the AABB to contain the box expressed as center pBoxCenter and half extent pHalfExtent.
     *
     * @param pBoxCenter <div class="const">in</div> The center of the box to include.
     * @param pHalfExtent <div class="const">in</div> The half extent of the box to include.
     * @return `false` if the pHalfExtent contains negative value else true.
     */
    growAABBCenterExtent(pBoxCenter: Vector3, pHalfExtent: Vector3): boolean;
    /**
     * Tells where the box is located relative to the plane defined by the equation pPlane : (ax+by+cz=d).
     *
     * The positive side is given by the normal vector of the plane : (a,b,c) which are the three first parameters of pPlane.
     * If the box is strictly included in the area of the normal, then 1 is returned, if the box is strictly
     * included in the area opposite to the normal of the plane, -1 is returned. If the box intersects the plan, 0 is returned.
     *
     * @param pPlane <div class="const">in</div> The plan to test with (equation of the plane ax+by+cz=d).
     * @return 1 on positive side, -1 on negative side, 0 if it intersects the plane.
     * @see {@link Vector4}
     */
    whichSide(pPlane: Vector4): 0 | -1 | 1;
    /**
     * Tests if a point is in the box.
     *
     * @param pPosition <div class="const">in</div> The point to test.
     * @return `true` if point is in the AABB, else false.
     */
    testPosition(pPosition: Vector3): boolean;
    /**
     * Return if the AABB is valid ie all values of the half extent must be positive.
     *
     * @return `true` if all HalfExtent values are positive.
     */
    isValid(): boolean;
    /**
     * Clamps the given AABB with the given limit AABB.
     *
     * If the boxes intersect, then the result is the intersection of the 2 boxes.
     * If the 2 boxes do not intersect, then the AABB is restricted to the intersection of each axis in each direction.
     * The result is therefore not the intersection of the two boxes, but the union of the intersection in each axis. If there is no
     * intersection in this axis, the extent in this axis is left unchanged.
     *
     * @param pLimitAABB <div class="const">in</div> The AABB to clamp with.
     * @return `true` if pLimitAABB is valid.
     */
    clampAABB(pLimitAABB: AABB): boolean;
    /**
     * Transforms the given AABB by multiplying all the extent points of the given AABB, and stores the AABB that contains this Oriented Bounding Box (OBB).
     *
     * If pOut is omitted, then the resulting AABB is stored in "this".
     *
     * @param pXForm <div class="const">in</div> The matrix to multiply the extents point with.
     * @param pOut <div class="const">out</div> The resulting AABB ("this" if omitted).
     * @return The transformed AABB pOut.
     */
    transformBy(pXForm: Matrix4, pOut?: AABB): AABB;
}
export { AABB };
/**
 * A 4x4 Matrix.
 *
 * [[include:tableStyle.md]]
 * Such a matrix contains a translation and a rotation/scale part.
 * The matrix contains an array of 16 floats, which memory layout is the following.
 *
 * If you consider the vector (V) matrix (M) multiplication in line : V' = V x M then the
 * memory layout of the matrix is the following, with the translation part [T0,T1,T2] on the
 * bottom :
 * <table class="matrix">
 * <tr><td>array[0]</td><td>array[1]</td><td>array[2]</td><td>0 (array[3])</td></tr>
 * <tr><td>array[4]</td><td>array[5]</td><td>array[6]</td><td>0 (array[7])</td></tr>
 * <tr><td>array[8]</td><td>array[9]</td><td>array[10]</td><td>0 (array[11])</td></tr>
 * <tr><td>T0 : array[12]</td><td>T1 : array[13]</td><td>T2 : array[14]</td><td>1 (array[15])</td></tr>
 * </table>
 * <br>
 * If on the contrary you consider the vector (V) matrix (M) multiplication in column : V' = M x V then the
 * memory layout of the matrix is the following, with the translation part [T0,T1,T2] on the
 * right :
 * <table class="matrix">
 * <tr><td>array[0]</td><td>array[4]</td><td>array[8]</td><td>T0 : array[12]</td></tr>
 * <tr><td>array[1]</td><td>array[5]</td><td>array[9]</td><td>T1 : array[13]</td></tr>
 * <tr><td>array[2]</td><td>array[6]</td><td>array[10]</td><td>T2 : array[14]</td></tr>
 * <tr><td>0 (array[3])</td><td>0 (array[7])</td><td>0 (array[11])</td><td>1 (array[15])</td></tr>
 * </table>
 * <br>
 * The multiplication in column is the convention adopted in the Matrix4 class, when a matrix M2 is multiplied on the left of M1,
 * then any point P transformed by the resulting transformation (P') will be first transformed by M1 and then M2 :
 * <br>
 * P' = M2 x M1 x P.
 *
 * [[include:constStyle.md]]<br><div class="const">Maths</div><br>
 * @category Maths
 */
declare class Matrix4 {
    /**
     * A 16 float array that contains the values of the matrix.
     */
    array: Float32Array;
    /**
     * Builds a new Matrix4 from the following memory layout.
     *
     * If you consider the matrix vector multiplication in line (V' = V x M), then the following matrix is built (translation [T0,T1,T2] on the bottom) :
     * <br>
     * <table class="matrix">
     * <tr><td>a</td><td>b</td><td>c</td><td>0 (d)</td></tr>
     * <tr><td>e</td><td>f</td><td>g</td><td>0 (h) </td></tr>
     * <tr><td>i</td><td>j</td><td>k</td><td>0 (l)</td></tr>
     * <tr><td>T0 : m</td><td>T1 : n</td><td>T2 : o</td><td>1 (p)</td></tr>
     * </table>
     * <br>
     * If you consider the matrix vector multiplication in column (V' = M x V), then the following matrix is built (translation [T0,T1,T2] on the right) :
     * <br>
     * <table class="matrix">
     * <tr><td>a</td><td>e</td><td>i</td><td>T0 : m</td></tr>
     * <tr><td>b</td><td>f</td><td>j</td><td>T1 : n</td></tr>
     * <tr><td>c</td><td>g</td><td>k</td><td>T2 : o</td></tr>
     * <tr><td>0 (d)</td><td>0 (h)</td><td>0 (l)</td><td>1 (p)</td></tr>
     * </table>
     * <br>
     * Returns the newly constructed matrix.
     *
     * @param a <div class="const">in</div> Rotation/scale part.
     * @param b <div class="const">in</div> Rotation/scale part.
     * @param c <div class="const">in</div> Rotation/scale part.
     * @param d <div class="const">in</div> Should be 0.
     * @param e <div class="const">in</div> Rotation/scale part.
     * @param f <div class="const">in</div> Rotation/scale part.
     * @param g <div class="const">in</div> Rotation/scale part.
     * @param h <div class="const">in</div> Should be 0.
     * @param i <div class="const">in</div> Rotation/scale part.
     * @param j <div class="const">in</div> Rotation/scale part.
     * @param k <div class="const">in</div> Rotation/scale part.
     * @param l <div class="const">in</div> Should be 0.
     * @param m <div class="const">in</div> Translation upon x.
     * @param n <div class="const">in</div> Translation upon y.
     * @param o <div class="const">in</div> Translation upon z.
     * @param p <div class="const">in</div> Should be 1.
     * @return The newly created Matrix4.
     */
    constructor(a?: number, b?: number, c?: number, d?: number, e?: number, f?: number, g?: number, h?: number, i?: number, j?: number, k?: number, l?: number, m?: number, n?: number, o?: number, p?: number);
    /**
     * Copies "this" to pOther matrix.
     *
     * If pOther is not supplied, a new Matrix4 is allocated.
     *
     * @param pOut <div class="const">out</div> The Matrix4 to copy to.
     * @return The copied matrix.
     */
    copy(pOut?: Matrix4): Matrix4;
    /**
     * Sets values for the matrix.
     *
     * If you consider the matrix vector multiplication in line (V' = V x M), then the following matrix is set (translation [T0,T1,T2] on the bottom) :
     * <br>
     * <table class="matrix">
     * <tr><td>a</td><td>b</td><td>c</td><td>0 (d)</td></tr>
     * <tr><td>e</td><td>f</td><td>g</td><td>0 (h) </td></tr>
     * <tr><td>i</td><td>j</td><td>k</td><td>0 (l)</td></tr>
     * <tr><td>T0 : m</td><td>T1 : n</td><td>T2 : o</td><td>1 (p)</td></tr>
     * </table>
     * <br>
     * If you consider the matrix vector multiplication in column (V' = M x V), then the following matrix is set (translation [T0,T1,T2] on the right) :
     * <br>
     * <table class="matrix">
     * <tr><td>a</td><td>e</td><td>i</td><td>T0 : m</td></tr>
     * <tr><td>b</td><td>f</td><td>j</td><td>T1 : n</td></tr>
     * <tr><td>c</td><td>g</td><td>k</td><td>T2 : o</td></tr>
     * <tr><td>0 (d)</td><td>0 (h)</td><td>0 (l)</td><td>1 (p)</td></tr>
     * </table>
     * <br>
     * Returns 'this'.
     *
     * @param a <div class="const">in</div> Rotation/scale part.
     * @param b <div class="const">in</div> Rotation/scale part.
     * @param c <div class="const">in</div> Rotation/scale part.
     * @param d <div class="const">in</div> Should be 0.
     * @param e <div class="const">in</div> Rotation/scale part.
     * @param f <div class="const">in</div> Rotation/scale part.
     * @param g <div class="const">in</div> Rotation/scale part.
     * @param h <div class="const">in</div> Should be 0.
     * @param i <div class="const">in</div> Rotation/scale part.
     * @param j <div class="const">in</div> Rotation/scale part.
     * @param k <div class="const">in</div> Rotation/scale part.
     * @param l <div class="const">in</div> Should be 0.
     * @param m <div class="const">in</div> Translation upon x.
     * @param n <div class="const">in</div> Translation upon y.
     * @param o <div class="const">in</div> Translation upon z.
     * @param p <div class="const">in</div> Should be 1.
     * @return `this`.
     */
    set(a?: number, b?: number, c?: number, d?: number, e?: number, f?: number, g?: number, h?: number, i?: number, j?: number, k?: number, l?: number, m?: number, n?: number, o?: number, p?: number): Matrix4;
    /**
     * Tells if the two matrices are strictly identical.
     *
     * @param pOtherMatrix <div class="const">in</div> The other matrix to compare to.
     * @return `true` if the two matrices are strictly identical.
     */
    equals(pOtherMatrix: Matrix4): boolean;
    /**
     * Serializes the given Matrix4 to JSON.
     *
     * @return the JSON representation of the object.
     * @internal
     */
    toJSON(_pKey?: any): Object;
    /**
     * Sets the matrix to be the rotation from the given quaternion values [pX,pY,pZ,pW].
     *
     * @param pX <div class="const">in</div> <code>x</code> component of the quaternion.
     * @param pY <div class="const">in</div> <code>y</code> component of the quaternion.
     * @param pZ <div class="const">in</div> <code>z</code> component of the quaternion.
     * @param pW <div class="const">in</div> <code>w</code> component of the quaternion.
     * @return `this`.
     * @see {@link Quaternion}
     */
    setQuaternion(pX: number, pY: number, pZ: number, pW: number): Matrix4;
    /**
     * Resets the matrix to be a translation.
     *
     * The rotation/scale part is set to the identity.
     *
     * @param pX <div class="const">in</div> <code>x</code> component of the translation.
     * @param pY <div class="const">in</div> <code>y</code> component of the translation.
     * @param pZ <div class="const">in</div> <code>z</code> component of the translation.
     * @return `this`.
     */
    setTranslation(pX: number, pY: number, pZ: number): Matrix4;
    /**
     * Resets the matrix to be a scale (pX,pY,pZ).
     *
     * @param pX <div class="const">in</div> <code>x</code> component of the scale to set.
     * @param pY <div class="const">in</div> <code>y</code> component of the scale to set.
     * @param pZ <div class="const">in</div> <code>z</code> component of the scale to set.
     * @return `this`.
     */
    resetToScale(pX: number, pY: number, pZ: number): Matrix4;
    /**
     * Inverts the transformation and stores the result to pOut.
     *
     * If the matrix "this" is non-invertible, then pOut stores a copy of "this".
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pOut <div class="const">out</div> The inverted matrix result.
     * @return `pOut`.
     */
    invert(pOut?: Matrix4): Matrix4;
    /**
     * Computes the matrix determinant.
     *
     * @return The determinant of the matrix.
     */
    determinant(): number;
    /**
     * Computes the rotation of "this" around the X axis and stores the result in pOut.
     *
     * This computes the composition of "this" and the rotation of pAngle around X.<br>
     * Any point P transformed by the resulting transformation will first be multiplied by "this", and then the
     * rotation around X.<br>
     * Be P' the transformed point of P, and Rx the rotation around X : <br>
     * Depending on your convention : P' = Rx x this x P <br>
     * or P' = P x this x Rx.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * @param pAngle <div class="const">in</div> The angle in radian for the rotation around X.
     * @param pOut <div class="const">out</div> The result of the rotation of the matrix.
     * @return `pOut`.
     */
    rotateX(pAngle: number, pOut?: Matrix4): Matrix4;
    /**
     * Computes the rotation of "this" around the Y axis and stores the result in pOut.
     *
     * This computes the composition of "this" and the rotation of pAngle around Y.<br>
     * Any point P transformed by the resulting transformation will first be multiplied by "this", and then the
     * rotation around Y.<br>
     * Be P' the transformed point of P, and Ry the rotation around Y :<br>
     * Depending on your convention : P' = Ry x this x P<br>
     * or P' = P x this x Ry.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * @param pAngle <div class="const">in</div> The angle in radian for the rotation around Y.
     * @param pOut <div class="const">out</div> The result of the rotation of the matrix.
     * @return `pOut`.
     */
    rotateY(pAngle: number, pOut?: Matrix4): Matrix4;
    /**
     * Computes the rotation of "this" around the Z axis and stores the result in pOut.
     *
     * This computes the composition of "this" and the rotation of pAngle around Z.<br>
     * Any point P transformed by the resulting transformation will first be multiplied by "this", and then the
     * rotation around Z.<br>
     * Be P' the transformed point of P, and Rz the rotation around Z :<br>
     * Depending on your convention : P' = Rz x this x P<br>
     * or P' = P x this x Rz.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * @param pAngle <div class="const">in</div> The angle in radian for the rotation around Z.
     * @param pOut <div class="const">out</div> The result of the rotation of the matrix.
     * @return `pOut`.
     */
    rotateZ(pAngle: number, pOut?: Matrix4): Matrix4;
    /**
     * Computes the composition of pFirstTransform and "this" and stores the result in pOut.
     *
     * Any point P transformed by the resulting transformation will first be multiplied by pFirstTransform, and then "this".
     * <br>Be P' the transformed point of P :<br>
     * Depending on your convention : P' = this x pFirstTransform x P (hence multiply right) <br>
     * or P' = P x pFirstTransform x this.
     *
     * <br>If pOut is omitted, the result is stored in "this".
     *
     * @param pFirstTransform <div class="const">in</div> The transform to apply first when transforming points.
     * @param pOut <div class="const">out</div> The result of the composition of the matrices.
     * @return `pOut`.
     */
    multiplyMatrixRight(pFirstTransform: Matrix4, pOut?: Matrix4): Matrix4;
    /**
     * Computes the composition of "this" and pSecondTransform and stores the result in pOut.
     *
     * Any point P transformed by the resulting transformation will first be multiplied by "this", and then pSecondTransform.
     * <br>Be P' the transformed point of P :<br>
     * Depending on your convention : P' = pSecondTransform x this x P (hence multiply left) <br>
     * or P' = P x this x pSecondTransform.
     *
     * <br>If pOut is omitted, the result is stored in "this".
     *
     * @param pSecondTransform <div class="const">in</div> The transform to apply after "this" when transforming points.
     * @param pOut <div class="const">out</div> The result of the composition of the matrices.
     * @return `pOut`.
     */
    multiplyMatrixLeft(pSecondTransform: Matrix4, pOut?: Matrix4): Matrix4;
    /**
     * Computes the composition of the matrix represented by the values Mval and "this" and stores the result in pOut.
     *
     * Any point P transformed by the resulting transformation will first be multiplied by Mval, and then "this".
     * <br>Be P' the transformed point of P :
     * <br>Depending on your convention : P' = this x Mval x P (hence multiply right) <br>
     * or P' = P x Mval x this.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * If you consider the matrix vector multiplication in line (V' = V x Mval), then Mval is set to (translation [T0,T1,T2] on the bottom) :
     *
     * <table class="matrix">
     * <tr><td>a</td><td>b</td><td>c</td><td>0 (d)</td></tr>
     * <tr><td>e</td><td>f</td><td>g</td><td>0 (h) </td></tr>
     * <tr><td>i</td><td>j</td><td>k</td><td>0 (l)</td></tr>
     * <tr><td>T0 : m</td><td>T1 : n</td><td>T2 : o</td><td>1 (p)</td></tr>
     * </table>
     * <br>
     * If you consider the matrix vector multiplication in column (V' = Mval x V), then Mval is set (translation [T0,T1,T2] on the right) :
     *
     * <table class="matrix">
     * <tr><td>a</td><td>e</td><td>i</td><td>T0 : m</td></tr>
     * <tr><td>b</td><td>f</td><td>j</td><td>T1 : n</td></tr>
     * <tr><td>c</td><td>g</td><td>k</td><td>T2 : o</td></tr>
     * <tr><td>0 (d)</td><td>0 (h)</td><td>0 (l)</td><td>1 (p)</td></tr>
     * </table><br>
     * <br>
     * Then when using Matrices :
     *
     * ```typescript
     * this.multiplyMatrixRight(pFirstTransform, pOut);
     * ```
     * <br>
     * is equivalent to :
     *
     * ```typescript
     * this.multiplyRight(
     *	pFirstTransform.array[0], pFirstTransform.array[1], pFirstTransform.array[2], pFirstTransform.array[3],
     *	pFirstTransform.array[4], pFirstTransform.array[5], pFirstTransform.array[6], pFirstTransform.array[7],
     *	pFirstTransform.array[8], pFirstTransform.array[9], pFirstTransform.array[10], pFirstTransform.array[11],
     *	pFirstTransform.array[12], pFirstTransform.array[13], pFirstTransform.array[14], pFirstTransform.array[15],
     *	pOut
     * );
     * ```
     * <br>
     * Please refer to some matrices courses to get more information.
     *
     * @param a <div class="const">in</div> Rotation/scale part.
     * @param b <div class="const">in</div> Rotation/scale part.
     * @param c <div class="const">in</div> Rotation/scale part.
     * @param d <div class="const">in</div> Should be 0.
     * @param e <div class="const">in</div> Rotation/scale part.
     * @param f <div class="const">in</div> Rotation/scale part.
     * @param g <div class="const">in</div> Rotation/scale part.
     * @param h <div class="const">in</div> Should be 0.
     * @param i <div class="const">in</div> Rotation/scale part.
     * @param j <div class="const">in</div> Rotation/scale part.
     * @param k <div class="const">in</div> Rotation/scale part.
     * @param l <div class="const">in</div> Should be 0.
     * @param m <div class="const">in</div> Translation upon x.
     * @param n <div class="const">in</div> Translation upon y.
     * @param o <div class="const">in</div> Translation upon z.
     * @param p <div class="const">in</div> Should be 1.
     * @param pOut <div class="const">out</div> The result of the composition of the matrices.
     * @return `pOut`.
     */
    multiplyRight(a: number, b: number, c: number, d: number, e: number, f: number, g: number, h: number, i: number, j: number, k: number, l: number, m: number, n: number, o: number, p: number, pOut?: Matrix4): Matrix4;
    /**
     * Computes the composition of "this" and the matrix represented by the values Mval and stores the result in pOut.
     *
     * Any point P transformed by the resulting transformation will first be multiplied by "this", and then Mval.
     * <br>Be P' the transformed point of P :
     * <br>Depending on your convention : P' = Mval x this x P (hence multiply left) <br>
     * or P' = P x this x Mval.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * If you consider the matrix vector multiplication in line (V' = V x Mval), then Mval is set to (translation [T0,T1,T2] on the bottom) :
     *
     * <table class="matrix">
     * <tr><td>a</td><td>b</td><td>c</td><td>0 (d)</td></tr>
     * <tr><td>e</td><td>f</td><td>g</td><td>0 (h) </td></tr>
     * <tr><td>i</td><td>j</td><td>k</td><td>0 (l)</td></tr>
     * <tr><td>T0 : m</td><td>T1 : n</td><td>T2 : o</td><td>1 (p)</td></tr>
     * </table>
     * <br>
     * If you consider the matrix vector multiplication in column (V' = Mval x V), then Mval is set (translation [T0,T1,T2] on the right) :
     *
     * <table class="matrix">
     * <tr><td>a</td><td>e</td><td>i</td><td>T0 : m</td></tr>
     * <tr><td>b</td><td>f</td><td>j</td><td>T1 : n</td></tr>
     * <tr><td>c</td><td>g</td><td>k</td><td>T2 : o</td></tr>
     * <tr><td>0 (d)</td><td>0 (h)</td><td>0 (l)</td><td>1 (p)</td></tr>
     * </table><br>
     * <br>
     * Then when using Matrices :
     *
     * ```typescript
     * this.multiplyMatrixLeft(pSecondTransform, pOut);
     * ```
     * <br>
     * is equivalent to :
     *
     * ```typescript
     * this.multiplyLeft(
     *	pSecondTransform.array[0], pSecondTransform.array[1], pSecondTransform.array[2], pSecondTransform.array[3],
     *	pSecondTransform.array[4], pSecondTransform.array[5], pSecondTransform.array[6], pSecondTransform.array[7],
     *	pSecondTransform.array[8], pSecondTransform.array[9], pSecondTransform.array[10], pSecondTransform.array[11],
     *	pSecondTransform.array[12], pSecondTransform.array[13], pSecondTransform.array[14], pSecondTransform.array[15],
     *	pOut
     * );
     * ```
     * <br>
     * Please refer to some matrices courses to get more information.
     *
     * @param a <div class="const">in</div> Rotation/scale part.
     * @param b <div class="const">in</div> Rotation/scale part.
     * @param c <div class="const">in</div> Rotation/scale part.
     * @param d <div class="const">in</div> Should be 0.
     * @param e <div class="const">in</div> Rotation/scale part.
     * @param f <div class="const">in</div> Rotation/scale part.
     * @param g <div class="const">in</div> Rotation/scale part.
     * @param h <div class="const">in</div> Should be 0.
     * @param i <div class="const">in</div> Rotation/scale part.
     * @param j <div class="const">in</div> Rotation/scale part.
     * @param k <div class="const">in</div> Rotation/scale part.
     * @param l <div class="const">in</div> Should be 0.
     * @param m <div class="const">in</div> Translation upon x.
     * @param n <div class="const">in</div> Translation upon y.
     * @param o <div class="const">in</div> Translation upon z.
     * @param p <div class="const">in</div> Should be 1.
     * @param pOut <div class="const">out</div> The result of the composition of the matrices.
     * @return `pOut`.
     */
    multiplyLeft(a: number, b: number, c: number, d: number, e: number, f: number, g: number, h: number, i: number, j: number, k: number, l: number, m: number, n: number, o: number, p: number, pOut?: Matrix4): Matrix4;
    /**
     * Gets the pColum th "column" of the matrix.
     *
     * This function actually takes the column if the
     * Matrix vector multiplication is in column : V' = M x V .
     *
     * If you consider the matrix vector multiplication in column (V' = M x V, translation [T0,T1,T2] on the right) :
     *
     * <table class="matrix">
     * <tr><td>a</td><td>e</td><td>i</td><td>T0 : m</td></tr>
     * <tr><td>b</td><td>f</td><td>j</td><td>T1 : n</td></tr>
     * <tr><td>c</td><td>g</td><td>k</td><td>T2 : o</td></tr>
     * <tr><td>0 (d)</td><td>0 (h)</td><td>0 (l)</td><td>1 (p)</td></tr>
     * </table>
     * <br>
     * Then getColumn(0) returns (a,b,c).
     *
     * If you consider the matrix vector multiplication in line (V' = V x M, translation [T0,T1,T2] on the bottom) :
     * <br>
     * <table class="matrix">
     * <tr><td>a</td><td>b</td><td>c</td><td>0 (d)</td></tr>
     * <tr><td>e</td><td>f</td><td>g</td><td>0 (h) </td></tr>
     * <tr><td>i</td><td>j</td><td>k</td><td>0 (l)</td></tr>
     * <tr><td>T0 : m</td><td>T1 : n</td><td>T2 : o</td><td>1 (p)</td></tr>
     * </table>
     * <br>
     * Then getColumn(0) returns (a,b,c).
     *
     * No check is done to ensure pColumn is in the range [0-3].
     *
     * @param pColumn <div class="const">in</div> The column to get in the range [0-3].
     * @param pOut <div class="const">out</div> The resulting "column" of the matrix.
     * @return `pOut`.
     */
    getColumn(pColumn: number, pOut: Vector3): Vector3;
    /**
     * Computes the transformation of pVector by "this" matrix, and stores the result in pOut.
     *
     * If pOut is omitted, then pVector is changed in place.
     *
     * Returns pOut.
     *
     * @param pVector <div class="const">in</div> The vector4 to transform.
     * @param pOut <div class="const">out</div> The resulting transformed vector.
     * @return `pOut`.
     */
    multiplyVector4(pVector: Vector4, pOut?: Vector4): Vector4;
    /**
     * Computes the transformation of the vector (pX,pY,pZ,pW) by "this" matrix, and stores the result in pOut.
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> code>x</code> coordinate of the vector4 to transform.
     * @param pY <div class="const">in</div> <code>y</code> coordinate of the vector4 to transform.
     * @param pZ <div class="const">in</div> <code>z</code> coordinate of the vector4 to transform.
     * @param pW <div class="const">in</div> <code>w</code> coordinate of the vector4 to transform.
     * @param pOut <div class="const">out</div> The resulting transformed vector.
     * @return `pOut`.
     */
    multiplyVector4XYZW(pX: number, pY: number, pZ: number, pW: number, pOut: Vector4): Vector4;
    /**
     * Computes the transformation of the vector pPoint by "this" matrix, and stores the result in pOut.
     *
     * `pPoint` is considered a point and thus, this is equivalent to using
     * multiplyVector4XYZW(pPoint[0],pPoint[1],pPoint[2],1), and computing the homogeneous coordinates of the
     * resulting Vector4 (i.e. dividing all coordinates by the w coordinate of the vector4).
     *
     * If pOut is omitted, then pPoint is changed in place.
     *
     * Returns pOut.
     *
     * @param pPoint <div class="const">in</div> The vector3 to transform.
     * @param pOut <div class="const">out</div> The resulting transformed vector.
     * @return `pOut`.
     */
    multiplyVector3Position(pPoint: Vector3, pOut?: Vector3): Vector3;
    /**
     * Computes the transformation of the vector (pX,pY,pZ) by "this" matrix, and stores the result in pOut.
     *
     * The vector (pX,pY,pZ) is considered a point and thus, this is equivalent to using
     * multiplyVector4XYZW(pX,pY,pZ,1), and computing the homogeneous coordinates of the
     * resulting Vector4 (i.e. dividing all coordinates by the w coordinate of the vector4).
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> <code>x</code> coordinate of the vector3 to transform.
     * @param pY <div class="const">in</div> <code>y</code> coordinate of the vector3 to transform.
     * @param pZ <div class="const">in</div> <code>z</code> coordinate of the vector3 to transform.
     * @param pOut <div class="const">out</div> the resulting transformed vector.
     * @return `pOut`.
     */
    multiplyVector3PositionXYZ(pX: number, pY: number, pZ: number, pOut: Vector3): Vector3;
    /**
     * Computes the transformation of the vector pDirection by "this" matrix, and stores the result in pOut.
     *
     * `pDirection` is considered a direction and thus, this is equivalent to using
     * multiplyVector4XYZW(pPoint[0],pPoint[1],pPoint[2],0).
     *
     * If pOut is omitted, then pDirection is changed in place.
     *
     * Returns pOut.
     *
     * @param pDirection <div class="const">in</div> the vector3 to transform.
     * @param pOut <div class="const">out</div> the resulting transformed vector.
     * @return `pOut`.
     */
    multiplyVector3Direction(pDirection: Vector3, pOut?: Vector3): Vector3;
    /**
     * Computes the transformation of the vector (pX,pY,pZ) by "this" matrix, and stores the result in pOut.
     *
     * (pX,pY,pZ) is considered a direction and thus, this is equivalent to using
     * multiplyVector4XYZW(pX,pY,pZ,0).
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> <code>x</code> coordinate of the vector3 to transform.
     * @param pY <div class="const">in</div> <code>y</code> coordinate of the vector3 to transform.
     * @param pZ <div class="const">in</div> <code>z</code> coordinate of the vector3 to transform.
     * @param pOut <div class="const">out</div> the resulting transformed vector.
     * @return `pOut`.
     */
    multiplyVector3DirectionXYZ(pX: number, pY: number, pZ: number, pOut: Vector3): Vector3;
    /**
     * Tells if the matrix is the identity.
     *
     * There is no tolerance on the check, any value other than 0
     * or 1 in the matrix will make this function return false.
     *
     * @return `true` if the matrix is identity (strictly).
     */
    isIdentity(): boolean;
    /**
     * Sets this matrix to the identity.
     *
     * @return `this`.
     */
    makeIdentity(): Matrix4;
}
export { Matrix4 };
/**
 * A quaternion is a way to represent rotation matrices in 3d with 4 values.
 *
 * This documentation will not integrate all the publicly available documentation.
 *
 * From wikipedia :
 * In 3-dimensional space, according to Euler rotation theorem,
 * any rotation or sequence of rotations of a rigid body or coordinate system about a fixed point is equivalent to a single rotation
 * by a given angle &#x3B8; about a fixed axis (called the Euler axis) that runs through the fixed point.
 * The Euler axis is typically represented by a unit vector u. Therefore, any rotation in three dimensions can be
 * represented as a combination of a vector u and a scalar &#x3B8;.
 * Quaternions give a simple way to encode this axis-angle representation in four numbers,
 * and can be used to apply the corresponding rotation to a position vector, representing a point relative to the origin in R3.
 *
 * Such a quaternion q, with axis u (ux, uy, uz) and angle &#x3B8; is expressed as :
 * q = cos(&#x3B8;/2) + sin(&#x3B8;/2) * u.
 * The scalar is the w coordinate of the quaternion.
 *
 * [[include:constStyle.md]]<br><div class="const">Maths</div><br>
 * @see <a href="https://en.wikipedia.org/wiki/Quaternion" target="_blank">https://en.wikipedia.org/wiki/Quaternion</a>
 * @category Maths
 */
declare class Quaternion extends Vector4 {
    /**
     * Builds a new Quaternion with directional vector (px,py,pz) (at sin(&#x3B8;/2) diff)
     * and an angle &#x3B8;/2 such as cos(&#x3B8;/2) = pW.
     *
     * @param pX <div class="const">in</div> <code>x</code> coordinate of the directional vector of the rotation (at sin(&#x3B8;/2) diff).
     * @param pY <div class="const">in</div> <code>y</code> coordinate of the directional vector of the rotation (at sin(&#x3B8;/2) diff).
     * @param pZ <div class="const">in</div> <code>z</code> coordinate of the directional vector of the rotation (at sin(&#x3B8;/2) diff).
     * @param pW <div class="const">in</div> Angle of the rotation such as cos(&#x3B8;/2) = pW.
     * @return The newly created quaternion.
     */
    constructor(pX?: number, pY?: number, pZ?: number, pW?: number);
    /**
     * Computes the composition (multiplication) of the two transformations this and other.
     *
     * Any point P will be transformed into P' such that P' = pOther x this x P (or P x this x pOther = P').
     *
     * @param pOther <div class="const">in</div> The rotation to compose with.
     * @param pOut <div class="const">out</div> The resulting transformation ("this" will be applied first, then pOther).
     * @return The composition (multiplication) of the two transformations, stored in "this" if pOut is omitted.
     */
    multiplyVector(pOther: Quaternion, pOut?: Quaternion): Quaternion;
    /**
     * Tells if the two quaternions represent the same rotation.
     *
     * Reminder : for two quaternions q1 and q2, q1 and q2 represent the same rotation if
     * either one of these two conditions hold:
     *
     * * q1 is component wise strictly equal to q2.
     * * q1 is component wise strictly equal to -q2.
     *
     * @param pOtherQuaternion <div class="const">in</div> The other quaternion to compare to.
     * @return `true` if the two quaternions represent the same rotation.
     */
    equals(pOtherQuaternion: Quaternion): boolean;
    /**
     * Computes the composition (multiplication) of the two transformations "this" and the quaternion expressed by the values pX, pY, pZ, pW.
     *
     * Any point P will be transformed into P' such that P' = pOther x this x P (or P x this x pOther = P').
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the quaternion rotation to compose with.
     * @param pY <div class="const">in</div> <code>y</code> value of the quaternion rotation to compose with.
     * @param pZ <div class="const">in</div> <code>z</code> value of the quaternion rotation to compose with.
     * @param pW <div class="const">in</div> <code>w</code> value of the quaternion rotation to compose with.
     * @param pOut <div class="const">out</div> The resulting transformation ("this" will be applied first, then Quat(pX,pY,pZ,pW)).
     * @return The composition (multiplication) of the two transformations, stored in "this" if pOut is undefined.
     */
    multiply(pX: number, pY: number, pZ: number, pW: number, pOut?: Quaternion): Quaternion;
    /**
     * Computes the rotation quaternion from the given axis/angle representation.
     *
     * No normalization is computed on the directional vector of the axis, thus make sure
     * pAxis is normalized to avoid undesired effect (or keep a non normalized directional vector if you know
     * what you are doing).
     *
     * @param pAxis <div class="const">in</div> Directional vector of the axis (likely normalized).
     * @param pAngle <div class="const">in</div> Angle in radian of the rotation.
     * @return `this`.
     */
    fromAxisAngle(pAxis: Vector3, pAngle: number): Quaternion;
    /**
     * Extracts this rotation quaternion as an axis/angle representation in the resulting Vector4.
     *
     * The directional vector of the axis is the x,y,z components of the Vector4, the angle in radian is the
     * w component.
     *
     * @param pAxisAngle <div class="const">out</div> The resulting axis/angle representation.
     */
    toAxisAngle(pAxisAngle: Vector4): void;
    /**
     * Computes the conjugate (i.e. inverse of the rotation) of this quaternion.
     *
     * @param pOut <div class="const">out</div> The computed inverse of the rotation (stored in "this" if omitted).
     * @return The conjugate of this quaternion.
     */
    conjugate(pOut?: Quaternion): Quaternion;
    /**
     * Copies this quaternion onto another quaternion.
     *
     * Defaults to allocating a new quaternion.
     *
     * @param pOut <div class="const">out</div> The copy of this quaternion (allocated if omitted).
     * @return A copy of the quaternion (allocated by the library if pOther is omitted).
     */
    copy(pOut?: Quaternion): Quaternion;
    /**
     * Sets this quaternion from the given rotation matrix.
     *
     * @param pMatrix <div class="const">in</div> The rotation matrix to copy data from.
     * @return `this`.
     */
    setFromRotationMatrix(pMatrix: Matrix4): Quaternion;
    /**
     * Rotates (i.e computes the transformation) the given vector by the rotation representation of this quaternion.
     *
     * @param pVec <div class="const">in</div> The input vector to rotate.
     * @param pOut <div class="const">out</div> The result of the rotation of the vector pVec by this quaternion. If omitted, the input vector
     * pVec is transformed.
     * @return The result of the rotation of pVec (pOut is returned).
     */
    rotate(pVec: Vector3, pOut?: Vector3): Vector3;
    /**
     * Computes the spherical linear interpolation between the starting quaternion "this" and the end quaternion pEnd.
     *
     * If pTime=0, pOut stores "this". If pTime=1, pOut stores pEnd. `pTime` is in the range [0,1].
     *
     * No check is done to ensure pTime is
     * in the correct range.
     *
     * The resulting transformation is stored in pOut.
     *
     * @param pEnd <div class="const">in</div> The end quaternion of the interpolation.
     * @param pTime <div class="const">in</div> Number in the range [0,1].
     * @param pOut <div class="const">out</div> The result of the interpolation.
     * @return The result of the interpolation (pOut). "this" is returned if pOut is omitted.
     */
    slerpQuaternion(pEnd: Quaternion, pTime: number, pOut?: Quaternion): Quaternion;
    /**
     * Computes the spherical linear interpolation between the starting quaternion "this" and the end quaternion (pEndX,pEndY,pEndZ,pEndW).
     *
     * If pTime=0, pOut stores "this". If pTime=1, pOut stores pEnd. `pTime` is in the range [0,1].
     *
     * No check is done to ensure pTime is
     * in the correct range.
     *
     * The resulting transformation is stored in pOut.
     *
     * @param pEndX <div class="const">in</div> <code>x</code> component of the end quaternion of the interpolation.
     * @param pEndY <div class="const">in</div> <code>y</code> component of the end quaternion of the interpolation.
     * @param pEndZ <div class="const">in</div> <code>z</code> component of the end quaternion of the interpolation.
     * @param pEndW <div class="const">in</div> <code>w</code> component of the end quaternion of the interpolation.
     * @param pTime <div class="const">in</div> Number in the range [0,1].
     * @param pOut <div class="const">out</div> The result of the interpolation.
     * @return The result of the interpolation (pOut). "this" is returned if pOut is omitted.
     */
    slerp(pEndX: number, pEndY: number, pEndZ: number, pEndW: number, pTime: number, pOut?: Quaternion): Quaternion;
    /**
     * Computes the rotation that aligns `pSource` onto `pDestination`.
     *
     * The cross product `this.rotate(pSource)` and `pDestination` should be a zero vector.
     *
     * @param pSource <div class="const">in</div> The start vector (not necessarily unit).
     * @param pDestination <div class="const">in</div> The vector to align to (not necessarily unit).
     * @return `this`.
     */
    align(pSource: Vector3, pDestination: Vector3): Quaternion;
}
export { Quaternion };
/**
 * A 2D rectangle.
 *
 * Rectangles are used with 2D primitive managers.
 *
 * A rectangle is aligned on the pixel boundaries. Its origin is the lowest
 * value in x and y.
 *
 * [[include:constStyle.md]]<br><div class="const">Maths</div><br>
 * @category Maths
 */
declare class Rectangle {
    /**
     * `x` coordinate of the origin of the rectangle (minimum of x coordinate of the rectangle).
     *
     * This corresponds to the minimum abscissa of the rectangle. Since minimum screen abscissa is the left  of the screen,
     * x=0 for a rectangle on the left of the screen.
     */
    x: number;
    /**
     * `y` coordinate of the origin of the rectangle (minimum of y coordinate of the rectangle).
     *
     * This
     * corresponds to the minimum ordinate of the rectangle. Since minimum screen ordinate is the top of the screen, y=0 for a rectangle on top of the screen.
     */
    y: number;
    /**
     * Width of the rect (negative value is invalid).
     */
    width: number;
    /**
     * Height of the rect (negative value is invalid).
     */
    height: number;
    /**
     * Creates an rectangle with an origin point [pX,pY] and a size of [pWidth * pHeight].
     *
     * The other end point has the coordinate [pX+pWidth,pY+pHeight]. For the screen,
     * [pX,pY] is the top left position of the 2D rectangle.
     *
     * @param pX <div class="const">in</div> <code>x</code> coordinate of the origin of the rectangle.
     * @param pY <div class="const">in</div> <code>y</code> coordinate of the origin of the rectangle.
     * @param pWidth <div class="const">in</div> Width of the rectangle.
     * @param pHeight <div class="const">in</div> Height of the rectangle.
     * @return The newly constructed Rectangle.
     */
    constructor(pX?: number, pY?: number, pWidth?: number, pHeight?: number);
    /**
     * Tells if the two rectangles are strictly identical.
     *
     * @param pOtherRectangle <div class="const">in</div> The other rectangle to compare to.
     * @return `true` if the two rectangles are strictly identical.
     */
    equals(pOtherRectangle: Rectangle): boolean;
    /**
     * Clears the rectangle, i.e. sets the width and height to negative values.
     */
    clear(): void;
    /**
     * Copies "this" to pOther rectangle.
     *
     * If pOther is not supplied, a new Rectangle is allocated.
     *
     * @param pOut <div class="const">out</div> The Rectangle to copy to.
     * @return The copied Rectangle.
     */
    copy(pOut?: Rectangle): Rectangle;
    /**
     * Tells if the width and the height of the rectangle are positive or equal to zero.
     *
     * @return `true` if the rectangle is a valid mathematical object.
     */
    isValid(): boolean;
    /**
     * Tells if the rectangle is empty.
     *
     * An empty rectangle is either a invalid rectangle or a valid rectangle with zero height or width.
     *
     * @return `true` if the rectangle is empty.
     */
    isEmpty(): boolean;
}
export { Rectangle };
/**
 * This factory holds a static method to build UUIDs.
 *
 * As UUIDs strongly rely on random numbers,
 * a random number generation method is also provided.
 *
 * [[include:constStyle.md]]<br><div class="const">Factories</div><br>
 * @category Factories
 */
declare class UUIDFactory {
    /**
     * Generates a random number in the range [0,1[.
     *
     * Depending on your platform, the
     * function may use cryptographic functions or use Math.random.
     *
     * @return A random number in the range [0,1[.
     */
    static GenerateRandom(): number;
    /**
     * Generates an new UUID string in the form
     * xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx if pFormatted or xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx if not pFormatted
     * where x is in the set [0-9a-f].
     *
     * @param pFormatted <div class="const">in</div> If true, the uuid string will contain "-".
     * @return A random UUID string.
     */
    static GenerateUUID(pFormatted: boolean): string;
}
export { UUIDFactory };
/**
 * A 2 dimensional vector.
 *
 * [[include:constStyle.md]]<br><div class="const">Maths</div><br>
 * @category Maths
 */
declare class Vector2 {
    /**
     * `x` value of the vector.
     */
    x: number;
    /**
     * `y` value of the vector.
     */
    y: number;
    /**
     * Creates a vector2.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector.
     * @return The newly constructed Vector2.
     */
    constructor(pX?: number, pY?: number);
    /**
     * Gets the string representation of the vector.
     *
     * @return The string representation of the vector.
     */
    toString(): string;
    /**
     * Serializes the given Vector2 to JSON.
     *
     * @return the JSON representation of the object.
     * @internal
     */
    toJSON(_pKey?: any): Object;
    /**
     * Tells if the two vectors are strictly identical.
     *
     * @param pOtherVector <div class="const">in</div> The other vector to compare to.
     * @return `true` if the two vectors are strictly identical.
     */
    equals(pOtherVector: Vector2): boolean;
    /**
     * Copies "this" to pOther vector.
     *
     * If pOther is not supplied, a new Vector2 is allocated.
     *
     * @param pOut <div class="const">out</div> The Vector2 to copy to.
     * @return The copied vector.
     */
    copy(pOut?: Vector2): Vector2;
    /**
     * Sets the vector components.
     *
     * @param pX <div class="const">in</div> <code>x</code> value.
     * @param pY <div class="const">in</div> <code>y</code> value.
     * @return `this`.
     */
    set(pX: number, pY: number): Vector2;
    /**
     * Computes this + pOther and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pOther <div class="const">in</div> The vector2 to add.
     * @param pOut <div class="const">out</div> The resulting vector2.
     * @return `pOut`.
     */
    addVector(pOther: Vector2, pOut?: Vector2): Vector2;
    /**
     * Computes this + (pX,pY) and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to add.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to add.
     * @param pOut <div class="const">out</div> The resulting Vector2.
     * @return `pOut`.
     */
    add(pX: number, pY: number, pOut?: Vector2): Vector2;
    /**
     * Computes this - pOther and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pOther <div class="const">in</div> The vector2 to subtract.
     * @param pOut <div class="const">out</div> The resulting vector2.
     * @return `pOut`.
     */
    subtractVector(pOther: Vector2, pOut?: Vector2): Vector2;
    /**
     * Computes this - (pX,pY) and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to subtract.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to subtract.
     * @param pOut <div class="const">out</div> The resulting vector2.
     * @return `pOut`.
     */
    subtract(pX: number, pY: number, pOut?: Vector2): Vector2;
    /**
     * Computes the linear interpolation between `this` and `pOther` and stores the result in `pOut`.
     *
     * `pOut` = `this` + pTime x ( `pOther` - `this`).<br>
     * If pTime=0, `pOut` = `this`.
     * <br>If pTime=1, `pOut` = `pOther`.
     *
     * If `pOut` is omitted, the result of the interpolation is stored in `this`.
     *
     * No check is done to ensure pTime is in the range [0,1].
     *
     * @param pOther <div class="const">in</div> The destination vector.
     * @param pTime <div class="const">in</div> Interpolation parameter in the range [0,1].
     * @param pOut <div class="const">out</div> The resulting vector2.
     * @return `pOut`.
     */
    lerpVector(pOther: Vector2, pTime: number, pOut?: Vector2): Vector2;
    /**
     * Computes the linear interpolation between `this` and (pX,pY) and stores the result in `pOut`.
     *
     * `pOut` = `this` + `pTime` x ( (pX,pY) - `this`).<br>
     * If `pTime`=0, `pOut` = `this`.
     * <br>If `pTime`=1, `pOut` = (pX,pY).
     *
     * If `pOut` is omitted, the result of the interpolation is stored in `this`.
     *
     * No check is done to ensure `pTime` is in the range [0,1].
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the destination vector.
     * @param pY <div class="const">in</div> <code>y</code> value of the destination vector.
     * @param pTime <div class="const">in</div> Interpolation parameter in the range [0,1].
     * @param pOut <div class="const">out</div> The resulting vector2.
     * @return `pOut`.
     */
    lerp(pX: number, pY: number, pTime: number, pOut?: Vector2): Vector2;
    /**
     * Computes the opposite of this (-this) and stores the result in pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * @param pOut <div class="const">out</div> The resulting vector.
     * @return `pOut`.
     */
    negate(pOut?: Vector2): Vector2;
    /**
     * Computes the scale of `this` by the scalar `pScalar`, and stores the result in `pOut`.
     *
     * `pOut` = `pScalar` x `this`.
     *
     * If `pOut` is omitted, the result is stored in `this`.
     *
     * @param pScalar <div class="const">in</div> The scale factor.
     * @param pOut <div class="const">out</div> The resulting vector.
     * @return `pOut`.
     */
    scale(pScalar: number, pOut?: Vector2): Vector2;
    /**
     * Computes the normalization of this (i.e. scales "this" such that pOut.length() == 1, a zero vector is left unchanged) and stores the result in pOut.
     *
     * If pOut is omitted, "this" is normalized.
     *
     * @param pOut <div class="const">out</div> The resulting vector.
     * @return `pOut`.
     */
    normalize(pOut?: Vector2): Vector2;
    /**
     * Gets the euclidean length of "this".
     *
     * @return The length of this.
     */
    length(): number;
    /**
     * Gets the squared euclidean length of "this".
     *
     * @return The squared length of this.
     */
    lengthSquare(): number;
    /**
     * Computes the euclidean distance between this and pVec.
     *
     * @param pVec <div class="const">in</div> The other vector to compute the distance to.
     * @return The distance between this and pVec.
     */
    distanceToVector(pVec: Vector2): number;
    /**
     * Computes the squared euclidean distance between this and pVec.
     *
     * @param pVec <div class="const">in</div> The other vector to compute the distance to.
     * @return The squared distance between this and pVec.
     */
    squaredDistanceToVector(pVec: Vector2): number;
    /**
     * Computes the dot product between this and pOther.
     *
     * @param pOther <div class="const">in</div> The other vector2 to compute the dot with.
     * @return The dot product ( dot(this, pOther) ) .
     */
    dotVector(pOther: Vector2): number;
    /**
     * Computes the dot product between this and (pX,pY).
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to compute the dot with.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to compute the dot with.
     * @return The dot product ( dot(this, (pX,pY)) ) .
     */
    dot(pX: number, pY: number): number;
}
export { Vector2 };
/**
 * A 3 dimensional vector.
 *
 * [[include:constStyle.md]]<br><div class="const">Maths</div><br>
 * @category Maths
 */
declare class Vector3 {
    /**
     * `x` value of the vector.
     */
    x: number;
    /**
     * `y` value of the vector.
     */
    y: number;
    /**
     * `z` value of the vector.
     */
    z: number;
    /**
     * Creates a vector3.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector.
     * @param pZ <div class="const">in</div> <code>z</code> value of the vector.
     * @return The newly constructed Vector3.
     */
    constructor(pX?: number, pY?: number, pZ?: number);
    /**
     * Gets the string representation of the vector.
     *
     * @return The string representation of the vector.
     */
    toString(): string;
    /**
     * Copies "this" to pOther vector. If pOther is not supplied, a new Vector3 is allocated.
     *
     * @param pOut <div class="const">out</div> The Vector3 to copy to.
     * @return The copied vector.
     */
    copy(pOut?: Vector3): Vector3;
    /**
     * Serializes the given Vector3 to JSON.
     *
     * @return the JSON representation of the object.
     * @internal
     */
    toJSON(_pKey?: any): Object;
    /**
     * Tells if the two vectors are strictly identical.
     *
     * @param pOtherVector <div class="const">in</div> The other vector to compare to.
     * @return `true` if the two vectors are strictly identical.
     */
    equals(pOtherVector: Vector3): boolean;
    /**
     * Sets the vector components.
     *
     * @param pX <div class="const">in</div> <code>x</code> value.
     * @param pY <div class="const">in</div> <code>y</code> value.
     * @param pZ <div class="const">in</div> <code>z</code> value.
     * @return `this`.
     */
    set(pX: number, pY: number, pZ: number): Vector3;
    /**
     * Computes this + pOther and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pOther <div class="const">in</div> The vector3 to add.
     * @param pOut <div class="const">out</div> The resulting vector3.
     * @return `pOut`.
     */
    addVector(pOther: Vector3, pOut?: Vector3): Vector3;
    /**
     * Computes this + (pX,pY,pZ) and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to add.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to add.
     * @param pZ <div class="const">in</div> <code>z</code> value of the vector to add.
     * @param pOut <div class="const">out</div> The resulting vector3.
     * @return `pOut`.
     */
    add(pX: number, pY: number, pZ: number, pOut?: Vector3): Vector3;
    /**
     * Computes this - pOther and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pOther <div class="const">in</div> The vector3 to subtract.
     * @param pOut <div class="const">out</div> The resulting vector3.
     * @return `pOut`.
     */
    subtractVector(pOther: Vector3, pOut?: Vector3): Vector3;
    /**
     * Computes this - (pX,pY,pZ) and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to subtract.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to subtract.
     * @param pZ <div class="const">in</div> <code>z</code> value of the vector to subtract.
     * @param pOut <div class="const">out</div> The resulting vector3.
     * @return `pOut`.
     */
    subtract(pX: number, pY: number, pZ: number, pOut?: Vector3): Vector3;
    /**
     * Computes the dot product between this and pOther.
     *
     * @param pOther <div class="const">in</div> The other vector3 to compute the dot with.
     * @return The dot product ( dot(this, pOther) ).
     */
    dotVector(pOther: Vector3): number;
    /**
     * Computes the dot product between this and (pX,pY,pZ).
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to compute the dot with.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to compute the dot with.
     * @param pZ <div class="const">in</div> <code>z</code> value of the vector to compute the dot with.
     * @return The dot product ( dot(this, (pX,pY,pZ)) ) .
     */
    dot(pX: number, pY: number, pZ: number): number;
    /**
     * Computes the cross vector between this and pOther and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pOther <div class="const">in</div> The other vector3 to compute the cross product with.
     * @param pOut <div class="const">out</div> The resulting vector3, cross product (this ^ pOther).
     * @return `pOut`.
     */
    crossVector(pOther: Vector3, pOut?: Vector3): Vector3;
    /**
     * Computes the cross vector between this and (pX,pY,pZ) and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to compute the cross product with.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to compute the cross product with.
     * @param pZ <div class="const">in</div> <code>z</code> value of the vector to compute the cross product with.
     * @param pOut <div class="const">out</div> The resulting vector3, cross product (this ^ pOther).
     * @return `pOut`.
     */
    cross(pX: number, pY: number, pZ: number, pOut?: Vector3): Vector3;
    /**
     * Computes the linear interpolation between `this` and `pOther` and stores the result in `pOut`.
     *
     * `pOut` = `this` + `pTime` x ( `pOther` - `this`).<br>
     * If `pTime`=0, `pOut` = `this`.
     * <br>If `pTime`=1, `pOut` = `pOther`.
     *
     * If `pOut` is omitted, the result of the interpolation is stored in `this`.
     *
     * No check is done to ensure `pTime` is in the range [0,1].
     *
     * @param pOther <div class="const">in</div> The destination vector.
     * @param pTime <div class="const">in</div> Interpolation parameter in the range [0,1].
     * @param pOut <div class="const">out</div> The resulting vector3.
     * @return `pOut`.
     */
    lerpVector(pOther: Vector3, pTime: number, pOut?: Vector3): Vector3;
    /**
     * Computes the linear interpolation between `this` and (pX,pY,pZ) and stores the result in `pOut`.
     *
     * `pOut` = `this` + `pTime` x ( (pX,pY,pZ) - `this`).
     * <br>If `pTime`=0, `pOut` = `this`.
     * <br>If `pTime`=1, `pOut` = (pX,pY,pZ).
     *
     * If `pOut` is omitted, the result of the interpolation is stored in `this`.
     *
     * No check is done to ensure `pTime` is in the range [0,1].
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the destination vector.
     * @param pY <div class="const">in</div> <code>y</code> value of the destination vector.
     * @param pZ <div class="const">in</div> <code>z</code> value of the destination vector.
     * @param pTime <div class="const">in</div> Interpolation parameter in the range [0,1].
     * @param pOut <div class="const">out</div> The resulting vector3.
     * @return `pOut`.
     */
    lerp(pX: number, pY: number, pZ: number, pTime: number, pOut?: Vector3): Vector3;
    /**
     * Computes the opposite of this (-this) and stores the result in pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * @param pOut <div class="const">out</div> The resulting vector.
     * @return `pOut`.
     */
    negate(pOut?: Vector3): Vector3;
    /**
     * Computes the scale of `this` by the scalar `pScalar`, and stores the result in `pOut`.
     *
     * `pOut` = `pScalar` x `this`.
     *
     * If `pOut` is omitted, the result is stored in `this`.
     *
     * @param pScalar <div class="const">in</div> The scale factor.
     * @param pOut <div class="const">out</div> The resulting vector.
     * @return `pOut`.
     */
    scale(pScalar: number, pOut?: Vector3): Vector3;
    /**
     * Computes the normalization of this (i.e. scales "this" such that pOut.length() == 1, a zero vector is left unchanged) and stores the result in pOut.
     *
     * If pOut is omitted, "this" is normalized.
     *
     * @param pOut <div class="const">out</div> The resulting vector.
     * @return `pOut`.
     */
    normalize(pOut?: Vector3): Vector3;
    /**
     * Gets the euclidean length of `this`.
     *
     * @return The length of `this`.
     */
    length(): number;
    /**
     * Gets the squared euclidean length of "this".
     *
     * @return The squared length of this.
     */
    lengthSquare(): number;
    /**
     * Computes the euclidean distance between this and pVec.
     *
     * @param pVec <div class="const">in</div> The other vector to compute the distance to.
     * @return The distance between this and pVec.
     */
    distanceToVector(pVec: Vector3): number;
    /**
     * Computes the squared euclidean distance between this and pVec.
     *
     * @param pVec <div class="const">in</div> The other vector to compute the distance to.
     * @return The squared distance between this and pVec.
     */
    squaredDistanceToVector(pVec: Vector3): number;
}
export { Vector3 };
/**
 * A 3D Ray.
 *
 * The 3D ray is a starting position and a direction.
 *
 * The 3D ray is a 'half' in a sense it comes from the start position and goes through the direction.
 *
 * [[include:constStyle.md]]<br><div class="const">Maths</div><br>
 * @category Maths
 */
interface Ray3 {
    /**
     * The origin of the ray.
     */
    origin: Vector3;
    /**
     * The direction of the ray.
     */
    direction: Vector3;
}
export { Ray3 };
/**
 * A 4 dimensional vector.
 *
 * [[include:constStyle.md]]<br><div class="const">Maths</div><br>
 * @category Maths
 */
declare class Vector4 {
    /**
     * `x` value of the vector, i.e. first component.
     */
    x: number;
    /**
     * `y` value of the vector, i.e. second component.
     */
    y: number;
    /**
     * `z` value of the vector, i.e. third component.
     */
    z: number;
    /**
     * `w` value of the vector, i.e. fourth component.
     */
    w: number;
    /**
     * Creates a vector4.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector.
     * @param pZ <div class="const">in</div> <code>z</code> value of the vector.
     * @param pW <div class="const">in</div> <code>w</code> value of the vector.
     * @return A new Vector4.
     */
    constructor(pX?: number, pY?: number, pZ?: number, pW?: number);
    /**
     * Gets the string representation of the vector.
     *
     * @return The string representation of the vector.
     */
    toString(): string;
    /**
     * Serializes the given Vector4 to JSON.
     *
     * @return the JSON representation of the object.
     * @internal
     */
    toJSON(_pKey?: any): Object;
    /**
     * Copies "this" to pOther vector.
     *
     * If pOther is not supplied, a new vector4 is allocated.
     *
     * @param pOut <div class="const">out</div> The vector4 to copy to.
     * @return The copied vector.
     */
    copy(pOut?: Vector4): Vector4;
    /**
     * Tells if the two vectors are strictly identical.
     *
     * If you want to tell if the two vector4 represent the same 3D cartesian point, use
     * {@link isHomogeneousEqual}.
     *
     * @param pOtherVector <div class="const">in</div> The other vector to compare to.
     * @return `true` if the two vectors are strictly identical.
     */
    equals(pOtherVector: Vector4): boolean;
    /**
     * Tells if the two vectors are homogeneously equivalent.
     *
     * This function tells if the two Vector4 represent the same 3D cartesian point.
     *
     * @param pOtherVector <div class="const">in</div> The other vector to compare to.
     * @return `true` if the two vectors are homogeneously equivalent.
     */
    isHomogeneousEqual(pOtherVector: Vector4): boolean;
    /**
     * Sets the vector components.
     *
     * @param pX <div class="const">in</div> <code>x</code> value.
     * @param pY <div class="const">in</div> <code>y</code> value.
     * @param pZ <div class="const">in</div> <code>z</code> value.
     * @param pW <div class="const">in</div> <code>w</code> value.
     * @return `this`.
     */
    set(pX: number, pY: number, pZ: number, pW: number): Vector4;
    /**
     * Computes this + pOther and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pOther <div class="const">in</div> The vector4 to add.
     * @param pOut <div class="const">out</div> The resulting vector4.
     * @return `pOut`.
     */
    addVector(pOther: Vector4, pOut?: Vector4): Vector4;
    /**
     * Computes this + (pX,pY,pZ,pW) and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to add.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to add.
     * @param pZ <div class="const">in</div> <code>z</code> value of the vector to add.
     * @param pW <div class="const">in</div> <code>w</code> value of the vector to add.
     * @param pOut <div class="const">out</div> The resulting vector4.
     * @return `pOut`.
     */
    add(pX: number, pY: number, pZ: number, pW: number, pOut?: Vector4): Vector4;
    /**
     * Computes this - pOther and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pOther <div class="const">in</div> The vector4 to subtract.
     * @param pOut <div class="const">out</div> The resulting vector4.
     * @return `pOut`.
     */
    subtractVector(pOther: Vector4, pOut?: Vector4): Vector4;
    /**
     * Computes this - (pX,pY,pZ,pW) and stores it into pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * Returns pOut.
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to subtract.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to subtract.
     * @param pZ <div class="const">in</div> <code>z</code> value of the vector to subtract.
     * @param pW <div class="const">in</div> <code>w</code> value of the vector to subtract.
     * @param pOut <div class="const">out</div> The resulting vector4.
     * @return `pOut`.
     */
    subtract(pX: number, pY: number, pZ: number, pW: number, pOut?: Vector4): Vector4;
    /**
     * Computes the dot product between this and pOther.
     *
     * @param pOther <div class="const">in</div> The other vector4 to compute the dot with.
     * @return The dot product ( dot(this, pOther) ).
     */
    dotVector(pOther: Vector4): number;
    /**
     * Computes the dot product between this and (pX,pY,pZ,pW).
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the vector to compute the dot with.
     * @param pY <div class="const">in</div> <code>y</code> value of the vector to compute the dot with.
     * @param pZ <div class="const">in</div> <code>z</code> value of the vector to compute the dot with.
     * @param pW <div class="const">in</div> <code>w</code> value of the vector to compute the dot with.
     * @return The dot product ( dot(this, (pX,pY,pZ,pW)) ).
     */
    dot(pX: number, pY: number, pZ: number, pW: number): number;
    /**
     * Computes the linear interpolation between `this` and `pOther` and stores the result in `pOut`.
     *
     * `pOut` = `this` + `pTime` x ( `pOther` - `this`).
     * <br>If `pTime`=0, `pOut` = `this`.
     * <br>If `pTime`=1, `pOut` = `pOther`.
     *
     * If `pOut` is omitted, the result of the interpolation is stored in `this`.
     *
     * No check is done to ensure pTime is in the range [0,1].
     *
     * @param pOther <div class="const">in</div> The destination vector.
     * @param pTime <div class="const">in</div> Interpolation parameter in the range [0,1].
     * @param pOut <div class="const">out</div> The resulting vector4.
     * @return `pOut`.
     */
    lerpVector(pOther: Vector4, pTime: number, pOut?: Vector4): Vector4;
    /**
     * Computes the linear interpolation between `this` and (pX,pY,pZ,pW) and stores the result in `pOut`.
     *
     * `pOut` = `this` + `pTime` x ( (pX,pY,pZ,pW) - `this`).
     * <br>If `pTime`=0, `pOut` = `this`.
     * <br>If `pTime`=1, `pOut` = (pX,pY,pZ,pW).
     *
     * If `pOut` is omitted, the result of the interpolation is stored in `this`.
     *
     * No check is done to ensure `pTime` is in the range [0,1].
     *
     * @param pX <div class="const">in</div> <code>x</code> value of the destination vector.
     * @param pY <div class="const">in</div> <code>y</code> value of the destination vector.
     * @param pZ <div class="const">in</div> <code>z</code> value of the destination vector.
     * @param pW <div class="const">in</div> <code>w</code> value of the destination vector.
     * @param pTime <div class="const">in</div> Interpolation parameter in the range [0,1].
     * @param pOut <div class="const">out</div> The resulting vector4.
     * @return `pOut`.
     */
    lerp(pX: number, pY: number, pZ: number, pW: number, pTime: number, pOut?: Vector4): Vector4;
    /**
     * Computes the opposite of this (-this) and stores the result in pOut.
     *
     * If pOut is omitted, the result is stored in "this".
     *
     * @param pOut <div class="const">out</div> The resulting vector.
     * @return `pOut`.
     */
    negate(pOut?: Vector4): Vector4;
    /**
     * Computes the scale of `this` by the scalar `pScalar`, and stores the result in `pOut`.
     *
     * `pOut` = `pScalar` x `this`.
     *
     * If `pOut` is omitted, the result is stored in `this`.
     *
     * @param pScalar <div class="const">in</div> The scale factor.
     * @param pOut <div class="const">out</div> The resulting vector.
     * @return `pOut`.
     */
    scale(pScalar: number, pOut?: Vector4): Vector4;
    /**
     * Computes the normalization of this (i.e. scales "this" such that pOut.length() == 1, a zero vector is left unchanged) and stores the result in pOut.
     *
     * If pOut is omitted, "this" is normalized.
     *
     * Warning, this function does not assume that w=1, and does not assume the vector4 as a vector3. The w component is
     * used in the calculation.
     *
     * @param pOut <div class="const">out</div> The resulting vector.
     * @return `pOut`.
     */
    normalize(pOut?: Vector4): Vector4;
    /**
     * Gets the euclidean length of "this".
     *
     * Warning, this function does not assume that w=1, and does not assume the vector4 as a vector3. The w component is
     * used in the calculation.
     *
     * @return The length of this.
     */
    length(): number;
    /**
     * Gets the squared euclidean length of "this".
     *
     * Warning, this function does not assume that w=1, and does not assume the vector4 as a vector3. The w component is
     * used in the calculation.
     *
     * @return The squared length of this.
     */
    lengthSquare(): number;
    /**
     * Computes the euclidean distance between this and pVec.
     *
     * Warning, this function does not assume that w=1, and does not assume the vector4 as a vector3. The w component is
     * used in the calculation.
     *
     * @param pVec <div class="const">in</div> The other vector to compute the distance to.
     * @return The distance between this and pVec.
     */
    distanceToVector(pVec: Vector4): number;
    /**
     * Computes the squared euclidean distance between this and pVec.
     *
     * Warning, this function does not assume that w=1, and does not assume the vector4 as a vector3. The w component is
     * used in the calculation.
     *
     * @param pVec <div class="const">in</div> The other vector to compute the distance to.
     * @return The squared distance between this and pVec.
     */
    squaredDistanceToVector(pVec: Vector4): number;
    /**
     * Sets this Vector4 to be the representation of the plane defined by its normal and a point of this plane.
     *
     * No check is done to ensure that pNormal is normalized.
     *
     * The Vector4 can be the parametric equation of a plane in the form P: &#123;t,u,v&#125; / at + bu + cv = d.<br>
     * So, with the components x,y,z,w of the vector 4 : &#123;t,u,v&#125; / xt + yu + zv = w where (x,y,z) is the normal
     * vector of the plane.
     *
     * @param pNormal <div class="const">in</div> The normal of the plane.
     * @param pPlanePoint <div class="const">in</div> A point that belongs to the plane.
     * @return `this`.
     */
    setFromPlane(pNormal: Vector3, pPlanePoint: Vector3): Vector4;
    /**
     * Computes the euclidean distance from the plane represented by this to pPoint.
     *
     * The Vector4 can be the parametric equation of a plane in the form P: &#123;t,u,v&#125; / at + bu + cv = d.<br>
     * So, with the components x,y,z,w of the vector 4 : &#123;t,u,v&#125; / xt + yu + zv = w where (x,y,z) is the normal
     * vector of the plane.
     *
     * @param pPoint <div class="const">in</div> The distance to compute the distance to.
     * @return The distance between the plane represented by this and pPoint.
     */
    planeDistanceFromPoint(pPoint: Vector3): number;
}
export { Vector4 };
